(define-module (alterator ahttpd)
    :use-module (ice-9 regex)
    :use-module (srfi srfi-1)
    :use-module (srfi srfi-2)
    :use-module (srfi srfi-4)
    :use-module (srfi srfi-11)
    :use-module (vhttpd)
    :use-module (alterator algo)
    :use-module (alterator plist)
    :use-module (alterator str)
    :use-module (alterator homovector)
    :use-module (alterator mime)
    :use-module (alterator session)
    :re-export (;;useful functions from lowlevel library
		message?
		message-plain?
		message-header
		message-header-set!
		message-cookie
		message-method
		message-uri
		message-query
		message-body
		message-startline
		message-code
		message-code-string

		decode-url-component
		encode-url-component)
    :export (;;low level message processing
             parse-accept-language
             parse-url-args
             parse-multipart
             ;;high level message processing
             message-accept-language
             message-expert-mode
             message-url-args
             message-request-args
             message-full-args
             message-woo-args
             message-action
             ;;global session
             ahttpd-session-id
             ahttpd-session-ref
             ahttpd-session-set!
             with-ahttpd-session
             ahttpd-session-pause
             ahttpd-session-resume
             ;;pause/resume
             ahttpd-pause
             ahttpd-resume
             ahttpd-pause/resume
             ;;additinal helpers
             uri-prefix?
             backend-args))

;;;

;; memoized message property
(define-macro (define-message-property func . body)
  (let* ((func-name (car func))
	 (func-args (cdr func))
	 (int-name (gensym))
	 (prop-name (gensym))
	 (temp (gensym)))
    `(begin (define (,int-name ,@func-args)
	      ,@body)
	    (define ,prop-name (make-object-property))
	    (define (,func-name ,@func-args)
	      (or (,prop-name ,(car func-args))
		  (let ((,temp (,int-name ,@func-args)))
		    (set! (,prop-name ,(car func-args)) ,temp)
		    ,temp))))))

;;;

(define *default-lang* '(("uk" . "ua")
                         ("en" . "us")
                         ("be" . "by")))

(define (locale-name lang country)
  (string-append
    (string-downcase lang)
    "_"
    (string-upcase (or country
                       (cond-assoc lang *default-lang* lang)))))

(define *accept-language-re* "([a-zA-Z]+)(-([a-zA-Z]+))?[[:space:]]*(;[[:space:]]*q=([0-9.]+))?")

(define (parse-accept-language str)
      (map car
       (stable-sort
	 (map (lambda(m)
		(cons (locale-name (match:substring m 1) (match:substring m 3))
		       (string->number (or (match:substring m 5) "1"))))
	      (list-matches *accept-language-re*  str))
	(lambda(x y)
	  (> (cdr x) (cdr y))))))

(define-message-property (message-accept-language msg)
  (let ((cookie (message-cookie msg "language"))
	(header (message-header msg "accept-language")))
    (or cookie
	(and header (string-join (parse-accept-language header) ";"))
	"en_US")))

(define-message-property (message-expert-mode msg)
  (or (message-cookie msg "expert_mode") "0"))
;;;

(define *url-args-re* "(^|&)([^&=]+)=([^&=]+)?(&|$)")

(define (decode-url-component/nocr x)
 (if (string? x)
     (string-delete (decode-url-component x) #\cr)
     ""))

(define (parse-url-args str)
  (map
    (lambda(m)
      (cons (decode-url-component/nocr (match:substring m 2))
	    (decode-url-component/nocr (match:substring m 3))))
    (list-matches *url-args-re* str)))

(define-message-property (message-url-args msg)
  (parse-url-args (message-query msg)))

;;;


(define (parse-multipart s boundary)
  (let* ((v (ISO-8859-1-string->u8vector s))
	 (l (u8vector-length v))
	 (parts (mime-decode-multipart-with-boundary v 0 l boundary)))
    (if parts
      (map
	(lambda(x)
	  (let* ((content-disposition (cond-assoc 'content-disposition (car x)))
		 (name (cond-assoc 'name (cdr content-disposition))))
	    (cons name (cdr x))))
	parts)
      (error "wrong multipart content" s))))


(define-message-property (message-request-args msg)
  (or
    (and-let* ((body (message-body msg))
	       ((not (string-null? body)))
	       (mime-string (message-header msg "content-type"))
	       (type (mime-string->content-type mime-string))
	       (type-name  (cadr type))
	       (type-args  (cddr type)))
	      (cond
		((equal? type-name '(application . x-www-form-urlencoded))
		 (parse-url-args body))
		((equal? type-name '(multipart . form-data))
		 (parse-multipart body (cond-assq 'boundary type-args)))))
    '()))


(define-message-property (message-full-args msg)
  (acons "language"
	 (message-accept-language msg)
	 (append (message-request-args msg)
		 (message-url-args msg)
		 (acons "expert_mode" (message-expert-mode msg) '()))))

;;;

(define-message-property (message-action msg)
  (let ((method (message-method msg)))
   (cond
     ((string-ci=? "post" method)
      (or (cond-assoc "action" (message-url-args msg))
          (cond-assoc "action" (message-request-args msg))
          "write"))
     ((string-ci=? "get" method)
      "read")
     (else (error (format #f "unsupported method ~S~%" method))))))
;;;

(define (true-string? value)
  (and (string? value) (string=? value "#t")))
(define (false-string? value)
  (and (string? value) (string=? value "#f")))

;; convert strings to appropriate scheme objects
(define (scm-arg-list lst)
  (map
    (lambda(arg)
      (let ((name (string->symbol (car arg)))
	    (value (cdr arg)))
	(cond
	  ((true-string? value) (cons name #t))
	  ((false-string? value) (cons name #f))
	  ((u8vector? value) (cons name (u8vector->ISO-8859-1-string value)))
	  (else (cons name value)))))
    lst))

(define (merge-arg first args)
  (cons (car first)
        (if (pair? args)
            (cons (cdr first) (map cdr args))
            (cdr first))))

;; join args with same names into array
(define (join-arg-list lst)
  (let loop ((result '())
             (rest lst))
    (if (null? rest)
        (reverse result)
        (let*-values (( (first) (car rest) )
		      ( (first-name) (car first) )
                      ( (out-merge out-rest) (partition (lambda(x) (eq? (car x) first-name)) (cdr rest)) ))
          (loop (cons (merge-arg first out-merge) result)
                out-rest)))))

;; replace values of list type with strings
(define (unlist-arg-list lst)
  (map
    (lambda (x)
      (let ((name (car x))
            (value (cdr x)))
        (cons name (if (pair? value) (string-join value ";") value))))
    lst))


(define-message-property (message-woo-args msg)
  (unlist-arg-list
    (join-arg-list
      (scm-arg-list (message-full-args msg)))))

;;;

(define (uri-prefix? prefix-uri main-uri)
  (and (string-prefix? prefix-uri main-uri)
       (let ((prefix-length (string-length prefix-uri))
             (main-length (string-length main-uri)))
         (or (= prefix-length 1)
             (= prefix-length main-length)
             (char=? (string-ref main-uri prefix-length) #\/)))))

(define (backend-args args . namelist)
  (let ((namelist (if (pair? namelist) (map string->symbol (car namelist)) '())))
    (alist->plist
      (filter (lambda (x)
		(or (null? namelist) (memq (car x) namelist)))
		args))))


;;; ahttpd global session

;; Note: work with session objects not with session ids,
;; because this code should work in logout process when session will be removed from list

(define *default-session* (session-obj (make-session "root" (* 60 60 24 365))))
(define *current-session* (make-fluid))

(define (with-ahttpd-session msg uri proc)
  (with-fluids ((*current-session* (session-obj (message-cookie msg "session"))))
	       (ahttpd-session-set! 'ahttpd-request msg) ;; for form-cookie
	       (ahttpd-session-set! 'ahttpd-full-args (message-full-args msg)) ;; for form-blob, call-with-form-file
	       (ahttpd-session-set! 'ahttpd-woo-args (message-woo-args msg)) ;; for form-value, form-update-value
	       (ahttpd-session-set! 'ahttpd-uri uri)
	       (proc)))

;; list of variable to save between pause and resume operations
(define *inter-resume-variable-list* '(ahttpd-full-args ahttpd-woo-args))

;; save before resume
(define (ahttpd-session-pause)
  (reverse
    (fold (lambda(x y) (cons (ahttpd-session-ref x) y))
          '()
          *inter-resume-variable-list*)))

;; restore data after resume
(define (ahttpd-session-resume data)
  (for-each (lambda(x y) (ahttpd-session-set! x y))
            *inter-resume-variable-list*
            data))

(define (ahttpd-session-obj)
  (or (fluid-ref *current-session*)
      *default-session*
      (error "ahttpd-session-ref: session is not defined")))

(define (ahttpd-session-id)
  (session-obj-id (ahttpd-session-obj)))

(define (ahttpd-session-ref key . default)
  (apply session-obj-ref (ahttpd-session-obj) key default))

(define (ahttpd-session-set! key value)
  (session-obj-set! (ahttpd-session-obj) key value))

;;; session's pause/resume
(define (ahttpd-pause proc)
 (call-with-current-continuation
   (lambda(resume)
     (ahttpd-session-set! 'ahttpd-resume resume)
     (proc))))

(define (ahttpd-resume . args)
 (let ((resume (ahttpd-session-ref 'ahttpd-resume #f)))
   (if (procedure? resume)
       (apply resume args)
       (error "ahttpd-resume: resume point not found"))))

;;; continuation break point
(define (ahttpd-pause/resume proc)
  (ahttpd-pause
    (lambda()
      (ahttpd-resume (proc)))))
