(define-module (alterator ajax)

	       :use-module (vhttpd)
	       :use-module (alterator ahttpd)
	       :use-module (alterator gettext)
	       :use-module (alterator woo)
	       :use-module (alterator algo)
	       :use-module (alterator str)
	       :use-module (alterator base64)
	       :use-module (alterator homovector)

	       :use-module (alterator plist)
	       :use-module (srfi srfi-1)
	       :use-module (srfi srfi-2)

	       :re-export (;;
			   message?
			   message-header
			   message-cookie
			   message-method
			   message-uri
			   message-query
			   message-body

			   decode-url-component
			   encode-url-component)
	       :export (;; for internal use
			form-callback
			make-ajax-text
			make-ajax-response
			ui-callback
			make-ui-response
			;; basic form actions
			form-bind
			form-set-timeout
			form-replace
			form-update-enum
			form-update-value
			form-update-value-list
			form-value
			form-value-list
			form-update-visibility
			form-update-activity
			;; i18n
			_
			;; timeedit
			form-bind-timeedit
			;; form session
			form-session-ref
			form-session-set!
			;; file uploading
			form-bind-upload
			form-blob
			call-with-form-file
			;; error handling and messages
			catch/message
			form-warning
			form-error
			form-confirm
			;; cookie support
			form-update-cookie
			form-cookie
			;; json
			->json
			;; UI generation
			ui-blob
			ui-blob?
			ui-file
			ui-file?
			ui-replace
			ui-replace?))

;;; response generation

(define (init-response value)
  (ahttpd-session-set! 'ajax-response value))

(define (response-append fmt . args)
  (let ((response (ahttpd-session-ref 'ajax-response #f)))
    (or (null? response) (pair? response) (error "function called out of ajax context"))
    (ahttpd-session-set! 'ajax-response
			 (cons (apply format #f fmt args) response))))

(define (response-text)
  (string-concatenate (reverse (ahttpd-session-ref 'ajax-response))))

(define (response)
  (make-string-response 200
			;; "text/javascript;charset=utf-8"
			"text/html" ;; konqueror4 can load to iframe (file upload) this type only
			(response-text)))

(define (make-ajax-text uri callbackname)
  (init-response '())
  (form-callback callbackname uri)
  (response-text))

(define (make-ajax-response uri callbackname)
  (init-response '())
  (form-callback callbackname uri)
  (response))

(define (message-filename-set! msg file-name)
  (message-header-set! msg
		       "content-disposition"
		       (format #f "attachment;filename=~A" file-name)))

(define (make-ui-response uri)
  (init-response #f)
  (let ((answer (ui-callback uri)))
    (cond
      ((ui-blob? answer)
       (let* ((file-name (ui-blob-filename answer))
	      (response  (make-string-response 200
					       (ui-blob-type answer)
					       (u8vector->ISO-8859-1-string (ui-blob-data answer)))))
	 (and file-name (message-filename-set! response file-name))
	 response))
      ((ui-file? answer)
       (let* ((file-path (ui-file-path answer))
	      (response (make-file-response file-path "")))
	 (if (message? response)
	   (let ((file-type (ui-file-type answer))
		 (file-name (ui-file-filename answer)))
	     (and file-type (message-header-set! response "content-type" file-type))
	     (and file-name (message-filename-set! response file-name))
	     response)
	   (throw 'ui-error (format #f "ui-file: Unable to find file=~S~%" file-path)))))
      ((ui-replace? answer)
       (make-redirect-response (ui-replace-path answer)))
      (else
	;; ignore other answers
	#f))))

;;; callbacks storage

;; two level table:
;; 1st level: by session callback storage
;; 2nd level: by url callback storage

;; access 1st table
(define (callback-table1-ref)
  (let ((table1 (ahttpd-session-ref  'ajax-callback)))
    (if (hash-table? table1)
      table1
      (let ((table1 (make-hash-table 20)))
	(ahttpd-session-set! 'ajax-callback (make-hash-table 20))
	table1))))

;; access 2nd table
(define (callback-table2-ref url)
  (let* ((table1 (callback-table1-ref))
	 (table2 (hash-ref table1 url)))
    (if (hash-table? table2)
      table2
      (let ((table2 (make-hash-table 20)))
	(hash-set! table1 url table2)
	table2))))

;; remove 2nd table
(define (callback-table2-remove! url)
  (hash-remove! (callback-table1-ref) url))

;; get callback by url and by name
(define (callback-ref url name)
  (format (current-error-port) "callback-ref:url=~S,name=~S~%" url name)
  (hash-ref (callback-table2-ref url) name))

;; remove callback by url and by name
(define (callback-remove! url name)
  (format (current-error-port) "callback ~S for url ~S was removed~%" name url)
  (hash-remove! (callback-table2-ref url) name))

;; remove all callbacks by url
(define (callback-remove-all! url)
  (format (current-error-port) "all callbacks for url ~S was removed~%" url)
  (callback-table2-remove! url))

;; set callback by url and by name
;; singleshot callback will be removed from hash right after execution
(define (callback-set! url name proc . single-shot)
  (let ((single-shot (if (pair? single-shot) (car single-shot) #f)))
    (format (current-error-port) "add callback: url=~A, name=~A,single-shot=~A~%" url name single-shot)
    (hash-set! (callback-table2-ref url)
	       name
	       (if single-shot
		 (lambda() (callback-remove! url name) (proc))
		 proc))))

;;; dynamic resolving

(define (dynamic-callback url name)
  (format (current-error-port) "dynamic-callback,url=~S,name=~S~%" url name)
  (let ((url (string-trim url #\/)))
    (dynamic-require `(ui ,@(map string->symbol (string-split url #\/)) ajax)
		     (string->symbol name))))

;;; callbacks resolving

(define *random-state* (seed->random-state (current-time)))
(define *random-max* 100000000)

(define (form-callback name uri)
  (catch/ignore
    'form-error
    (lambda()
      ;;constructor: remove callbacks from previous call and call synonym
      (if (string=? name "init")
	(begin
	  (callback-remove-all! uri)
	  (form-callback "on-load" uri)))
      (let ((proc (or (callback-ref uri name)
		      (dynamic-callback uri name))))
	(and (procedure? proc) (proc))))))

(define (callback-name proc)
  (let ((real-name (procedure-name proc)))
    (if real-name
      (symbol->string real-name)
      (number->string (random *random-max* *random-state*) 16))))

(define (callback-id proc . single-shot)
  (let ((name (callback-name proc))
        (uri (ahttpd-session-ref 'ahttpd-uri)))
    (apply callback-set! uri name proc single-shot)
    name))

;; note: continuation procedure is a single shot
(define (continuation-id proc)
  (let ((name (ahttpd-session-id))
	(uri (ahttpd-session-ref 'ahttpd-uri)))
    (and (callback-ref uri name) (form-error "continuation in use"))
    (callback-set! uri name proc #t)
    name))

;;; json

(define (real-plist? x)
  (and (plist? x)
       (or (null? x)
	   (not-pair? (car x)))))

(define (->json obj)
  (cond
    ((and (number? obj) (zero? (imag-part obj))) (number->json obj))
    ((string? obj) (string->json obj))
    ((symbol? obj) (symbol->json obj))
    ((boolean? obj) (boolean->json obj))
    ((real-plist? obj)  (plist->json obj))
    ((list? obj) (list->json obj))
    (else (error "unsupported object for serialization" obj))))

(define (number->json num)
  (number->string (exact->inexact num)))

(define (string->json str)
  (string-append "\""
		 (string-quote (lambda (ch)
				 (case ch
				   ((#\cr) "")
				   ((#\newline) "\\n")
				   ((#\\) "\\\\")
				   ((#\") "\\\"")
				   (else (string ch))))
			       str)
		 "\""))

(define (symbol->json sym)
  (string->json (symbol->string sym)))

(define (boolean->json b)
  (string->json (if b "#t" "#f")))

(define (plist->json lst)
  (string-append "{"
		 (string-join
		   (plist-map (lambda(x y)
				(format #f "~A:~A" (->json x) (->json y)))
			      lst)
		   ",")
		 "}"))

(define (list->json lst)
  (string-append "["
		 (string-join (map ->json lst) ",")
		 "]"))

;;; user functions for forms

(define (cond-caddr x)
  (and (pair? x) (pair? (cdr x)) (pair? (cddr x)) (caddr x)))

(define (form-confirm msg . args)
  (let ((title (or (cond-car args) (_ "Question" "alterator-ahttpd")))
	(okButton (or (cond-cadr args) (_ "OK" "alterator-ahttpd")))
	(cancelButton (or (cond-caddr args) (_ "Cancel" "alterator-ahttpd")))
	(saved-session (ahttpd-session-pause)))
    (call-with-current-continuation
      (lambda(resume)
	(response-append "form_confirm(~A, {msg:~A, title:~A, okButton:~A, cancelButton:~A });~%"
			 (->json (continuation-id (lambda()
						    (let ((retcode (form-value "retcode")))
						      (ahttpd-session-resume saved-session)
						      (resume retcode)))))
			 (->json msg)
			 (->json title)
			 (->json okButton)
			 (->json cancelButton))
	(ahttpd-resume (response))))))

(define (form-bind name event proc)
  (response-append "form_bind(~A,~A,~A);~%"
		   (->json name)
		   (->json event)
		   (->json (callback-id proc))))


;; note: this is a singleshot callback
(define (form-set-timeout proc timeout)
  (response-append "form_set_timeout(~A,~A);~%"
		   (->json (callback-id proc #t))
		   (->json (inexact->exact (round (* timeout 1000))))))

(define (plist->url-args lst)
  (string-join
    (plist-map
      (lambda(x y) (format #f "~A=~A"
			   (encode-url-component (->string x))
			   (encode-url-component (->string y))))
      lst)
    "&"))

(define (form-replace url . url-args)
  (response-append "form_replace(~A);~%"
		   (->json
		     (string-append url
				    (if (pair? url-args)
				      (if (string-index url #\?) "&" "?")
				      "")
				    (plist->url-args url-args)))))

(define (form-update-enum name cmdlist)
  (response-append "form_update_enum(~A,~A);~%"
		   (->json name)
		   (->json (map cdr cmdlist))))

(define (form-update-value name value)
  ;; update local value cache
  (ahttpd-session-set! 'ahttpd-woo-args
		      (alist-set (string->symbol name)
				 value
				 (ahttpd-session-ref 'ahttpd-woo-args)))
  ;; notify interface
  (response-append "form_update_value(~A,~A);~%"
		   (->json name)
		   (->json value)))

;; (form-update-value-list namelist cmd)
;; or
;; (form-update-value-list cmd)
(define (form-update-value-list arg1 . arg2)
  (let* ((namelist (map string->symbol (if (pair? arg2) arg1 '())))
         (cmd (if (pair? arg2) (car arg2) arg1))
	 (cmdlist (if (pair? (cond-car cmd)) cmd (list cmd))))
    (for-each
      (lambda(cmd)
	(plist-for-each
	  (lambda(x y)
	    (if (or (null? namelist) (member x namelist))
	      (form-update-value (symbol->string x) y)))
	  (or (cond-cdr cmd) '())))
      cmdlist)))

(define (form-value-list . namelist)
  (apply backend-args (ahttpd-session-ref 'ahttpd-woo-args) namelist))

(define (form-value name . default)
  (apply cond-assoc (string->symbol name)
	 (ahttpd-session-ref 'ahttpd-woo-args)
	 default))

(define (form-update-visibility name status)
  (let ((namelist (if (pair? name) name (list name))))
    (for-each
      (lambda(name)
	(response-append "form_update_visibility(~A,~A)~%"
			 (->json name)
			 (if status "true" "false")))
      namelist)))

(define (form-update-activity name status)
  (let ((namelist (if (pair? name) name (list name))))
    (for-each
      (lambda(name)
	(response-append "form_update_activity(~A,~A)~%"
			 (->json name)
			 (if status "true" "false")))
      namelist)))

;;; timeedit synchronization between server and browser

(define (form-bind-timeedit name proc)
  (response-append "form_bind_timeedit(~A,~A);~%"
		   (->json name)
		   (->json (callback-id proc))))

;;; file uploading

(define (form-bind-upload name event filefieldname proc)
  (response-append "form_bind_upload(~A,~A,~A,~A);~%"
		   (->json name)
		   (->json event)
		   (->json filefieldname)
		   (->json (callback-id proc))))

(define (->u8vector data)
 (cond
   ((u8vector? data) data)
   ((string? data) (ISO-8859-1-string->u8vector data))
   (else (error "form-blob: unsupported data type"))))

(define (form-blob name . default)
  (cond
    ((assoc name (ahttpd-session-ref 'ahttpd-full-args))
     => (lambda(x)
	  (u8vector->base64-string (->u8vector (cdr x)) 0)))
    (else (and (pair? default) (car default)))))

(define (call-with-form-file name proc)
  (and-let* ((data (cond-assoc name (ahttpd-session-ref 'ahttpd-full-args)))
	     ((or (string? data) (u8vector? data)))
	     (temp-path (string-append "/var/cache/ahttpd/" name  "XXXXXX"))
	     (temp-port (mkstemp! temp-path))
	     (rewind (lambda() (and (access? temp-path F_OK) (delete-file temp-path)))))
    (if (u8vector? data)
        (uniform-vector-write data temp-port)
        (display data temp-port))
    (close-port temp-port)
    (with-throw-handler
      #t
      (lambda() (proc temp-path))
      (lambda args (rewind) (apply throw args)))
    (rewind)))

;;; simple message boxes

(define (form-message msg . title)
  (let ((title (if (pair? title) (car title) "")))
    (response-append "form_message({ msg:~A, title:~A })~%"
		     (->json msg)
		     (->json title))))

(define (form-error fmt . args)
  (form-message (apply format #f fmt args) (_ "Error" "alterator-ahttpd"))
  (throw 'form-error))

(define (form-warning fmt . args)
  (form-message (apply format #f fmt args) (_ "Attention" "alterator-ahttpd")))

(define form-woo-error form-error)

(define (form-type-error reasonlist)
  (response-append "form_type_error(~A)~%"
		   (->json (alist->plist reasonlist))))

(define (catch/message proc)
  (catch
    #t
    proc
    (lambda (key . args)
      (case key
	((woo-error)  (form-woo-error "~A" (car args)))
	((type-error) (form-type-error (car args)))
	(else (form-error "key=~S, args=~S" key args)))
      #f)))

(define (form-update-cookie name value)
  (response-append "form_update_cookie(~A,~A)~%"
		   (->json name)
		   (->json value)))

(define (form-cookie name)
  (message-cookie (ahttpd-session-ref 'ahttpd-request) name))

;;; Basic UI processing

(define (ui-callback uri)
  (catch/ignore
    'form-error
    (lambda()
      (let ((proc (dynamic-callback uri "ui")))
	(and (procedure? proc)
	     (proc))))))

(define (ui-blob name data . content-type)
  (cons* 'blob
	 (base64-string->u8vector (woo-get-option data (string->symbol name)))       ;; content
	 (if (pair? content-type) (car content-type) "application/octet-stream")     ;; mime-type
	 (and (pair? content-type) (pair? (cdr content-type)) (cadr content-type)))) ;; filename

(define (ui-blob? data)
  (and (pair? data)
       (eq? (car data) 'blob)))

(define (ui-blob-data data)
  (cadr data))

(define (ui-blob-type data)
  (caddr data))

(define (ui-blob-filename data)
  (cdddr data))

(define (ui-file name data . content-type)
  (cons* 'file
         (woo-get-option data (string->symbol name))                                 ;; link to file
         (and (pair? content-type) (car content-type))                               ;; mime-type
	 (and (pair? content-type) (pair? (cdr content-type)) (cadr content-type)))) ;; filename

(define (ui-file? data)
  (and (pair? data)
       (eq? (car data) 'file)))

(define (ui-file-path data)
  (cadr data))

(define (ui-file-type data)
  (caddr data))

(define (ui-file-filename data)
  (cdddr data))

(define (ui-replace url . url-args)
  (cons 'replace
	(string-append url
		       (if (pair? url-args)
			 (if (string-index url #\?) "&" "?")
			 "")
		       (plist->url-args url-args))))

(define (ui-replace? data)
  (and (pair? data)
       (eq? (car data) 'replace)))

(define (ui-replace-path data)
  (cdr data))

;;; i18n

(define (default-form-domain)
  (default-po-domain (ahttpd-session-ref 'ahttpd-uri)))

(define (_ str . domain)
  (let* ((domain (if (pair? domain) (car domain) (default-form-domain)))
	 (language (string-cut (form-value "language") #\;))
	 (tr (make-translator domain language)))
    (tr str)))

;;; per form and per session global variables

(define (form-session-key name)
  (string->symbol (string-append "-form-session-" (ahttpd-session-ref 'ahttpd-uri) "-" name)))

(define (form-session-ref name)
  (ahttpd-session-ref (form-session-key name)))

(define (form-session-set! name value)
  (ahttpd-session-set! (form-session-key name) value))
