(define-module (alterator algo)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-11)
  :use-module (srfi srfi-2)
  :re-export (->bool)
  :export (
	   ;;public
           cond-car
           cond-cadr
           cond-cdr
           append1
           list-remove
           string-list-index

           alist-set
           cond-assoc
           cond-assq

           begin-1
           make-cell
           cell-set!
           cell-ref

	   ;;for internal purposes
	   with-first-readable
	   read-access?
	   write-access?

	   eat-rest

           sure-symbol
           thunk
	   compose

	   fold-values

           catch/ignore
           dynamic-require))

;description: some usefull algorithms, that I cannot find in srfi's

;;; read enchancements

;eat rest space characters from port
(define (eat-rest port)
  (let loop ()
    (and-let* ( ((char-ready? port))
                (char (peek-char port))
                ((char? char))
                ((char-whitespace? char)))
              (read-char port)
              (loop))))

;;; common file selection
(define (read-access? file)
  (and (string? file)
       (access? file R_OK)))

(define (write-access? file)
  (and (string? file)
       (access? file W_OK)))

(define (with-first-readable filelist proc)
  (let ((item (find read-access? filelist)))
    (and item (proc item))))
   
;;; conditional version of some operations

(define (cond-car key) (and (pair? key) (car key)))
(define (cond-cdr key) (and (pair? key) (cdr key)))
(define (cond-cadr key) (and (pair? key) (pair? (cdr key)) (cadr key)))

;;; symbols

(define (sure-symbol x)
  (cond
    ((symbol? x) x)
    ((macro? x) (macro-name x))
    ((string? x) (string->symbol x))
    (else (error "sure-symbol unsupported type for conversion" x))))

;;; lists
(define (append1 lst item)
   (append lst (list item)))

(define (list-remove lst num)
  (let-values (((head tail) (split-at lst num)))
    (append head (cdr tail))))

(define (string-list-index str lst)
  (list-index (lambda (x) (string=? str x)) lst))

(define (fold-values proc init-list lst)
  (apply values
         (fold (lambda(x res)
                 (call-with-values (lambda() (apply proc x res)) list))
               init-list
               lst)))

;;; syntatic sugar
(define (apply1 x y)
  (apply x y '()))
(define (compose . functions)
  (lambda(x)
    (fold-right apply1 x functions)))


;shortcut for simplest lambda 
(define-macro (thunk <first> . <next>)
    `(lambda() ,<first> . ,<next>))

;as a begin, but return result of 1st expression
(define-macro (begin-1 <first> . <next>)
  (let ((result (gensym)))
    `(let ((,result ,<first>))
       ,@<next>
       ,result)))

;;; alist functions
;we can use fold here, but we made a little optimization here
(define (alist-set name value alist)
  (let loop ((current alist)
             (result '()))
    (cond
     ((null? current) (append1 result (cons name value)))
     ((equal? name (caar current)) (append result (list (cons name value)) (cdr current)))
     (else (loop (cdr current) (append1 result (car current)))))))

(define (cond-assoc name alist . default)
  (cond
   ((assoc name alist) => cdr)
   (else (if (pair? default) (car default) #f))))

(define (cond-assq name alist . default)
  (cond
   ((assq name alist) => cdr)
   (else (if (pair? default) (car default) #f))))

;;; box and unbox

(define (make-cell value) (list value))
(define (cell-ref box) (car box))
(define (cell-set! box value) (set-car! box value))


;;; to boolean convertions
(define (->bool x) (not (not x)))

;;; exceptions
(define (catch/ignore key proc)
  (catch key
	 proc
	 (lambda (key . args) #f)))

;;; dynamic loaded libraries
(define (dynamic-require library sym)
  (let ((module (resolve-module library)))
    (and module (module-ref module sym #f))))
