(define-module (alterator backend meta)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-2)
  :use-module (alterator algo)
  :use-module (alterator plist)
  :use-module (alterator metalterator)
  :export (backend meta-cmd))

(define (objects->option-path objects)
  (string-append (objects->path objects) ".scm"))

(define (error-code args)
  (let ((p (last-pair args)))
    (and (list? p)
         (list (car p))
         (caar p))))

(define (no-object-error? key args)
  (and (eq? key 'system-error)
       (eq? 2 (error-code args))))

(define (unlink-missing path)
  (cond
   ((not (access? path F_OK))
    (catch 'system-error
      (lambda ()
        (cond
         ((eq? (stat:type (lstat path)) 'symlink)
          (print-debug 5 "Delete missing symlink ~s" path)
          (delete-file path))))
      (lambda (key . args) #f))
    #f)
   (else path)))

(define (on-error path)
  (lambda (key . args)
    (print-debug 1 "Reporting error for ~s: ~s . ~s"
                 path key args)
    (append (list 'error)
            (cond
             ((no-object-error? key args)
              (unlink-missing path)
              (list "no-such-object"))
             (else
              (list
               (apply format #f (cadr args)
                      (caddr args))))))))

(define (select-options query options)
  (print-debug 5 "Select options: ~s | ~s" query options)
  (fold (lambda (qpair reslist)
          (let* ((mapping (first-value qpair))
                 (mapped (if (procedure? mapping)
                             (mapping (assoc-ref options (car qpair)))
                             #f))
                 (name (if mapped
                           (if (pair? mapped) (car mapped) (car qpair))
                           mapping)))
            (if (or (eq? #f name)
                    (and (not (eq? name #t))
                         (equal? "#{\\#f}#" (sure-string name))))
                reslist
                (append reslist
                        (list
                         (cons (if (or (eq? #t name)
                                       (equal? "#{\\#t}#" (sure-string name)))
                                   (car qpair)
                                   (sure-symbol name))
                               (if mapped
                                   (if (pair? mapped) (cdr mapped) mapped)
                                   (assoc-ref options (car qpair)))))))))
          '()
          query))

(define (update-options new-opts old-opts)
  (print-debug 4 "Update options: ~s -> ~s" old-opts new-opts)
  (fold (lambda (pair new-opts)
          (assoc-set! new-opts
                      (car pair)
                      (cdr pair)))
        old-opts
        new-opts))

(define (read-some-options query opt-path)
  (print-debug 5 "Read some options: ~s | ~s" query opt-path)
  (if (not (null? query))
      (call-with-input-file opt-path
        (lambda (port)
          (select-options query (read port))))
      '()))

(define (read-all-options opt-path)
  (print-debug 5 "Read all options: ~s" opt-path)
  (call-with-input-file opt-path
    (lambda (port)
      (read port))))

(define (read-options objects options)
  (print-debug 3 "Read options: ~s | ~s" options objects)
  (let ((opt-path (objects->option-path objects)))
    (catch 'system-error
      (lambda ()
        (alist->plist
         (if (null? options)
             (read-all-options opt-path)
             (read-some-options options opt-path))))
    (on-error opt-path))))

(define (write-down-options opt-path options)
  (print-debug 4 "Write down options: ~s -> ~s" options opt-path)
  (call-with-output-file opt-path
    (lambda (port) (write options port)))
  '())

(define (write-options objects options)
  (print-debug 3 "Write options: ~s -> ~s" options objects)
  (let ((opt-path (objects->option-path objects)))
    (catch 'system-error
      (lambda ()
        (write-down-options opt-path
         (update-options options
                         (read-all-options opt-path))))
      (lambda (key . args)
        (cond
         ((no-object-error? key args)
          (unlink-missing opt-path)
          (print-debug 4 "Check container object: ~s"
                         (reverse (cdr (reverse objects))))
          (if (not (access? (objects->option-path
                              (reverse (cdr (reverse objects))))
                            F_OK))
	    (begin
              (print-debug 4 "Create container object: ~s"
                           (reverse (cdr (reverse objects))))
              (mkdir (objects->path
                      (reverse (cdr (reverse objects)))))))
          (write-down-options opt-path options))
         (else
          (apply (on-error opt-path) key args)))))))

(define (create-new-object objects write-down-proc)
  (let ((path
         (fold
          (lambda (obj path)
            (if (string? path)
                (let ((path (string-append path
                                           "/"
                                           (sure-string obj))))
                  (print-debug 5 "Check path ~s" path)
                  (if (access? path F_OK)
                      path
                      (catch 'system-error
                        (lambda ()
                          (print-debug 4 "Create container object: ~s" path)
                          (mkdir path)
                          path)
                        (on-error path)))
                  path)))
          ""
          (reverse (cdr (reverse objects))))))
    (if (list? path)
        path
        (let ((obj-path (objects->path objects))
              (opt-path (objects->option-path objects)))
          (if (or (access? obj-path F_OK)
                  (access? opt-path F_OK))
              `(error "object-already-exists" ,obj-path)
              (catch 'system-error
                (lambda ()
                  (write-down-proc opt-path))
                (on-error opt-path)))))))

(define (create-object objects options)
  (print-debug 3 "Create object: ~s -> ~s" options objects)
  (create-new-object objects
                     (lambda (opt-path)
                       (write-down-options opt-path options))))

(define (options-name? name)
  (let ((pos (- (string-length name) 4)))
    (and (> pos 0)
         (let ((suffix (substring name pos)))
           (and (equal? suffix ".scm")
                (substring name 0 pos))))))

(define (with-finally thunk thunk-after)
  (catch #t
    (lambda ()
      (let ((res (thunk)))
        (thunk-after)
        res))
    (lambda (key . args)
      (thunk-after)
      (apply throw key args))))

(define (scandir dir-path options)
  (print-debug 4 "Open directory ~s for reading" dir-path)
  (let ((dir (opendir dir-path)))
    (with-finally
      (lambda ()
        (let loop ((obj (readdir dir))
                   (obj-list '()))
          (if (not (eof-object? obj))
              (or
               (and-let*
                ((obj-path (string-append dir-path "/" obj))
                 (obj-stat (catch 'system-error
                             (lambda ()
                               (stat obj-path))
                             (lambda (key . args)
                               (if (no-object-error? key args)
                                   (unlink-missing obj-path))
                               #f))))
                (print-debug 4 "Next entry: ~s" obj)
                (case (stat:type obj-stat)
                  ((regular)
                   (cond
                    ((options-name? obj) =>
                     (lambda (obj-name)
                       (loop (readdir dir)
                             (assoc-set! obj-list obj-name
                                         (alist->plist
                                          (read-some-options options
                                                             obj-path))))))
                    (else
                     (loop (readdir dir) obj-list))))
                  ((directory)
                   (if (and (not (eq? #\. (string-ref obj 0)))
                            (not (assoc-ref obj-list obj)))
                       (loop (readdir dir)
                             (assoc-set! obj-list obj '()))
                       (loop (readdir dir) obj-list)))
                  (else
                   (loop (readdir dir) obj-list))))
               (loop (readdir dir) obj-list))
              obj-list)))
      (lambda ()
        (print-debug 4 "Close directory ~s" dir-path)
        (closedir dir)))))

(define (list-object objects options)
  (print-debug 3 "List object's subobjects: ~s" objects)
  (let ((obj-path (objects->path objects)))
    (catch 'system-error
      (lambda ()
        (cond
         ((access? obj-path F_OK)
          (scandir obj-path options))
         (else
          (let ((opt-path (objects->option-path objects)))
            (catch 'system-error
              (lambda ()
                (if (access? opt-path F_OK)
                    '()
                    (list 'error "no-such-object" obj-path)))
              (on-error opt-path))))))
      (on-error obj-path))))

(define (delete-object objects)
  (print-debug 3 "Delete object ~s" objects)
  (let* ((obj-path (unlink-missing (objects->path objects)))
         (opt-path (unlink-missing (objects->option-path objects)))
         (err-obj
          (and obj-path
               (let loop ((obj-list (list-object objects '())))
                 (cond
                  ((null? obj-list)
                   (catch 'system-error
                     (lambda ()
                       (print-debug 4 "Delete container ~s" obj-path)
                       (rmdir obj-path)
                       #f)
                     (on-error obj-path)))
                  ((eq? (car obj-list) 'error) obj-list)
                  (else
                   (let ((answer (delete-object
                                  (append objects (car obj-list)))))
                     (if (not (null? answer))
                         answer
                         (loop (cdr obj-list)))))))))
        (err-opt
         (and opt-path
              (catch 'system-error
                (lambda ()
                  (print-debug 4 "Delete file ~s" opt-path)
                  (delete-file opt-path)
                  #f)
                (on-error opt-path)))))
    (or err-obj err-opt '())))

(define (link-new-object opt-path to)
  (catch 'system-error
    (lambda ()
      (symlink (string-append to ".scm") opt-path)
      '())
    (on-error opt-path)))

(define (link-object objects to)
  (print-debug 3 "Link object: ~s -> ~s" objects to)
  (create-new-object objects
                     (lambda (opt-path)
                       (link-new-object opt-path to))))

(define (inc-val val)
  (+ 1 (string->number (sure-string val))))

(define (read-next-options objects query)
  (print-debug 3 "Read next id property: ~s | ~s" query objects)
  (let ((opt-path (objects->option-path objects)))
    (catch 'system-error
      (lambda ()
        (let ((options
               ; TODO: lock
               (read-all-options opt-path)))
          (alist->plist
           (let loop ((next-options '())
                      (updated-options '())
                      (options options))
             (cond
              ((null? options)
               (write-down-options opt-path
                                   updated-options)
               next-options)
              ((assoc-ref query (caar options)) =>
               (lambda (qname)
                 (loop (append next-options
                               (list
                                (cons (sure-symbol qname)
                                      (cdar options))))
                       (append updated-options
                               (list
                                (cons (caar options)
                                      (inc-val (cdar options)))))
                       (cdr options))))
              (else
               (loop next-options
                     (append updated-options
                             (list (car options)))
                     (cdr options))))))))
      (on-error opt-path))))

(define (backend objects options)
  (print-debug 1 "Command: ~s ~s" objects options)
  (let* ((options (plist->alist options))
         (action (sure-symbol (assoc-ref options 'action)))
         (options (alist-delete 'action options))
         (answer
          (cond
           ((null? objects)
            (case action
              ((read)
               `(debug-level ,(current-debug-level)))
              ((write)
               (cond
                ((assoc-ref options 'debug-level) =>
                 (lambda (level)
                   (set-debug-level! (if (number? level)
                                         level
                                         (string->number (sure-string level)))))))
               '())
              (else
               `(error ,(format #f "Unknown management action: ~s" action)))))
           (else
            (case action
              ((read)
               (read-options objects options))
              ((write)
               (write-options objects options))
              ((list)
               (list-object objects options))
              ((new)
               (create-object objects options))
              ((delete)
               (delete-object objects))
              ((link)
               (cond
                ((assoc-ref options 'name) =>
                 (lambda (name)
                   (link-object objects name)))
                (else
                 `(error "No link name specified"))))
              ((read-next)
               (read-next-options objects options))
              (else
               `(error ,(format #f "Unknown action: ~s" action))))))))
    (print-debug 2 "Answer: ~s" answer)
    answer))

(define (meta-cmd objects options)
  (let* ((objects (string-split (meta-name (car objects) (cdr objects)) #\/))
	 (cmdlist (backend (cdr objects) options)))
    (map (lambda (obj)
           (if (not (or (null? obj)
                    (plistq 'name (cdr obj))))
	       (append (list (car obj) 'name (car obj))
	               (cdr obj))
	       obj))
	 (cond
	  ((null? cmdlist) '())
	  ((not (pair? (car cmdlist)))
	   (if (eq? 'error (car cmdlist))
	       (throw 'meta-error
	              (string-join (map sure-string (cdr cmdlist))))
	       (list (cons* (car (reverse objects)) cmdlist))))
	  (else cmdlist)))))
