(define-module (alterator backend squid)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-2)
  :use-module (srfi srfi-13)
  :use-module (alterator algo)
  :use-module (alterator str)
  :use-module (alterator woo)
  :use-module (alterator plist)
  :use-module (alterator gettext)
  :use-module (alterator metalterator)
  :use-module (alterator metalterator match))

;; (define (semicolonize names options)
;;   (plist-fold (lambda (name val options)
;;                 (append options
;;                         (list name)
;;                         (if (member name names)
;;                             (list
;;                              (string-join (map sure-string
;;                                                (cond
;;                                                 ((not val) '())
;;                                                 ((list? val) val)
;;                                                 (else (list val))))
;;                                           ";"))
;;                             (list val))))
;;               '()
;;               options))

;;; Appends two woo commands.
(define (woo-append cmd1 cmd2)
  (if (or (null? cmd2)
          (not (list? cmd2))
          (and (pair? cmd1) (eq? 'error (car cmd1))))
      cmd1
      (if (null? cmd1)
          cmd2
          (if (pair? cmd1)
              (if (list? (car cmd1))
                  (if (list? (car cmd2))
                      (append cmd1 cmd2)
                      (map (lambda (l) (append l cmd2)) cmd1))
                  (append cmd1
                          (if (not (list? (car cmd2)))
                              cmd2
                              (car cmd2))))
              cmd1))))

;;; Reads the service status value.
(define (read-status)
  (list 'status
        (woo-get-option (woo-read-first "/squid-commit") 'status)))

(define (reconfigure . options)
  (apply woo-write "/squid-commit" 'commit #t options))

(define (format-network net)
  (let* ((address (cond-plistq 'address (cdr net)))
         (label (cond-plistq 'label (cdr net)))
         (label (string-append address
                               (if (and label (not (string-null? label)))
                                   (string-append " (" label ")")
                                    ""))))
    (append (list (car net))
            (plist-set! 'label label (cdr net)))))

(define *all-users* "___all")
(define *auth-users* "___authenticated")

(define (init-groups)
  (if (not (object-exists? 'squid (list 'groups *all-users*)))
    (meta new 'squid (list 'groups *all-users*)
                     '(comment "All users" deny #f suffix "")))
  (if (not (object-exists? 'squid (list 'groups *auth-users*)))
    (meta new 'squid (list 'groups *auth-users*)
                     '(comment "Authenticated users" deny #f
                       suffix ""))))

(define (translate-groups _ grp-list)
  (map (lambda (grp)
         (cons (car grp)
               (cond
                 ((equal? (car grp) *all-users*)
                  (plist-set! 'label (_ "All users") (cdr grp)))
                ((equal? (car grp) *auth-users*)
                 (plist-set! 'label (_ "Authenticated users")
                             (cdr grp)))
                (else (cdr grp)))))
       grp-list))

(define (plist-update-values pl1 pl2 pred)
  (plist-fold (lambda (n v params1)
                (if (pred n params1 pl2)
                  (plist-set! n v params1)
                  params1))
              pl1
              pl2))

(define (woo-list-apply-defaults lst defaults)
  (map
    (lambda (obj)
      (cons (car obj)
            (plist-update-values (cdr obj)
                                 defaults
                                (lambda (n pl1 pl2)
                                  (not (cond-plistq n pl1))))))
    lst))

(define (woo-list-merge . lists)
  (fold
    (lambda (l2 l1)
      (append
        (map (lambda (obj1)
               (cons (car obj1)
                 (cond
                   ((find (lambda (obj2)
                            (equal? (car obj1) (car obj2)))
                          l2) =>
                    (lambda (obj2)
                      (plist-update-values (cdr obj1)
                                           (cdr obj2)
                                           (lambda (n pl1 pl2) #t))))
                 (else (cdr obj1)))))
             l1)
        (filter (lambda (obj2)
                  (not (find (lambda (obj1)
                               (equal? (car obj1) (car obj2)))
                             l1)))
                l2)))
    '()
    lists))

(define (woo-list-name-objects lst)
  (map (lambda (obj)
         (cond
           ((cond-plistq 'name (cdr obj)) =>
            (lambda (n) (cons n (plist-delete 'name (cdr obj)))))
           (else obj)))
       lst))

(define (list-virtual-groups)
  (list
    (cons *all-users*
          (meta read 'squid (list 'groups *all-users*)
                            '(deny #t suffix #t comment "label")))
    (cons *auth-users*
          (meta read 'squid (list 'groups *auth-users*)
                            '(deny #t suffix #t "label")))))

(define (list-stored-groups)
  (filter (lambda (obj2)
            (not (find (lambda (obj1)
                         (equal? (car obj1) (car obj2)))
                       (list-virtual-groups))))
          (meta list 'squid '(groups)
                     '(deny #t comment "label"))))

(define (backend objects options)
  (print-debug 1 "[squid] Command: ~s ~s" objects options)
  (let* ((language (string-cut (sure-value options 'language "en_US") #\;))
         (_ (make-translator "alterator-squid" language)))
    (woo-case (objects options)
      ; General settings
      ((type ())
       (list 'http_port "tcp-port"))
      ((read ())
       (woo-append (meta read 'squid '()
                         (list 'http_port #t 'auth-mode (lambda (v) (or v "no-auth"))
                               'mode (lambda (v) (or v "normal"))))
                   (read-status)))
      ((write () ('http_port @p 'auth-mode @a 'mode @m 'status @s))
       (let* ((settings (meta read 'squid '() '(auth-mode #t)))
              (last-auth-mode (cond-plistq 'auth-mode
                                           settings
                                           "no-auth")))
         (meta write 'squid '()
               (list 'http_port @p 'auth-mode (or @a "no-auth")
                     'mode (or @m "normal")))
         (reconfigure 'status @s
                      'restart (not (equal? (or @a "no-auth")
                                            last-auth-mode)))))
      ((list ('modes))
       `(("transparent" label ,(_ "Transparent"))
         ("normal" label ,(_ "Normal"))))
      ((list ('auth-modes))
       `(("no-auth" label ,(_ "No authentication"))
         ("kerb" label ,(_ "Kerberos"))
         ("pam" label ,(_ "PAM"))
         ("kerb+pam" label ,(_ "Kerberos+PAM"))))
      ((list ('proxy-methods))
       `(("PROXY" label ,(_ "Proxy"))
         ("CONNECT" label ,(_ "Passthrough"))))
      ; Port-related ACLs
      ((read ('safe-ports)) '())
      ((list ('safe-ports))
       (meta list 'squid (list 'safe-ports)
             (list 'start_port #t 'end_port #t 'method #t
                   'transparent #t 'comment "label")))
      ((read-next ('safe-ports))
       (meta read-next 'squid (list 'safe-ports) (list 'next-id "id")))
      ((new ('safe-ports @n)
            ('start_port @s 'end_port @e 'method @m
             'transparent @t 'comment @c))
       (meta new 'squid (list 'safe-ports @n)
             (list 'start_port @s 'end_port @e 'method @m
                   'transparent @t 'comment @c))
       (reconfigure))
      ((delete ('safe-ports @n))
       (meta delete 'squid (list 'safe-ports @n) '())
       (reconfigure))
      ((type ('safe-ports @n))
       (list 'start_port "tcp-port" 'end_port "tcp-port"))
      ((read ('safe-ports @n))
       (meta read 'squid (list 'safe-ports @n)
             (list 'start_port #t 'end_port #t 'method #t
                   'transparent #t 'comment #t)))
      ((write ('safe-ports @n)
              ('start_port @s 'end_port @e 'method @m
               'transparent @t 'comment @c))
       (meta write 'squid (list 'safe-ports @n)
             (list 'start_port @s 'end_port @e 'method @m
                   'transparent @t 'comment @c))
       (reconfigure))
      ; Network-related ACLs
      ((read ('networks)) '())
      ((list ('networks))
       (map format-network
            (meta list 'squid (list 'networks)
              (list 'address #t 'comment "label"))))
      ((read-next ('networks))
       (meta read-next 'squid (list 'networks) (list 'next-id "id")))
      ((new ('networks @n) ('address @a 'comment @c))
       (meta new 'squid (list 'networks @n)
             (list 'address @a 'comment @c ))
       (reconfigure))
      ((delete ('networks @n))
       (meta delete 'squid (list 'networks @n) '())
       (reconfigure))
      ((type ('networks @n))
       (list 'address "ipv4-network"))
      ((read ('networks @n))
       (meta read 'squid (list 'networks @n)
             (list 'address #t 'comment #t)))
      ((write ('networks @n) ('address @a 'comment @c))
       (meta write 'squid (list 'networks @n)
             (list 'address @a 'comment @c))
       (reconfigure))
      ; Group-based access
      ((read ('groups)) '())
      ((list ('groups))
       (init-groups)
       (woo-list-apply-defaults
         (woo-list-merge
           (translate-groups _ (list-virtual-groups))
           (translate-groups _ (list-stored-groups))
           (woo-catch
             (lambda ()
               (woo-list-name-objects
                 (woo-list "/ldap-groups/avail_groups")))
             (lambda (arg) '())))
         '(deny #f suffix "")))
      ((type ('groups @n))
       (list 'deny "boolean"))
      ((read ('groups @n))
       (if (object-exists? 'squid (list 'groups @n))
         (append
           (meta read 'squid (list 'groups @n)
                             (list 'deny #t 'suffix (lambda (v)
                                                      (or v ""))))
           (list 'label
                 (cond
                   ((eq? @n (string->symbol *all-users*))
                    (_ "All users"))
                   ((eq? @n (string->symbol *auth-users*))
                    (_ "Athenticated users"))
                   (else @n))))
         (list 'deny #f 'suffix "" 'label @n)))
      ((write ('groups @n) ('deny @d 'suffix @s))
       (meta write 'squid (list 'groups @n) (list 'deny @d 'suffix @s))
       (reconfigure))
      (else #f))))
