(define-module (alterator ensign)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-2)
  :use-module (srfi srfi-13)
  :use-module (alterator str)
  :use-module (alterator plist)
  :use-module (alterator algo)
  :use-module (alterator woo)
  :use-module (alterator gettext)
  :use-module (alterator backend3)

  :use-module (alterator exit-handler)

  :export (ensign-query
           ensign))

;;; default backends

(define (make-backend name)
  (catch/ignore
    #t
    (lambda()
      (let ((proc (dynamic-require `(alterator backend ,(string->symbol name)) 'backend)))
	(and (procedure? proc) proc)))))

(define (ensign-backend objects options)
  (let ((action (cond-plistq 'action options "")))
      (cond
        ;;protect from exit
	((string=? action "quit")
	  #f)
	((string=? action "delete")
         (let* ((name (car objects))
                (mariner (resolv-mariner name)))
           (and mariner (forget-mariner mariner))
         '())))))

;;; low level backend processing

(define *mariners* (make-cell (acons "ensign" (cons ensign-backend (current-time)) '())))
(define *clean-interval* (make-cell (* 60 60 5)))

(define (quit-mariner mariner)
  (not ((cadr mariner) '() '(action "quit"))))

(define (forget-mariner mariner)
  (or (quit-mariner mariner)
      (cell-set! *mariners* (alist-delete (car mariner) (cell-ref *mariners*)))))

(define (new-mariner? mariner now)
  (< (- now (cddr mariner)) (cell-ref *clean-interval*)))

(define (sweep-mariner)
  (let ((now (current-time)))
    (cell-set! *mariners*
	       (filter
		 (lambda(x) (or (new-mariner? x now) (quit-mariner x)))
		 (cell-ref *mariners*)))))

(define (touch-mariner x)
    (set-cdr! (cdr x) (current-time))
    (sweep-mariner))

(define (resolv-mariner sym)
  (and-let* ((mariner (assoc sym (cell-ref *mariners*))))
    (touch-mariner mariner)
    mariner))

(define (load-mariner sym)
  (and-let* ((backend-proc (or (make-backend sym) (make-backend3 sym)))
             (backend-data (cons* sym backend-proc (current-time))))
	    (cell-set! *mariners* (cons backend-data (cell-ref *mariners*)))
	    backend-data))

(define (run-mariner mariner cmd)
  (define (pair-return z) (and (list? z) z))
   (or (pair-return ((cadr mariner) (cdar cmd) (cdr cmd)))
	(woo-error "unknown action for backend: " (car mariner))))

(define (call-mariner cmd)
  (let* ((name (car cmd))
	 (sym (car name))
	 (mariner (or (resolv-mariner sym) (load-mariner sym))))
    (cond
      ((not mariner) (woo-error "backend not found: " sym))
      (else
	(catch 'mariner-error
	       (lambda()
		 (run-mariner mariner cmd))
	       (lambda (key . args)
		 (forget-mariner mariner)
		 (woo-error "backend-communication error: " (format #f "~S" args))))))))

;;; highlevel backend processing

(define (error-answer? cmd) (and (pair? cmd) (eq? (car cmd) 'error)))
(define (quote-answer? cmd) (and (pair? cmd) (eq? (car cmd) 'quote)))
(define (list-answer? cmd)  (and (pair? cmd) (pair? (car cmd))))

(define (out-normal name options)
  (cons name options))

(define (out-list name options)
   (if (string? (car options))
       (out-normal (string-append name "/" (car options))
                   (cons* 'name (car options) (cdr options)))
       (out-normal name options)))

(define (prepare-name name)
  (if (pair? name)
      name
      (and (not-empty-string? name)
           (not (string-index name #\*))
           (string-cut-repeated name #\/))))

(define (prepare-cmd cmd)
  (or (and-let* (((pair? cmd))
		 (name (prepare-name (car cmd)))
		 ((not (null? name)))
		 ((not (member ".." name))))
		(cons name (cdr cmd)))
      (error (format #f "wrong command:~S~%" cmd))))

(define (ensign-raw-query cmd)
  (let* ((cmd  (prepare-cmd cmd))
	 (name (string-append "/" (string-join (car cmd) "/")))
	 (answer (call-mariner cmd)))
    (cond
      ((null? answer) '())
      ;;answer will be destroyed from output queue, use (quote) for really empty answers
      ((error-answer? answer) (woo-error (cadr answer)))
      ((quote-answer? answer) (list (cdr answer)))
      ((list-answer? answer)
       (map (lambda (x)
	      (if (quote-answer? x)
		(cdr x)
		(out-list name x)))
	    answer))
      ((pair? answer) (list (out-normal name answer)))
      (else
	(woo-error "unsupported answer type")))))

;;; type checking

(define *typelist* '())

(define (acons! key value lst)
  (set! lst (acons key value lst))
  value)

(define (get-typelist url)
  (cond
    ((assoc url *typelist*) => cdr)
    (else
     (catch/ignore
       #t
       (lambda()
         (let ((typelist (cond-car (ensign-raw-query `(,url action "type")))))
           (and (pair? typelist)
		(acons! url (cdr typelist) *typelist*))))))))

(define *type* '())

(define (get-type name)
 (catch/ignore
   #t
   (lambda()
     (cond
       ((assoc name *type*) => cdr)
       (else
         (acons! name
                 (or (dynamic-require `(type ,(string->symbol name)) 'type)
                     (dynamic-require `(alterator type ,(string->symbol name)) 'type))
	         *type*))))))


(define (test-type p t v _)
  (catch 'type-error
	 (lambda()
	   (let ((tproc (get-type t)))
	     (and (procedure? tproc) (tproc v _))
	     #f))
	 (lambda(key reason)
	   (cons (symbol->string p) reason))))

(define *not-found* (lambda() 'x))

(define (make-ensign-translator hsh)
  (lambda (str . domain)
    (translate hsh str (if (pair? domain) (car domain) "_") "alterator")))

(define (make-ensign-translator-cache cmd)
  (let* ((l (woo-get-option cmd 'language "en_US")))
    (make-translator-cache (if (pair? l) l (string-split l #\;)))))

(define (ensign-test-query cmd)
  (let* ((url (->string (car cmd)))
	 (action (woo-get-option cmd 'action "write"))
	 (typelist (or (get-typelist url) '()))
	 (hsh (make-ensign-translator-cache cmd)))
    (and (not (member action '("read" "list" "constraints")))
	 (filter pair?
		 (plist-map (lambda(p t)
			      (let ((v (woo-get-option cmd p *not-found*)))
				(or (eq? v *not-found*)
				    (test-type p t v (make-ensign-translator hsh)))))
			    typelist)))))

;;;main entry point
(define (ensign-query cmd)
  (with-fluids ((woo-gate ensign-query))
	       (let ((results (ensign-test-query cmd)))
		 (if (pair? results)
		   (type-error results)
		   (ensign-raw-query cmd)))))


(define (ensign)
  (lambda(cmds next)
    (next (append-map ensign-query cmds))))

;;; cleanup backend processes at finish

(define (stop-mariners . ignore)
  (cell-set! *mariners* (filter quit-mariner (cell-ref *mariners*))))

(sigaction SIGHUP stop-mariners) ;;at sighup reset
(at-exit stop-mariners); at exit reset
