(set! %load-path (cons "/usr/share/alterator/lookout" %load-path))

(define-module (alterator lookout)
  ;;common modules
  :use-module (alterator object)
  :use-module (alterator algo)
  :use-module (alterator plist)
  :use-module (alterator str)
  :use-module (alterator woo)

  ;;simplest message passing
  :use-module (alterator mailbox)

  ;;lookout modules
  :use-module (alterator lookout root)

  ;;session modules
  :use-module (alterator session common)

  :export (lookout))

(define (auth-permission-granted content)
  (define-operation get-id)
  (define-operation get-user)
  (define-operation resume)

  (let ((session (fluid-ref generic-session)))
    `(auth-answer user ,(get-user session)
                  session-id ,(number->string (get-id session))
                  content ,(resume session content))))

;;open a new session on '/ document and pass initial parameters to it
(define (auth-permission-denied url . url-args)
  (define-operation start)
  (define-operation set-lang)
  (with-fluids ((generic-session (session-open)))
               (let ((session (fluid-ref generic-session)))
                 (if session
                     (begin
                       (set-lang session
                                 (string-cut (or (cond-plistq 'language url-args) "en_US") #\;))
                       (start session
                              (thunk (apply document:popup url url-args) (session-close)))
                       (auth-permission-granted '(alterator-request action "re-get")))
                     '(auth-answer content ((command action "overflow")))))))

;;process incoming lookout message
(define (lookout-request session content url . url-args)
  (if session
      (with-fluids ((generic-session session))
                   (auth-permission-granted content))
      (apply auth-permission-denied url url-args)))

;;process incoming mailbox message
(define (mailbox-request session content)
  (define-operation read-messages)
  (run-mailbox (cdr content))
  `(auth-answer
    content ,(if session (read-messages session) '())))

;;authenticate incoming message
(define (auth-request cmd url . url-args)
  (let ((content (woo-get-option cmd 'content #f))
	(user	 (woo-get-option cmd 'user #f))
	(session-id (woo-get-option cmd 'session-id #f))
	(url-args (append url-args (cdr cmd))))

    (cond ;;switch to session or open a new one for '/ document
      (session-id
       (let ((session (session-find user session-id)))
         (case (cond-car content)
           ((mailbox-request) (mailbox-request session content))
           (else (lookout-request session content url url-args)))))
      (else (apply auth-permission-denied url url-args)))))

;main entry point

;;get next command from upper level pass it to engine, process only single command here
(define (lookout-main cmds next url . url-args)
  (list (apply auth-request (car cmds) url url-args)))

(define (make-lookout-gate next)
  (lambda(cmd)
    (next
      (list (cons* (car cmd)
		   'language (session-language)
		   (cdr cmd))))))

(define (lookout url . url-args)
  (lambda (cmds next)
    (with-fluids ((woo-gate (make-lookout-gate next)))
                 (apply lookout-main cmds next url url-args))))

