(define-module (alterator lookout attributes)

  :use-module (srfi srfi-1)
  :use-module (srfi srfi-13)
  :use-module (srfi srfi-2)
  :use-module (alterator object)
  :use-module (alterator algo)
  :use-module (alterator woo)
  :use-module (alterator str)
  :use-module (alterator vector)
  :use-module (alterator container)
  :use-module (alterator presentation common)
  :use-module (alterator presentation container)
  :use-module (alterator lookout tree)
  :use-module (alterator lookout globals)
  :use-module (alterator security wrap)

;;; attributes

  ;;common events
  :export (clicked double-clicked changed return-pressed selected toggled)

  ;;internal events
  :export (destroyed loaded)

  ;;general attributes
  :export (align pixmap focus widget-name max-width max-height tab-index)

  ;;text related attributes
  :export (alterability url echo text-wrap)

  ;;layout related attributes
  :export (margin spacing)

  ;;progress attributes
  :export (maximum minimum step)

  ;;tree/listbox/combobox attributes
  :export (current tree-items coordinates rows current-rows
           state-rows header current-text expanded
           row-remove rows-count expand-rows collapse-rows icon-rows)

  ;;other attributes
  :export (help state tooltip title)

  ;;menubar/wizardface attributes
  :export (actions current-action action-activity steps
           action-remove action-add current-step steps-clear
           step-text step-pixmap action-text action-pixmap)

  :export (enum-rows row)

  ;;; init attributes
  :export (parent type columns checked width height
           orientation colspan rowspan)


  ;;; meta-attributes
  :export (enumref value namelist name namereflist nameref)

  ;;; from std/meta-attributes
  :export (visibility activity current append-row
           row-item rows-clear text
           actions actions-clear action-add action-remove
           append-text tree-items tree-text)

  ;;; backward compatibility
  :export (checklist-rows checklist-active-rows active))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; attributes
(define-macro (define-attribute <name> . <rest>)
  `(define ,<name> (make-attribute ',<name> ,@<rest>)))

(define-macro (define-init-attribute <name>)
  `(define ,<name> (make-init-attribute ',<name>)))

(define-macro (define-meta-attribute <name> . <rest>)
  `(define ,<name> (make-meta-attribute ,<name> ,@<rest>)))

;;common events
(define-attribute clicked)
(define-attribute double-clicked)
(define-attribute changed)
(define-attribute return-pressed)
(define-attribute selected)
(define-attribute toggled)

;;internal events
(define-attribute destroyed)
(define-attribute loaded)

;;general attributes
(define-attribute align)
(define-attribute pixmap)
(define-attribute focus)
(define-attribute widget-name)
(define-attribute max-width)
(define-attribute max-height)
(define-attribute tab-index)

;;text related attributes
(define-attribute alterability)
(define-attribute url)
(define-attribute echo)
(define-attribute text-wrap)

;;layout related attributes
(define-attribute margin)
(define-attribute spacing)

;;progress attributes
(define-attribute maximum)
(define-attribute minimum)
(define-attribute step)

;;tree/listbox/combobox attributes
(define-attribute current)
(define-attribute tree-items)
(define-attribute coordinates)
(define-attribute rows)
(define-attribute current-rows)
(define-attribute state-rows)
(define-attribute header)
(define-attribute current-text)
(define-attribute expanded)
(define-attribute expand-rows)
(define-attribute collapse-rows)
(define-attribute icon-rows)

;;other attributes
(define-attribute help)
(define-attribute state)
(define-attribute tooltip)
(define-attribute title)

;;menubar/wizardface attributes
(define-attribute actions)
(define-attribute current-action)
(define-attribute action-activity 2)
(define-attribute steps)
(define-attribute action-remove)
(define-attribute action-add)
(define-attribute current-step)
(define-attribute steps-clear 0)
(define-attribute step-text 2)
(define-attribute step-pixmap 2)
(define-attribute action-text 2)
(define-attribute action-pixmap 2)

(define enum-rows (make-hidden-attribute 'enum-rows))
(define-attribute row)

;;; init attributes

(define-init-attribute parent)
(define-init-attribute type)
(define-init-attribute columns)
(define-init-attribute checked)
(define-init-attribute width)
(define-init-attribute height)
(define-init-attribute orientation)
(define-init-attribute colspan)
(define-init-attribute rowspan)

;;; meta-attributes


;;; enumref

(define (enumref-fill-row cmd row)
  (vector-map (lambda(x)
                (cons (woo-get-option cmd (car x))
                      (woo-get-option cmd (cdr x))))
              row))

(define-meta-attribute enumref
   ((meta-set self widget)
      (and-let* ((url-data (cond-car (value-of self)))
                 (url (if (pair? url-data) (car url-data) url-data))
                 (url-args (if (pair? url-data) (cdr url-data) '()))
                 (variants (apply woo-list url url-args))
                 (t (simple-get widget type)))
                (simple-set widget
                            (enum-rows
                             (map (lambda(x) (woo-get-option x 'name))
                                 variants)))
                (cond
                  ((member t '("listbox" "combobox"))
                   (let* ((r (or (simple-get widget row) '#((label . ""))))
                          (labels (map (lambda(x) (enumref-fill-row x r))
                                       variants)))
                     (simple-set widget (rows labels))
                     (simple-notify-set widget "rows" labels)))
                  ((string=? t "checklistbox")
                   (let ((labels (map (lambda(x) (vector (cons (woo-get-option x 'label) #f)))
                                      variants)))
                     (simple-set widget (rows labels))
                     (simple-notify-set widget "rows" labels)))))))

(define-meta-attribute value
   ((meta-set self widget)
    (let ((v (cond-car (value-of self)))
          (t (simple-get widget type))
          (l (or (simple-get widget enum-rows) '())))
      (cond
        ((member t '("listbox" "combobox"))
         (let* ((idx (or (string-list-index v l) -1)))
           (simple-set widget (current idx))
           (simple-notify-set widget "current" idx)))
        ((string=? t "checklistbox")
         (let* ((rlist (or (simple-get widget rows) '()))
                (vlist (string-cut (or v "") #\;))
                (slist (map (lambda(x) (->bool (member x vlist))) l)))
           (simple-notify-set widget "state-rows" slist)
           (simple-set widget (state-rows slist))
           (simple-set widget (rows (map (lambda (x y) (vector (cons (car (vector-ref x 0)) y)))
                                         rlist slist)))))
        (else
          (simple-set widget (value v))
          (simple-notify-set widget "value" v)))))
   ((meta-get self widget)
    (let ((t (simple-get widget type))
          (l (or (simple-get widget enum-rows) '())))
      (cond
        ((member t '("listbox" "combobox"))
         (let ((c (or (simple-get widget current) -1)))
           (if (>= c 0) (list-ref l c) "")))
        ((string=? t "checklistbox")
         (let ((slist (or (simple-get widget state-rows) '())))
           (string-join (filter-map (lambda(x y) (and x y)) slist l)
                        ";")))
        (else (simple-get widget value))))))


;;; DOM

(define namelist (make-hidden-attribute 'namelist))

(define-meta-attribute name
   ((meta-set self widget)
     (let* ((v (cond-car (value-of self)))
            (r (global 'document:root))
            (nlist (or (simple-get r namelist) '())))
     (simple-set r (namelist (acons v widget nlist)))
     (simple-set widget (name v))
     (simple-notify-set widget "name" v))))

(define namereflist (make-hidden-attribute 'namereflist))

(define-meta-attribute nameref
   ((meta-set self widget)
     (let* ((v (cond-car (value-of self)))
            (r (global 'document:root))
            (nlist (or (simple-get r namereflist) '())))
     (simple-set r (namereflist (acons v widget nlist)))
     (simple-set widget (nameref v))
     (simple-notify-set widget "nameref" v))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; from std/meta-helpers

;; listbox helpers

(define (safe-list-remove lst number)
  (and (container-check-range lst number)
       (list-remove lst number)))

(define (unify-item item)
       (if (pair? item)
           (cons (car item) (cdr item))
           (cons item "")))

(define (unify-row row)
  (vector-map unify-item row))

;; actions helpers

(define (valid-action? id)
  (lambda(elt) (not (eq? id (vector-ref elt 0)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; from std/meta-attributes

(define-meta-attribute visibility ((default-get self) #t))

(define-meta-attribute activity ((default-get self) #t))

(define-meta-attribute current ((default-get self) -1))

;;various list properties
(define-meta-attribute rows ((meta-set self widget)
       (let* ((columns (or (simple-get widget columns) 1))
              (elts (map (lambda(vec) (vector-fit-size vec columns ""))
                         (concatenate (value-of self))))
              (u-elts (map unify-row elts)))
         (simple-set widget (rows u-elts))
	    (simple-set widget (current -1))
         (simple-notify-set widget "rows" u-elts))))

(define-meta-attribute append-row ((meta-set self widget)
             (let* ((columns (or (simple-get widget columns) 1))
                    (elt (unify-row (vector-fit-size
                                     (car (value-of self))
                                      columns ""))))
               (simple-set widget (rows (append1 (or (simple-get widget rows) '()) elt)))
               (simple-notify-set widget "append-row" elt))))

(define-meta-attribute row-item 2
               ((meta-get self widget)
                (and-let* ((coords (car (value-of self)))
                           (column (safe-container-ref (simple-get widget rows)
                                                       (if (pair? coords)
                                                           (car coords)
                                                           coords))))
                          (safe-container-ref column (or (cond-cadr coords) 0) )))
               ((meta-set self widget)
                (let* ((value (value-of self))
                       (item (meta-get self widget))
                       (content (cadr value))
                       (txt (if (pair? content) (car content) content))
                       (pix (cond-cdr content)))
                  (and item (string? txt)
                       (begin (set-car! item txt)
                              (simple-notify-set
                               widget
                               "row-item-text"
                               (list txt (car value)))))
                  (and item (string? pix)
                       (begin (set-cdr! item pix)
                              (simple-notify-set
                               widget
                               "row-item-pixmap"
                               (list pix (car value))))))))

(define-meta-attribute rows-clear 0 ((meta-set self widget)
               (simple-set widget (current -1))
               (simple-set widget (rows '()))
               (simple-notify-set widget "rows-clear" "")))

(define-meta-attribute rows-count ((meta-get self widget)
		(length (or (simple-get widget rows) '()) )))

(define-meta-attribute row-remove ((remove-one self widget lst number)
         (and (list? lst)
              (begin
                (simple-set widget (rows lst))
                (simple-notify-set widget "remove-row" number))))
        ((meta-set self widget)
         (define-operation remove-one)
         (let ((number (cond-car (value-of self))))
           (simple-set widget (current -1))
           (and (number? number)
                (remove-one self widget
                            (safe-list-remove (simple-get widget rows) number)
                            number)))))

(define-meta-attribute text ((meta-get self widget)
       (or (simple-get widget text);for widgets with real text property
           (cond-car (meta-get (row-item (simple-get widget current))
                               widget))
           "")))

;;menubar attributes
(define-meta-attribute actions ((meta-set self widget)
          (widget actions
                  (map (lambda(item)
                         (vector-fit-size item 3 ""))
                       (car (value-of self))))))

(define-meta-attribute actions-clear 0 ((meta-set self widget)
                  (simple-set widget (actions '() ))
                  (simple-notify-set widget "actions-clear" "")))

(define-meta-attribute action-add ((meta-set self widget)
             (let ((elt (vector-fit-size (car (value-of self)) 3 "")))
               (simple-set widget (actions (append1 (or (simple-get widget actions) '()) elt)))
               (simple-notify-set widget "action-add" elt))))

(define-meta-attribute action-remove ((meta-set self widget)
                (let ((id (car (value-of self))))
                  (simple-set widget
                              (actions
                               (filter (valid-action? id)
                                       (simple-get widget actions))))
                  (simple-notify-set widget "action-remove" id))))

;;textbox attributes
(define-meta-attribute append-text ((meta-set self widget)
              (let* ((prev-text (simple-get widget text))
                     (app-text (string-join (value-of self) ""))
                     (new-text (string-append prev-text
                                              (string #\newline)
                                              app-text)))
                (simple-set widget (text new-text))
                (simple-notify-set widget "text-append" app-text))))

;;tree attributes
(define-meta-attribute tree-items ((meta-set self widget)
             (let* ((columns (or (simple-get widget columns) 1))
                    (elts (tree-map (lambda(vec) (vector-fit-size vec columns ""))
                                    (concatenate (value-of self))))
                    (u-elts (tree-map unify-row elts))
                    (coords (make-coordinates u-elts)))
               (simple-set widget (tree-items u-elts))
               (simple-set widget (coordinates coords))
               (simple-set widget (current #f))
               (simple-notify-set widget "tree-items" u-elts)
               (simple-notify-set widget "coordinates" coords))))

(define-meta-attribute tree-text ((meta-get self widget)
            (define (extract-text item)
              (car (vector-ref item 0)))
            (let ((elt (tree-path (simple-get widget tree-items)
                                  (simple-get widget current))))
              (map extract-text elt))))

;;; backward compatibility

(define active current)

(define-meta-attribute checklist-rows
    ((meta-set self widget)
     (let* ((v (cond-car (value-of self)))
            (vlist (map (lambda(x) (if (pair? x) x (cons x #t))) v))
            (slist (map cdr vlist)))
       (simple-set widget (rows vlist))
       (simple-set widget (state-rows slist))
       (simple-notify-set widget "rows" vlist)
       (simple-notify-set widget "state-rows" slist)))
    ((meta-get self widget)
     (let ((vlist (simple-get widget rows))
           (slist (simple-get widget state-rows)))
       (map (lambda(x y) (cons (car x) y))
            vlist slist))))

(define-meta-attribute checklist-active-rows
    ((meta-get self widget)
     (let ((vlist (simple-get widget rows))
           (slist (simple-get widget state-rows)))
       (filter-map (lambda(x y) (and y (car x)))
            vlist slist))))
