(define-module (alterator metalterator)
  :use-module (srfi srfi-1)
  :use-module (alterator algo)
  :use-module (alterator plist)
  :use-module (alterator woo)
  :use-module (alterator metalterator match)
  :export (current-debug-level set-debug-level! print-debug
           sure-string plist-delete first-value sure-value
           sure-all-values without-values append-values
           objects->path meta-prefix meta-root-prefix
           meta-name plist-set! plist-update plist-prefix-val
           object-exists? woo-case meta/throw format-error
           catch/report meta))

(define metalterator-debug-level 0)

(define (current-debug-level)
  metalterator-debug-level)

(define (set-debug-level! level)
  (set! metalterator-debug-level level))

(define-macro (print-debug level message . args)
  `(if (>= (current-debug-level) ,level)
       (format (current-error-port)
               ,(string-append "[meta] " message "~%")
               ,@args)))

(define (sure-string obj)
  (cond
   ((string? obj) obj)
   (else
    (object->string obj))))

(define (plist-delete del-name plist)
  (plist-fold (lambda (name val result)
                (if (not (equal? name del-name))
                    (append result (list name val))
                    result))
              '()
              plist))

(define (first-value entry)
  (if (pair? entry)
      (if (list? (cdr entry))
          (cadr entry)
          (cdr entry))
      entry))

(define (sure-value options name default)
  (cond
   ((plistq name options) =>
    (lambda (entry)
      (first-value entry)))
   (else default)))

(define (sure-all-values options name default)
  (cond
   ((plistq name options) =>
    (lambda (entry)
      (cdr entry)))
   (else default)))

(define (without-values options . names)
  (fold (lambda (name options)
          (plist-delete name options))
        options
        names))

(define (append-values objects options . names)
  (fold (lambda (name objects)
          (let ((entry (plistq name options)))
            (if entry
                (append objects (if (list? (cdr entry))
                                    (cdr entry)
                                    (list (cdr entry))))
                objects)))
        objects
        names))

(define (objects->path objects)
  (if (null? objects)
      ""
      (fold (lambda (obj path)
              (string-append path
                             "/"
                             (sure-string obj)))
            ""
            objects)))

(define (meta-root-prefix basename val)
  (let ((basename (sure-string basename)))
    (if (and (not (string-null? val))
             (eq? #\/ (string-ref basename 0)))
        (string-append basename val)
        (string-append "/etc/metalterator/"
                       basename "/" basename
                       val))))

(define (meta-prefix basename val)
  (string-append "meta" (meta-root-prefix basename val)))

(define (meta-name basename objects)
  (print-debug 5 "Add meta-prefix: ~s" objects)
  (meta-prefix basename (objects->path objects)))

(define (plist-set! name val plist)
  (append (plist-delete name plist)
          (cons name
                (if (list? val)
                    val
                    (list val)))))

(define (plist-update name val plist)
  (cond
   ((plistq name plist) =>
    (lambda (entry)
      (plist-set! (car entry)
                  val
                  plist)))
   (else plist)))

(define (plist-prefix-val name prefix-proc plist)
  (cond
   ((plistq name plist) =>
    (lambda (entry)
      (plist-set! (car entry)
                  (if (list? (cdr entry))
                      (map (lambda (val)
                             (prefix-proc val))
                           (cdr entry))
                      (prefix-proc (cdr entry)))
                  plist)))
   (else plist)))

(define (object-exists? basename objects)
  (woo-catch
   (lambda ()
     (woo-read (meta-name basename objects)))
   (lambda (arg)
     (if (string-prefix? "no-such-object" arg)
       #f
       (woo-error arg)))))

(define-macro (woo-case woo-cmd . cases)
  `(let ((val-table '()))
     (or ,@(map
            (lambda (cas)
              (if (eq? 'else (car cas))
                  `(begin ,@(cdr cas))
                  `(and (equal? (sure-string (sure-value ,(cadr woo-cmd)
                                                         'action
                                                         '*))
                                ,(sure-string (caar cas)))
                        (print-debug 5 "Match objects: ~s | ~s"
                                     (quote ,(cadar cas))
                                     (map sure-symbol ,(car woo-cmd)))
                        (let-match ((,(cadar cas)
                                     (map sure-symbol ,(car woo-cmd))))
                          ,@(if (not (null? (cddar cas)))
                                `((let (,@(plist-map
                                           (lambda (name val)
                                             (cond
                                              ((singular-parameter? val)
                                               `(,val (sure-value ,(cadr woo-cmd)
                                                                  ,name
                                                                  #f)))
                                              ((interval-parameter? val)
                                               `(,val (sure-all-values ,(cadr woo-cmd)
                                                                       ,name
                                                                       #f)))
                                              (else '())))
                                           (caddar cas)))
                                    ,@(cdr cas)))
                                (cdr cas))))))
                cases))))

(define-macro (meta/throw action basename objects options)
  `(let* ((answer
           (catch 'woo-error
            (lambda ()
              (apply ,@(case action
                         ((read) `(woo-read))
                         ((write) `(woo-write))
                         ((list) `(woo-list))
                         ((new) `(woo-new))
                         ((delete) `(woo-delete))
                         (else `(woo ,(symbol->string action))))
                     (append (list (meta-name ,basename ,objects))
                             ,(case action
                                ((link)
                                 `(list 'name
                                        (meta-root-prefix (car ,options)
                                                          (objects->path (cdr ,options)))))
                                (else options)))))
            (lambda (key . args)
              (throw 'woo-meta-error
                     (string-append (meta-name ,basename ,objects)
                                    " "
                                    ,(sure-string action))
                     (car args)
                     '()
                     #f))))
          (answer
           (map (lambda (obj)
                  (cond
                   ((list? obj)
                    (print-debug 5 "Normalize answer object: ~s" obj)
                    (let* ((obj (cdr obj))
                           (options (plist->alist obj))
                           (name (assoc-ref options 'name))
                           (options (alist-delete 'name options)))
                      (append (if name (list name) '())
                              (alist->plist options))))
                   (else obj)))
                answer)))
     (cond
      ((null? answer)
       (print-debug 2 "Meta-answer: ~s" '())
       '())
      ,@(if (not (eq? action 'list))
            `(((null? (cdr answer))
               (print-debug 2 "Meta-answer: ~s" (car answer))
               (car answer)))
            '())
      (else
       (print-debug 2 "Meta-answer: ~s" answer)
       answer))))

(define (format-error subr message args rest)
  (apply format #f
         (string-concatenate
          (list
           (if subr "(~s) " "")
           message
           (if rest " (~s)" "")))
         (append
          (if subr (list subr) '())
          args
          (if rest (list rest) '()))))

(define (is-error-args? args)
  (and (eq? (length args) 4)
       (string? (cadr args))
       (list? (caddr args))
       (or (list? (cadddr args))
           (not (cadddr args)))))

(define (catch/report thunk)
  (catch #t
    thunk
    (lambda (key . args)
      (case key
        ((woo-error)
         (print-debug 5 "Answer with error message: ~s" args)
         (append '(error) args))
        (else
         (cond
          ((is-error-args? args)
           (print-debug 5 "Answer with error message: ~s" (apply format-error args))
           `(error ,(apply format-error args)))
          (else
           (print-debug 5 "Answer with error message: ~s ~s" key args)
           `(error ,(format #f "~s ~s" key args)))))))))

(define-macro (meta action basename objects options)
  `(catch/report
    (lambda ()
      (meta/throw ,action ,basename ,objects ,options))))
