(define-module (alterator metalterator match)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-2)
  :export (make-token concatenate-token lookup lookup-quote multiply
           match-token singular-parameter? interval-parameter?
           pattern->anchors match-anchors match-each
           match-any assoc-merge let-match))

;;; Map-in-order macro.
(define-macro (map proc . lists)
  (let ((arglist (fold (lambda (a b)
                         (cons (gensym) b))
                       '() lists)))
    `(fold (lambda (,@arglist last)
             (append last (list (,proc ,@arglist))))
           '()
           ,@lists)))

;;; Filter-map-in-order macro.
(define-macro (filter-map proc . lists)
  (let ((arglist (fold (lambda (a b)
                         (cons (gensym) b))
                       '() lists)))
    `(fold (lambda (,@arglist last)
             (append last
                     ((lambda (a)
                        (or (and a (list a)) '()))
                      (,proc ,@arglist))))
           '()
           ,@lists)))

;;; Append-map-in-order macro.
(define-macro (append-map proc . lists)
  (let ((arglist (fold (lambda (a b)
                         (cons (gensym) b))
                       '() lists)))
    `(fold (lambda (,@arglist last)
             (append last (,proc ,@arglist)))
           '()
           ,@lists)))

;;; Merges values from association lists.
(define (assoc-merge . alists)
  (let list-loop ((alist (car alists))
                  (tail (cdr alists)))
    (cond
     ((null? tail)
      alist)
     (else
      (list-loop (let loop ((alist alist)
                            (tail (car tail)))
                   (cond
                    ((null? tail)
                     alist)
                   (else
                    (loop (alist-cons (caar tail)
                                      (cdar tail)
                                      (alist-delete (caar tail) alist))
                          (cdr tail)))))
                 (cdr tail))))))


;;; Retrieves all pairs with the specified name from an
;;; association list.
(define (assoc-all key alist)
  (let ((pairs (filter (lambda (pair)
                         (equal? key (car pair)))
                       alist)))
    (and (not (null? pairs))
         pairs)))

;;; Predicate for singular pattern parameter.
(define (singular-parameter? val)
  (and (symbol? val)
       (let ((txtval (symbol->string val)))
         (and (eq? #\@ (string-ref txtval 0))
              (or (< (string-length txtval) 2)
                  (not (eq? #\@ (string-ref txtval 1))))))))

;;; Predicate for interval pattern parameter.
(define (interval-parameter? val)
  (and (symbol? val)
       (let ((txtval (symbol->string val)))
         (and (eq? #\@ (string-ref txtval 0))
              (> (string-length txtval) 1)
              (eq? #\@ (string-ref txtval 1))))))

;;; Predicate for optional singular pattern parameter.
(define (opt-singular-parameter? val)
  (and (symbol? val)
       (let ((txtval (symbol->string val)))
         (and (>= (string-length txtval) 2)
              (eq? #\* (string-ref txtval 0))
              (eq? #\@ (string-ref txtval 1))
              (or (< (string-length txtval) 3)
                  (not (eq? #\@ (string-ref txtval 2))))))))

;;; Predicate for optional interval pattern parameter.
(define (opt-interval-parameter? val)
  (and (symbol? val)
       (let ((txtval (symbol->string val)))
         (and (>= (string-length txtval) 3)
              (eq? #\* (string-ref txtval 0))
              (eq? #\@ (string-ref txtval 1))
              (eq? #\@ (string-ref txtval 2))
              (or (< (string-length txtval) 4)
                  (not (eq? #\@ (string-ref txtval 3))))))))

;;; Predicate for singular pattern.
(define (singular-pattern? val)  
  (and (symbol? val)
       (let ((txtval (symbol->string val)))
         (and (eq? 1 (string-length txtval))
              (eq? #\? (string-ref txtval 0))))))

;;; Predicate for optional interval pattern.
(define (optional-pattern? val)
  (and (symbol? val)
       (let ((txtval (symbol->string val)))
         (and (eq? 1 (string-length txtval))
              (eq? #\* (string-ref txtval 0))))))

;;; Predicate for parameter symbols.
(define (parameter-symbol? val)
  (or (singular-parameter? val)
      (interval-parameter? val)
      (opt-singular-parameter? val)
      (opt-interval-parameter? val)))

;;; Predicate for all pattern symbols.
(define (pattern-symbol? val)
  (or (parameter-symbol? val)
      (singular-pattern? val)
      (optional-pattern? val)))

;;; Predicate for all optional parameters.
(define (optional-parameter? val)
  (or (opt-singular-parameter? val)
      (opt-interval-parameter? val)))

;;; Transforms interval parameter pattern into a
;;; singular pattern.
(define (parameter->singular sym)
  (cond
   ((opt-singular-parameter? sym)
    (string->symbol (substring (symbol->string sym) 1)))
   ((interval-parameter? sym)
    (string->symbol (substring (symbol->string sym) 1)))
   ((opt-interval-parameter? sym)
    (string->symbol (substring (symbol->string sym) 2)))
   (else sym)))

;;; Normalizes a token pattern into a set of anchors.
(define (pattern->anchors pattern)
  (let loop ((anchors '())
             (anchor '())
             (anchored? #t)
             (pattern pattern))
    (if (null? pattern)
        (or (and (not (null? anchor))
                 (append anchors (list anchor)))
            anchors)
        (let ((sym (car pattern)))
          (cond
           ((not (pattern-symbol? sym))
            (loop anchors
                  (append anchor (list sym))
                  #t
                  (cdr pattern)))
           ((optional-pattern? sym)
            (if anchored?
                (loop (or (and (not (null? anchor))
                               (append anchors (list anchor)))
                          anchors)
                      (list sym)
                      #f
                      (cdr pattern))
                (loop anchors anchor #f (cdr pattern))))
           ((or (interval-parameter? sym)
                (opt-interval-parameter? sym))
            (if anchored?
                (loop (or (and (not (null? anchor))
                               (append anchors (list anchor)))
                          anchors)
                      (list sym)
                      #f
                      (cdr pattern))
                (loop anchors
                      (append anchor (list (parameter->singular sym)))
                      #f
                      (cdr pattern))))
           (else (loop anchors
                       (append anchor (list sym))
                       anchored?
                       (cdr pattern))))))))

;;; Prototypes a pattern.
(define (prototype-pattern pat)
  (let loop ((head '())
             (tail pat))
    (cond
     ((null? tail)
      (fold (lambda (a b)
              (append b (if (and (eq? a '*)
                                 (not (null? b))
                                 (eq? (car (reverse b)) '*))
                            '()
                            (list a))))
            '()
            head))
     (else
      (loop (append head
                    ((lambda (val)
                       (cond
                        ((singular-parameter? val) (list '?))
                        ((interval-parameter? val) (list '? '*))
                        ((singular-pattern? val) (list '?))
                        ((optional-pattern? val) (list '*))
                        ((opt-singular-parameter? val) (list '*))
                        ((opt-interval-parameter? val) (list '*))
                        (else (list val))))
                     (car tail)))
            (cdr tail))))))

;;; Determines the equivalence of two symbol-patterns.
(define (pattern-eqv? pat1 pat2)
  (equal? (prototype-pattern pat1)
          (prototype-pattern pat2)))

;;; Matches an anchor against a list of values.
(define (match-anchor anchor vals)
  ;(display anchor)(display vals)(newline)
  (let loop ((val-table '())
             (anchor anchor)
             (vals vals))
    (cond
     ((null? anchor) (cons val-table vals))
     ((null? vals) (and (or (optional-pattern? (car anchor))
                            (optional-parameter? (car anchor)))
                        (null? (cdr anchor))
                        (cons val-table vals)))
     (else
      (let ((sym (car anchor))
            (val (car vals)))
        (cond
         ((or (singular-parameter? sym)
              (opt-singular-parameter? sym))
          (loop (append (list (cons (parameter->singular sym) val))
                        val-table)
                (cdr anchor)
                (cdr vals)))
         ((or (interval-parameter? sym)
              (opt-interval-parameter? sym))
          (loop (append (list (cons sym (list val))) val-table)
                (cdr anchor)
                (cdr vals)))
         ((singular-pattern? sym)
          (loop val-table (cdr anchor) (cdr vals)))
         ((optional-pattern? sym)
          (loop val-table (cdr anchor) vals))
         ((equal? sym val)
          (loop val-table (cdr anchor) (cdr vals)))
         (else #f)))))))

;;; Predicate for interval symbol at the head of a
;;; value table.
(define (interval-head? val-table)
  (and (not (null? val-table))
       (or (interval-parameter? (caar val-table))
           (opt-interval-parameter? (caar val-table)))))

;;; Transforms head symbol of a value table into a
;;; singular symbol.
(define (normalize-head val-table)
  (if (not (null? val-table))
      (append (list (cons (parameter->singular (caar val-table))
                          (cdar val-table)))
              (cdr val-table))
      val-table))

;;; Removes one optional parameter from the anchor.
(define (reduce-anchor anchor)
  (let loop ((head '())
             (tail anchor)
             (removed? #f))
    (cond
     ((null? tail)
      (and removed? (if (not (null? head))
                        head
                        '(*))))
     ((optional-parameter? (car tail))
      (loop (append head (cdr tail)) '() #t))
     (else
      (loop (append head (list (car tail))) (cdr tail) #f)))))

;;; Matches a non-null list of token-pattern anchors against a token
;;; and extracts a set of values.
(define (match-anchors-non-null anchors tok)
  (let loop ((val-table '())
             (ranchors (reverse (map reverse anchors)))
             (rtail (reverse tok))
             (opt-pat? #f))
    (cond
     ((and (null? ranchors) (null? rtail))
      (normalize-head val-table))
     ((null? rtail)
      (and (or (optional-pattern? (caar ranchors))
               (optional-parameter? (caar ranchors)))
           (null? (cdar ranchors))
           (null? (cdr ranchors))))
     ((and (not (null? ranchors))
           (match-anchor (car ranchors) rtail)) =>
      (lambda (match)
        (loop (append (car match) (normalize-head val-table))
              (cdr ranchors)
              (cdr match)
              (optional-pattern? (car (reverse (car ranchors)))))))
     ((interval-head? val-table)
      (loop (append (list (cons (caar val-table)
                                (append (list (car rtail))
                                        (cdar val-table))))
                    (cdr val-table))
            ranchors
            (cdr rtail) #f))
     (opt-pat?
      (loop val-table ranchors (cdr rtail) #t))
     (else #f))))

;;; Matches a list of token-pattern anchors against a token
;;; and extracts a set of values.
(define (match-anchors anchors tok)
  (cond
   ((or (and (null? anchors)
             (null? tok))
          (and (not (null? anchors))
               (not (null? (car anchors)))
               (optional-pattern? (caar anchors))
               (null? (cdar anchors))
               (null? (cdr anchors))))
    '())
   ((null? tok) #f)
   (else
    (match-anchors-non-null anchors tok))))

;;; Value table lookup procedure
(define-macro (lookup val-name)
  `(cond
    ((assoc (quote ,(parameter->singular val-name))
            val-table) =>
     (lambda (aval)
       (cdr aval)))
    (else #f)))

;;; Value table lookup procedure with quotation of
;;; undefined values.
(define-macro (lookup-quote val-name)
  `(or (lookup ,val-name) (quote ,val-name)))

;;; Makes a singular list from a singular argument
;;; if it is not specially quoted.
(define (concatenate-token tok)
  (append-map (lambda (a)
                (if (list? a)
                    (case (car a)
                      ((each)
                       (list a))
                      ((any)
                       (list a))
                      (else a))
                    (list a)))
              tok))

;;; Quotation macro of type `each'.
(define-macro (match-each . args)
  `(append (list 'each)
           (concatenate-token (list ,@args))))

;;; Quotation macro of type `any'.
(define-macro (match-any . args)
  `(append (list 'any)
           (concatenate-token (list ,@args))))

;;; Token multiplication.
(define (multiply tok)
  (let loop ((head '((())))
             (tail tok))
    (cond
     ((null? tail)
      head)
     ((list? (car tail))
      (loop (case (caar tail)
              ((each)
               (append-map (lambda (val)
                             (map (lambda (token-case)
                                    (map (lambda (tok)
                                           (append tok (list val)))
                                         token-case))
                                  head))
                           (cdar tail)))
              ((any)
               (map (lambda (token-case)
                      (append-map (lambda (tok)
                                    (map (lambda (val)
                                           (append tok (list val)))
                                         (cdar tail)))
                                  token-case))
                    head))
              (else
               (map (lambda (token-case)
                      (map (lambda (tok)
                             (append tok (list (car tail))))
                           token-case))
                    head)))
            (cdr tail)))
     (else
      (loop (map (lambda (token-case)
                      (map (lambda (tok)
                             (append tok (list (car tail))))
                           token-case))
                    head)
            (cdr tail))))))

;;; Expands a token.
(define-macro (make-token pattern)
  (if (null? pattern)
      `(multiply '())
      `(multiply (concatenate-token
                  ,(append
                    `(list ,(car pattern))
                    (map (lambda (sym)
                           (if (and (symbol? sym)
                                    (eq? ': sym))                                    
                               `(quote ,sym)
                               sym))
                         (cdr pattern)))))))

;;; Translates pattern symbols into corresponging lookups.
(define (pattern->lookups pattern)
  (map (lambda (sym)
         (cond
          ((list? sym)
           (pattern->lookups sym))
          ((parameter-symbol? sym)
           `(lookup-quote ,sym))
          (else sym)))
       pattern))

;;; Quotes pattern-symbols.
(define (quote-pattern pattern)
  (map (lambda (sym)
         (cond
          ((and (pattern-symbol? sym)
                (not (parameter-symbol? sym)))
           `(quote ,sym))
          (else sym)))
       pattern))

;;; Token pattern match macro.
(define-macro (match-token pattern val)
  `(fold
    (lambda (a b)
      (and a b (assoc-merge a b))) '()
      (map (lambda (tok-case)
             (let loop ((tok-case tok-case))
               (and (not (null? tok-case))
                    (or (and (not (null? tok-case))
                             (match-anchors
                              (pattern->anchors (car tok-case))
                              ,val))
                        (loop (cdr tok-case))))))
           (make-token ,(quote-pattern (pattern->lookups pattern))))))

;;; Translates pattern into the variable lookups.
(define (pattern->vars pattern)
  (delete-duplicates
   (append-map (lambda (sym)
                 (cond
                  ((list? sym)
                   (pattern->vars sym))
                  ((pattern-symbol? sym)
                   (list (cons sym `((lookup ,sym)))))
                  (else '())))
               pattern)))

;;; Translates pattern-bindings into match-bindings.
(define (pattern->match-bindings pattern-bindings)
  (map (lambda (pattern-binding)
         `(val-table
           (cond
            ((match-token ,(car pattern-binding)
                          ,(cadr pattern-binding)) =>
             (lambda (val-table-b)
               (assoc-merge val-table val-table-b)))
            (else #f))))
       pattern-bindings))

;;; Translates pattern-bindings into token-variable bindings.
(define (pattern->token-bindings pattern-bindings)
  (let loop ((i 1)
             (token-bindings '())
             (pattern-bindings pattern-bindings))
    (if (null? pattern-bindings)
        token-bindings
        (loop (+ i 1)
              (append token-bindings
                      `((,(string->symbol
                           (string-append
                            "@" (number->string i)))
                         (multiply ,(cadar pattern-bindings)))))
              (cdr pattern-bindings)))))

;;; Pattern-matching environment macro.
(define-macro (let-match pattern-bindings . tail)
  `(and-let* ,(pattern->match-bindings pattern-bindings)
     (let ,(append
            (delete-duplicates
             (append-map pattern->vars (map car pattern-bindings)))
            (pattern->token-bindings pattern-bindings))
       ,@tail)))
