;description: general support for scheme objects

;there are two level of objects
; object - main component of object-simple
; callable - simplification for alterator over object
(define-module (alterator object)
  :use-module (srfi srfi-1)
  :use-module (alterator algo)
  :export (object-dispatcher
           make-object
           object?
           select-operation
           run-operation
           
           set-object-slots!
           get-object-slots
           
           object
           join
           
           operation
           define-operation
           
           instance-of
           is-a?))

(define (set-object-slots! obj slots)
  (and (procedure? obj) (set-procedure-property! obj 'object-slots slots))
  obj)

(define (get-object-slots obj)
  (and (procedure? obj) (procedure-property obj 'object-slots)))

;determine object - only objects in our model has object-slots property
(define (object? obj)
  (and (procedure? obj) (procedure-property obj 'object-slots)))

(define (object-dispatcher op table)
  (and (pair? table)
       (cond
	 ((assq op table) => cdr)
	 (else #f))))

(define (make-object call-proc table)
  (set-object-slots! 
    (lambda args
      (let ((proc (call-proc)))
	(if (procedure? proc)
	  (apply proc args)
	  proc)))
    table))

(define (select-operation obj op)
  (and (object? obj) (object-dispatcher op (get-object-slots obj))))

(define (run-operation obj op default . args)
  (let ((result (select-operation obj op)))
    (cond
     ((procedure? result) (apply result obj args))
     ((procedure? default) (apply default obj args))
     (else default))))


;create object
(define-macro (object <proc> . <methods>)
  (let ((slots (map (lambda (method-slot)
                      (let ((definition (car method-slot))
                            (expressions (cdr method-slot)))
                        `(cons ',(car definition)
                               (lambda ,(cdr definition) ,@expressions))))
                    <methods>)))
    `(make-object (lambda() ,<proc>) (list ,@slots))))

;join objects together, some kind of inheritance
(define (join object . rest-objects)
  (set-object-slots!
   (lambda args (if (procedure? object) (apply object args) object))
   (apply append (or (get-object-slots object) '())
          (filter-map get-object-slots rest-objects))))

;create external operation for calling object
(define-macro (operation <op> . <default>)
  (let ((object (gensym))
        (default (if (= (length <default>) 1) (car <default>) (lambda(obj . args) (error "unknown operation for object:" <op>))))
        (args (gensym)))
    `(lambda (,object . ,args)
       (apply run-operation ,object ',<op> ,default ,args))))

;define operation with same name
(define-macro (define-operation <op> . <default>)
     `(define ,<op> (operation ,<op> ,@<default>)))

;check are some object of some class
(define-macro (is-a? <object> <classname>)
    `(eq? (instance-of ,<object>) ',<classname>))

(define-operation instance-of (lambda(object)
                                (cond
                                 ((promise? object) '<promise>)
                                 ((boolean? object) '<boolean>)
                                 ((vector? object) '<vector>)
                                 ((port? object) '<port>)
                                 ((number? object) '<number>)
                                 ((string? object) '<string>)
                                 ((symbol? object) '<symbol>)
                                 ((char? object) '<char>)
                                 ((pair? object) '<pair>)
                                 ((procedure? object) '<procedure>)
                                 (else '<unknown>))))



