;description: working with pipes and subprocesses

(define-module (alterator pipe)
	       :use-module  (alterator algo)
	       :export (create-process
			 stop-process
			 with-ignored-sigpipe

			 process-pid
			 process-read-port
			 process-write-port
			 process-terminal-port))

(load-extension "libguile-pipe.so" "scm_init_pipe")

;we have made special controlling terminal for all childs
;when master process exiting ... all childs will receive SIGHUP signal

;extended version of ice-9 popen
(define (read-only? x) (eq? x 'read-only))
(define (write-only? x) (eq? x 'write-only))
(define (read-write? x) (eq? x 'read-write))

;create pipe and made nonbuffered write side
(define (create-pipe)
  (let ((p (pipe)))
    (setvbuf (cdr p) _IONBF)
    p))

;return (pid,read-port,write-port)
;TODO: port other code from ice-9 popen
; e.g. guardian's, hooks, etc.
(define (create-process interactive prog . args)
  (catch 'system-error
    (thunk
     (let ((read-pipe (and interactive (create-pipe)))
           (write-pipe (and interactive (create-pipe)))
           (terminal-pipe (make-pty)))
       (let ((pid (primitive-fork)))
         (cond 
          ((zero? pid)
	   ;;child work
	   (catch #t ;;exit if any problem in high level code will be detected
		  (lambda()
		    ;;setup controling terminal
		    (if terminal-pipe
		      (set-controlling-terminal (cdr terminal-pipe))
		      (format (current-error-port) "warning: unable to setup controlling terminal for backend"))

		    (if interactive
		      (begin (close-port (car read-pipe))
			     (dup2 (fileno (cdr read-pipe)) 1)))
		    (if interactive
		      (begin (close-port (cdr write-pipe))
			     (dup2 (fileno (car write-pipe)) 0)))
		    ;;ports cleanup
		    (port-for-each (lambda (pt-entry)
				     (catch/ignore #t
						   (lambda()
						     (let ((pt-fileno (fileno pt-entry)))
						       (or (= pt-fileno 0)
							   (= pt-fileno 1)
							   (= pt-fileno 2)
							   (close-fdes pt-fileno)))))))
		    ;;start subprocess
		    (apply execlp prog prog args))
		  (lambda (key . args)
		    (format (current-error-port) "fatal: backend startup:~S:~S~%" key args)
		    (force-output (current-error-port))))
	   (primitive-exit 1));;execlp should replace our process with other
	  (else
	    ;;parent work
	    (vector
	          pid
		  (and interactive
		       (begin (close-port (cdr read-pipe))
			      (car read-pipe)))
		  (and interactive
		       (begin (close-port (car write-pipe))
			      (cdr write-pipe)))
		  (and terminal-pipe
		       (begin (close-port (cdr terminal-pipe))
			      (car terminal-pipe)))))))))
    (lambda args
      (throw 'misc-error
             (cadr args)
             (format #f "~A: ~A" prog (strerror (system-error-errno args)))
	     #f))))

(define (close-port/safe port)
 (and (port? port)
      (not (port-closed? port))
      (close-port port)))

(define (stop-process action p)
  (let ((pid (process-pid p))
        (read-port (process-read-port p))
        (write-port (process-write-port p))
        (terminal-port (process-terminal-port p)))
  (close-port/safe write-port)
  (begin-1
   (catch #t
     (thunk
       (if (eq? action 'terminate) (kill pid SIGTERM))
       (status:exit-val (cdr (waitpid pid))))
     (lambda (key . args) 0))
   (close-port/safe read-port)
   (close-port/safe terminal-port))))

(define (with-ignored-sigpipe main-thunk)
  (let ((saved-action #f))
    (dynamic-wind
      (thunk (set! saved-action (sigaction SIGPIPE SIG_IGN)))
      main-thunk
      (thunk (sigaction SIGPIPE (car saved-action))))))

;all functions works with pair of ports: (for-read,for-write)
(define (process-pid p) (vector-ref p 0))
(define (process-read-port p) (vector-ref p 1))
(define (process-write-port p) (vector-ref p 2))
(define (process-terminal-port p) (vector-ref p 3))
