(define-module (alterator presentation container)

  :use-module (srfi srfi-1)
  :use-module (srfi srfi-13)

  :use-module (alterator object)
  :use-module (alterator algo)
  :use-module (alterator plist)
  :use-module (alterator str)
  :use-module (alterator context)

  :use-module (alterator session common)

  :use-module (alterator presentation args)
  :use-module (alterator presentation common)
  :use-module (alterator presentation events)

  :export ( make-container
            meta-get
            simple-get
	    default-get
	   
	    meta-set
	    simple-set

            simple-notify
	    simple-notify-set))

;;meta-* and fallbacks to simple-*
(define-operation simple-get)
(define-operation meta-get (lambda (attr widget)
                             (simple-get widget attr)))

;half-extraction for single value attributes
;TODO: possible we need move this to value-of method
(define (simplify-value x)
  (if (and (pair? x) (= (length x) 1)) (car x) x))


(define-operation run-callback)
(define-operation simple-set)
(define-operation simple-notify)
(define-operation simple-notify-set)
(define-operation meta-set (lambda(attr widget)
                             (and (simple-set widget attr)
                                  (simple-notify-set widget
                                                     (name-of attr)
                                                     (simplify-value (value-of attr))))))
(define-operation default-get #f)

(define (extract-proxy x attr)
  (define-operation proxy-get)
  (let ((x (cdr x)))
    ;;mix saved proxy attribute with existing one to give
    ;;access proxy-get to value of half defined attribute
    (proxy-get (join attr x))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;; flat args (need for the old qt browser protocol)

;flatten mix of vector and list
(define (container-flatten lst)
  (fold (lambda(x y)
          (cond
           ((list? x) (append y (container-flatten x)))
           ((vector? x) (append y (container-flatten (vector->list x))))
           (else (append1 y x))))
        '()
        lst))

(define (pair->string x)
  (if (pair? x)
      (string-append (sure-string (car x))
                     ";"
                     (sure-string (cdr x)))
      (sure-string x)))

(define (flat-value x)
  (if (is-a? x <container>)
      x ;;container will be converted to id in lower level
      (string-join (map pair->string (container-flatten (list x))) ";")))

(define (flat-args args)
  (reverse (plist-fold (lambda(x y args)
                         (cons* (flat-value y) x args))
                       '()
                       args)))

;;;;;;;;;;;;;;;;;;;;;;;;;;

;extended version of the alist-set with ability to compose callbacks
(define (alist-compose name value alist)
  (define (compositor name value1 value2)
    (if (callback? value1)
        (cons name (make-callback (value1) (value2)))
        (cons name value2)))
  
  (let loop ((current alist)
             (result '()))
    (cond
     ((null? current) (append1 result (cons name value)))
     ((eq? name (caar current)) (append result
                                        (list (compositor name  (cdar current) value))
                                        (cdr current)))
     (else (loop (cdr current) (append1 result (car current)))))))


(define (make-attributes)
  (let ((attrs '())
        (proxies '())
        (inits '())
        (hiddens '()))
    (object
     #f
     ((proxy-set self attr)
      (set! proxies (alist-set (name-of attr)
                               attr
                               proxies)))
     ((simple-get self attr)
      (let ((name (name-of attr)))
        (cond ((assq name proxies) => (lambda(x) (extract-proxy x attr)))
              ((or (assq name attrs)
                   (assq name inits)
                   (assq name hiddens)) => cdr)
              (else (default-get attr)))))
     ((simple-set self attr)
      (let ((name (name-of attr))
            (value (simplify-value (value-of attr))))
        (cond
         ((assq name proxies)
          => (lambda(item)
               (define-operation proxy-set)
               (proxy-set (cdr item) value) 
	       #f));;no set notification
         ((initial? attr) ;;initial save to proxy list
          (set! inits (alist-set name
                                 value
                                 inits))
           #f);;no set notification
         ((hidden? attr) ;;initial save to proxy list
          (set! hiddens (alist-set name
                                 value
                                 hiddens))
          #f);;no set notification
         (else ;;otherwise save in ordinal attribute list
          (set! attrs (alist-compose name
                                     value
                                     attrs))
          #t))));;do set notification
     ((get self attr)
      (run-callback self (thunk ;;sure run in widget's own context
                          (meta-get attr self))))
     ((set self attr)
      (run-callback self (thunk ;;sure run in widget's own context
                          (meta-set attr self))))
     ((clear-events self)
      (set! attrs (filter (lambda(x) (not (callback? (cdr x)))) attrs)))

     ((recreate-new self)
      (apply simple-notify self
             'action "new"
             (alist->plist inits)))
     ((recreate-attrs self)
      (for-each (lambda(x)
                  (simple-notify-set self (car x) (cdr x)))
                attrs)))))

(define (make-widgets)
  (define-operation call)
  (define-operation remove #f)

  (define-operation delete)
  (define-operation delete-notify)
  (define-operation delete-subwidgets)

  (define-operation recreate)
  (define-operation recreate-subwidgets)
  (define-operation recreate-new)
  (define-operation recreate-attrs)

  (let ((widgets (list #f)))
    (object
     #f
     ((instance-of self) '<container>)
     ((list-subwidgets self)
      (cdr widgets))
     ((insert self w)
      (append! widgets (list w)))
     ((remove self w)
      (set! widgets (delq w widgets)))

     ((delete-subwidgets self)
      (for-each delete (cdr widgets)))
     ((delete self)
      (remove  (self (make-attribute 'parent)) self) ;;remove from parent
      (delete-subwidgets self)
      (call self (make-attribute 'destroyed))
      (delete-notify self))
     ((recreate-subwidgets self)
      (for-each recreate (cdr widgets)))
     ((recreate self)
      (recreate-new self)
      (recreate-subwidgets self)
      (recreate-attrs self)))))

(define (make-context-holder)
  (let ((ctxt (fluid-ref lookout-context)))
    (object
     #f
     ((get-context self) ctxt)
     ((reset-context self) (set! ctxt (fluid-ref lookout-context)))
     ((run-callback self thunk)
      (with-fluids ((lookout-context ctxt)) (thunk)))
     ((call self arg)
      (let ((result (simple-get self arg)))
        (and (callback? result) (run-callback self result)))))))

(define (make-session-glue)
  (define-operation append-message)
  (define-operation take-widget)
  (object
   #f
   ((simple-notify-set self name value)
    (if (callback? value)
        (simple-notify self
                       'action "create-event"
                       'value name)
        (simple-notify self
                       'action "set"
                       'name name
                       'value value)))
   ((simple-notify self . args)
    (let ((session (fluid-ref generic-session)))
      (and session (apply append-message session self (flat-args args)))))
   ((delete-notify self)
    (let ((session (fluid-ref generic-session)))
      (and session (take-widget session self))))))

(define (make-container . type)
  (letrec ((container
            (join
             (lambda args (parse-args container args))
             (make-attributes)
             (make-widgets)
             (make-session-glue)
             (make-context-holder))))
    (container (make-init-attribute 'type 0 (or (cond-car type) "box")))
    container))
