(define-module (alterator session)
	       :use-module (srfi srfi-1)
	       :use-module (srfi srfi-2)
	       :use-module (alterator algo)
	       :export (make-session
			session-set!
			session-ref
			session-user
			session-exists?
			session-delete

			session-obj
			session-obj-id
			session-obj-ref
			session-obj-set!))
;; *sessions* contains data in format (id . user-data), where user-data is an association list
(define *sessions* '())
(define *random-state* (seed->random-state (current-time)))
(define *random-max* 10000000000000000000)

(define (make-id timestamp)
  (string-append (number->string (random *random-max* *random-state*) 16)
		 (number->string timestamp 16)
		 (number->string (random *random-max* *random-state*) 16)))

(define (make-session user lifetime)
  (let* ((timestamp (+ (current-time) lifetime))
	 (id (make-id timestamp))
	 (data `((timestamp . ,timestamp) (user . ,user))))
    (set! *sessions* (acons id data *sessions*))
    id))

(define (session-obj id)
  (assoc id *sessions*))

(define (session-obj-ref x key . default)
  (apply cond-assq key (cdr x) default))

(define (session-obj-set! x key value)
  (set-cdr! x (alist-set key value (cdr x))))

(define (session-obj-id x)
    (car x))

(define (session-exists? id)
  (pair? (assoc id *sessions*)))


(define (session-ref id key . default)
  (cond
    ((assoc id *sessions*)
     => (lambda(x) (apply cond-assq key (cdr x) default)))
    (else (and (pair? default) (car default)))))

(define (session-set! id key value)
  (cond
    ((assoc id *sessions*)
     => (lambda(x) (set-cdr! x (alist-set key value (cdr x)))))))

(define (session-expired? data)
  (<= (cond-assq 'timestamp (cdr data)  0) (current-time)))

(define (session-cleanup)
  (set! *sessions*
    (remove session-expired? *sessions*)))

(define (session-user id)
  (session-cleanup)
  (session-ref id 'user))

(define (session-delete id)
  (set! *sessions* (alist-delete id *sessions*)))

;;; sample data
;; (("id1" . (('user . "bna1") (timestamp . "123123123") (foo . "bar")))
;;  ("id2" . (('user . "bna2") (timestamp . "456345634") (foo . "buz"))))
