(define-module (alterator unit-test)
  :export (test-error
           test-catch

           assert-equal
           assert-not-equal
           assert-eq
           assert-not-eq
           assert-true
           assert-false
	   assert-exception

	   make-test
	   test
	   define-test
	   run-tests
	   test-guard

	   tests-total
	   tests-passed
	   tests-failed
	   print-tests-summary
	   run-test-suite))

;;; assertions

(define (test-error reason)
  (throw 'test-error reason))

(define (test-catch proc handler)
  (catch 'test-error
	 proc
	 (lambda (key . args)
	   (handler (car args)))))

(define (assert-equal reason a b)
  (or (equal? a b) (test-error  (format #f "~A:(equal? ~S ~S)" reason a b))))

(define (assert-not-equal reason a b)
  (and (equal? a b) (test-error  (format #f "~A:(not-equal? ~S ~S)" reason a b))))

(define (assert-eq reason a b)
  (or (eq? a b) (test-error  (format #f "~A:(eq? ~S ~S)" reason a b))))

(define (assert-not-eq reason a b)
  (or (eq? a b) (test-error  (format #f "~A:(not-eq? ~S ~S)" reason a b))))

(define (assert-true reason a)
  (or a (test-error (format #f "~A:(true? ~S)" reason a))))

(define (assert-false reason a)
  (and a (test-error (format #f "~A:(false? ~S)" reason a))))

(define (assert-exception reason exception-lst proc)
  (catch
    #t
    (lambda()
      (proc)
      (or (null? exception-lst)
          (test-error (format #f "~A: exception ~S expected but none was thrown" reason exception-lst))))
    (lambda (key . args)
      (cond
       ((eq? key 'test-error)
         (test-error (car args)))
       ((memq key exception-lst)
         #t)
       ((null? exception-lst)
         (test-error (format #f "~A: no exception expected but was (~S ~S)" reason key args)))
       (else
         (test-error (format #f "~A:exception ~S expected but was (~S ~S)" reason exception-lst key)))))))

;;; setup/teardown
(define (test-guard setup proc teardown)
  (setup)
  (test-catch (lambda() (proc) (teardown))
              (lambda(reason) (teardown) (test-error reason))))

;;; test object

(define (make-test name proc)
    (cons name proc))

(define (run-proc proc)
  (and (procedure? proc) (proc)))

(define (test tst)
  (let ((test-name (car tst))
        (test-proc (cdr tst)))
    (test-catch
	   (lambda()
	     (test-proc)
	     (format #t "[done] (~A)~%" test-name)
	     #t)
	   (lambda (reason)
	     (format #t "[FAIL] (~A) ~A~%" test-name reason)
	     #f))))

(define (run-tests . lst)
  (let loop ((total 0)
	     (passed 0)
	     (failed 0)
	     (lst lst))
    (if (null? lst)
      (vector total passed failed)
      (if (test (car lst))
	(loop (+ 1 total)
	      (+ 1 passed)
	      failed
	      (cdr lst))
	(loop (+ 1 total)
	      passed
	      (+ 1 failed)
	      (cdr lst))))))

(define-macro (define-test <name> <command> . <commands>)
	     `(define ,<name> (make-test ,(symbol->string <name>)
	                                 (lambda() ,<command> ,@<commands>))))

(define (tests-total result)
  (vector-ref result 0))

(define (tests-passed result)
  (vector-ref result 1))

(define (tests-failed result)
  (vector-ref result 2))

(define (run-test-suite name . lst)
  (format #t "~%Running \"~A\"...~%" name)
  (let ((result (apply run-tests lst)))
    (format #t "Completed \"~A\": ~A passes, ~A failures, ~A total~%"
      name
      (tests-passed result)
      (tests-failed result)
      (tests-total result))
    (if (= (tests-failed result) 0)
        (quit 0)
        (quit 1))))
