(define-module (alterator woo)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-13)
  :use-module (alterator plist)
  :use-module (alterator algo)
  :use-module (alterator str)
  :export (woo-gate
	   woo-query

	   woo-error
	   type-error
	   internal-error

	   pack-exception
	   unpack-exception

           woo
           woo-call
           woo-get-option

           woo-catch
	   woo-catch/ignore

           woo-read
           woo-list
           woo-delete
           woo-new
           woo-write

	   woo-first-command
           woo-read-first
	   name+label
	   woo-list/name
	   woo-list/name+label
           woo-list-names

           woo-throw
           woo-try))

;;; exceptions
(define *woo-exceptions* '(woo-error type-error internal-error))

(define (woo-error . args)
  (throw 'woo-error (string-concatenate args)))

(define (type-error . args)
  (apply throw 'type-error args))

(define (internal-error . args)
  (apply throw 'internal-error args))

(define (pack-exception key . args)
  (if (member key *woo-exceptions*)
      (cons key args)
      (cons* 'internal-error key args)))

(define (unpack-exception cmdlist)
  (if (and (pair? cmdlist)
	   (symbol? (car cmdlist)))
    (if (member (car cmdlist) *woo-exceptions*)
      (apply throw cmdlist)
      (error "unsupported exception" cmdlist))
    cmdlist))

;;set of functions to work with woobus from lookout interfaces or second level backends

;; gateway to next telegraph entry
(define woo-gate (make-fluid))
(fluid-set! woo-gate #f)

;;special version of woo-query
(define (woo-query cmd)
  (let ((gate (fluid-ref woo-gate)))
    (if (procedure? gate)
        (gate cmd)
        '((/error reason "woo-gate undefined")))))

(define (woo-get-option cmd option . default)
  (cond
   ((and (pair? cmd) (plistq option (cdr cmd))) => cdr)
   (else (if (pair? default) (car default) ""))))

;;version with predefinded action field
(define (woo action name . args)
  (woo-query (cons* (if (list? name) name (->string name))
                    'action action
                    args)))

(define (woo-call name . args)
  (woo-query (cons* (if (list? name) name (->string name)) args)))

;;; wrappers over standard actions

(define (woo-read name . args) (apply woo "read" name args))
(define (woo-list name . args) (apply woo "list" name args))
(define (woo-delete name . args) (apply woo "delete" name args))
(define (woo-new name . args) (apply woo "new" name args))
(define (woo-write name . args) (apply woo "write" name args))

(define (woo-catch run-code catch-code)
  (catch 'woo-error
    run-code
    (lambda (key arg . rest)
      (catch-code arg))))

(define (woo-catch/ignore thunk)
   (woo-catch
       thunk
       (lambda args #f)))

;;; helpers

(define woo-first-command cond-car)

(define (woo-read-first name . args)
  (cond-car (apply woo-read name args)))

(define (name+label x)
  (let ((name (woo-get-option x 'name)))
    (cons name (woo-get-option x 'label name))))

(define (woo-list/name+label path . args)
  (map name+label (apply woo-list path args)))

(define (woo-list/name path . args)
  (define (name x) (woo-get-option x 'name))
  (map name (apply woo-list path args)))

(define (woo-list-names path . args)
  (apply woo-list/name path args))


;;; backward compatibility
(define woo-throw woo-error)
(define woo-try woo)
