;;; Module for "form-style" work with widgets
;;; We find elements by attribute "name" and get/set attribute "value"

(define-module (alterator ajax)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-2)
  :use-module (alterator woo)
  :use-module (alterator gettext)
  :use-module (alterator plist)
  :use-module (alterator algo)
  :use-module (alterator vector)
  :use-module (alterator object)
  :use-module (alterator context)
  :use-module (alterator session common)
  :use-module (alterator presentation common)
  :use-module (alterator presentation events)
  :use-module (alterator lookout attributes)
  :use-module (alterator lookout messagebox)
  :use-module (alterator lookout globals)
  :export (
	   ;;private API
           form-get-elements-by-name
	   form-element-apply
	   form-label-apply

           ;;new API
           form-callback
	   form-bind
           form-update-enum
           form-update-value
           form-update-value-list
           form-value
           form-value-list
	   form-update-visibility
	   form-update-activity

	   _
	   form-error
	   form-warning
	   form-confirm
	   catch/message

	   form-session-ref
	   form-session-set!))

;;;


(define (form-get-elements-by-name document name)
  (let  ((name-list (if (pair? name) name (list name))))
    (filter-map
      (lambda (x) (and (member (car x) name-list) (cdr x)))
      (document namelist))))

(define (form-get-label document name)
  (or (cond-assoc name (or (document namereflist) '())) ;;new style label binding to field
      (any (lambda(w) (and (string=? (w type) "label") w))
	   (form-get-elements-by-name document name)))) ;;old style label binding to field

;;; interactive form elements (inputs)

(define *input-list* '("checkbox"
			   "edit"
			   "listbox"
			   "spinbox"
			   "slider"
			   "timeedit"
			   "dateedit"
			   "combobox"
			   "listbox"
			   "textbox"
			   "radiolistbox"
			   "multilistbox"
			   "checklistbox"
			   "radio"
			   "button"))

;; is element interactive?
;; (can we set or get value for given element?)
(define (input-element? el)
  (member (el type) *input-list*))

;; find input element by name
(define (form-get-input document name)
  (let ((elements (form-get-elements-by-name document name)))
    (cond
      ((null? elements)
       #f)
      ((null? (cdr elements))
       (car elements))
      (else
	(let ((inputs (filter input-element? elements)))
	  (cond
	    ((null? inputs)
	     #f)
	    ((null? (cdr inputs))
	     (car inputs))
	    (else
	      inputs)))))))


;; apply arguments to widgets specified by name
;; (not only interactive elements!)

(define (form-element-apply document namelist . args)
  (map
    (lambda (el) (apply el args))
    (form-get-elements-by-name document namelist)))

(define (form-label-apply document namelist . args)
  (map
    (lambda (name)
      (let ((el (form-get-label document name)))
        (and el (apply el args))))
    namelist))

;;; new form API
(define (enumref-fill-row cmd row)
  (vector-map (lambda(x)
		(cons (woo-get-option cmd (car x))
		      (woo-get-option cmd (cdr x))))
	      row))

(define (radio? x)
  (string=? (x type) "radio"))

(define (radiolist? lst)
  (and (pair? lst)
       (every radio? lst)))

(define (form-value name)
  (let* ((document (global 'document:root))
	 (el (form-get-input document name)))
    (cond
      ((not el)
       (and (string=? name "language") (session-language)))
      ((radiolist? el)
       (or (any (lambda (x) (and (x state) (x value))) el) ""))
      ((radio? el)
       (if (el state) (el value) ""))
      ((string=? (el type) "button")
       #f)
      ((string=? (el type) "label")
       (el text))
      (else (el value)))))

(define (form-value-list . name-list)
  (let* ((document (global 'document:root))
	 (name-list (if (pair? name-list)
		      (car name-list)
		      (delete-duplicates (map car (document namelist))))))
    (fold (lambda (name lst)
	    (let* ((value (form-value name)))
	      (if value
		(cons* (string->symbol name) value lst)
		lst)))
	  '()
	  name-list)))

(define (form-update-enum name variants)
  (and-let* ((document (global 'document:root))
	     (widget (form-get-input document name))
	     (t (widget type)))
	    (widget enum-rows (map (lambda(x) (woo-get-option x 'name))
				   variants))
	    (cond
	      ((member t '("listbox" "combobox"))
	       (let* ((r (or (widget row) '#((label . ""))))
		      (labels (map (lambda(x) (enumref-fill-row x r))
				   variants)))
		 (widget rows labels)))
	      ((string=? t "checklistbox")
	       (let ((labels (map (lambda(x) (cons (woo-get-option x 'label) #f))
				  variants)))
		 (widget rows labels)))
	      (else (form-error "widget type without enum" t)))))

(define (form-update-value name val)
  (let* ((document (global 'document:root))
	 (el (form-get-input document name)))
    (cond
      ((not el)
       #f)
      ((radiolist? el)
       (for-each (lambda(x) (x state (string=? (x value) val))) el))
      ((radio? el)
       (el state (string=? (el value) val)))
      ((string=? (el type) "label")
       (el text val))
      (else (el value val)))))

;; (form-update-value-list namelist cmd)
;; or
;; (form-update-value-list cmd)
(define (form-update-value-list arg1 . arg2)
  (let* ((namelist (map string->symbol (if (pair? arg2) arg1 '())))
         (cmd (if (pair? arg2) (car arg2) arg1))
	 (cmdlist (if (pair? (cond-car cmd)) cmd (list cmd))))
    (for-each
      (lambda(cmd)
	(plist-for-each
	  (lambda(x y)
	    (if (or (null? namelist) (member x namelist))
	      (form-update-value (symbol->string x) y)))
	  (or (cond-cdr cmd) '())))
      cmdlist)))

(define *event-map* '(("click" . clicked)
		      ("change" . changed)))

(define (make-event event el)
  (case event
    ((changed)
     (if (string=? (el type) "radio")
       (make-attribute 'toggled)
       (make-attribute 'changed)))
    (else (make-attribute event))))

(define (form-bind name orig-event proc)
  (let* ((document (global 'document:root))
         (ctxt (fluid-ref lookout-context))
	 (event (cond-assoc orig-event *event-map*))
	 (el (or (form-get-input document name)
		 (form-error "form-bind: widget for name ~S not found" name)))
	 (el-list (if (pair? el) el (list el))))
    (or event (form-error "form-bind: event ~S unsupported" orig-event))
    (for-each (lambda(x)
		(x (make-event event x)
		   (make-callback (with-context ctxt (proc)))))
	      el-list)))

(define (form-update-activity name status)
  (let ((document (global 'document:root))
	(namelist (if (pair? name) name (list name))))
    (form-element-apply document namelist activity status)
    (form-label-apply document namelist activity status))) ;; also enable/disable label

(define (form-update-visibility name status)
  (let ((document (global 'document:root))
	(namelist (if (pair? name) name (list name))))
    (form-element-apply document namelist visibility status)
    (form-label-apply document namelist visibility status))) ;; also show/hide label

(define (form-callback name url)
  (let* ((url (string-trim url #\/))
	 (proc (dynamic-require `(ui ,@(map string->symbol (string-split url #\/)) ajax)
				(string->symbol name))))
    (and (procedure? proc)
	 (proc))))

;; various error messages

(define (form-error fmt . args)
  (document:popup-critical (apply format #f fmt args) 'ok)
  #f)

(define (form-warning fmt . args)
  (document:popup-warning (apply format #f fmt args) 'ok)
  #f)

(define (form-confirm msg)
  (eq? 'yes (document:popup-question msg 'no 'yes)))

(define (type-error-string reasonlist)
  (string-concatenate
    (filter-map (lambda (x)
                  (let ((l (form-get-label (global 'document:root) (car x))))
                      (and l (format #f "~A ~A<br>" (l text) (cdr x)))))
                reasonlist)))

(define (catch/message proc)
  (catch 'type-error
	 (lambda()
	   (catch 'woo-error
		  proc
		  (lambda (key reason)
		    (form-error "~A" reason))))
	 (lambda(key reasonlist)
	   (form-error "~A" (type-error-string  reasonlist)))))

;;; i18n

(define (default-form-domain)
  (default-po-domain (or (global 'document:url) "")))

(define (_ str . domain)
  (let* ((domain (if (pair? domain) (car domain) (default-form-domain)))
	 (language (form-value "language"))
	 (tr (make-translator domain language)))
    (tr str)))


;;; per form and per session global variables

(define (form-session-key name)
  (string->symbol (string-append "-form-session-" (global 'document:url) "-" name)))

(define (form-session-ref name)
  (global (form-session-key name)))

(define (form-session-set! name value)
  (set-global! (form-session-key name) value))
