(define-module (ui openvpn-server ajax)
			   :use-module (srfi srfi-2)
			   :use-module (alterator ajax)
			   :use-module (alterator woo)
			   :export (init))

(define (update-type-interface)
  (let ((routed (string=? (form-value "type") "routed")))
	(form-update-visibility "vpnnet" routed)
	(form-update-visibility "bridge" (not routed))
	(form-update-visibility "server_networks" routed)
	(form-update-visibility "server_net" routed)
	(form-update-visibility "server_netmask" routed)
	(form-update-visibility "add_network" routed)
	(form-update-visibility "remove_network" routed)
	(form-update-visibility "gateway_vpnaddr" (not routed))
	(form-update-visibility "vpnpool_start" (not routed))
	(form-update-visibility "vpnpool_end" (not routed))
	(form-update-visibility "clients_managment" routed)))

(define (form-update-enum-set-first name variants)
  (form-update-enum name variants)
  (form-update-value name (woo-get-option (car variants) 'name)))

(define (ui-init)
  (form-update-visibility "ca_upload_message" #f)
  (form-update-enum-set-first "type" (woo-list "/openvpn-server/avail_types" 'language (form-value "language")))
  (form-update-enum "server_netmask" (woo-list "/openvpn-server/avail_masks"))
  (form-update-enum "vpnnetmask" (woo-list "/openvpn-server/avail_masks"))
  (read-config))

(define (read-config)
  (let ((cmd (woo-read-first "/openvpn-server" 'type (form-value "type") 'language (form-value "language"))))
	(form-update-enum "server_networks" (woo-list "/openvpn-server/avail_server_networks"))
	(form-update-enum "bridge" (woo-list "/openvpn-server/avail_bridges"))
	(form-update-value-list
	  '("enabled" "type" "bridge" "port" "server_net" "server_netmask" "vpnnet" "vpnnetmask"
		"gateway_vpnaddr" "vpnpool_start" "vpnpool_end" "lzo" "use_tcp")
	  cmd)
	(update-type-interface)))

(define (get-addr network)
  (and network
	   (car (string-split network #\/))))

(define (get-netmask network)
  (and network
	   (cadr (string-split network #\/))))

(define (update-server-net-values network)
  (and network
	   (begin
		 (form-update-value "server_net" (get-addr network))
		 (form-update-value "server_netmask" (get-netmask network)))))

(define (write-config reason)
  (catch/message
	(lambda()
	  (apply woo-write
			 "/openvpn-server"
			 'operation reason
			 (form-value-list
			   '("enabled" "type" "bridge" "port" "server_net" "server_netmask" "vpnnet" "vpnnetmask"
				 "gateway_vpnaddr" "vpnpool_start" "vpnpool_end" "lzo" "use_tcp" "language"))))))

(define (add-network)
  (write-config "add-server-network")
  (read-config))

(define (remove-network)
  (and-let* ((network (form-value "server_networks"))
			 (addr (get-addr network))
			 (mask (get-netmask network)))
			(catch/message
			  (lambda()
				(woo-write
					   "/openvpn-server"
					   'operation "remove-server-network"
					   'type "routed"
					   'server_net addr
					   'server_netmask mask
                       'language (form-value "language"))
				(read-config)))))

(define (apply-config)
  (write-config "apply"))

(define (reset-config)
  (write-config "reset")
  (read-config))

(define (clients-networks-interface)
  (write-config "config")
  (form-replace "/openvpn-server/clients"))

(define (certificate-interface)
  (write-config "config")
  (form-replace "/sslkey/generate?backend=openvpn-server"))

(define (on-upload)
  (form-update-visibility "ca_upload_message" #f)
  (call-with-form-file
    "ca_cert"
    (lambda(path)
      (and
        (catch/message
          (lambda() (woo-write "/openvpn-server/upload-ca" 'path path 'language (form-value "language"))))
        (form-update-visibility "ca_upload_message" #t)))))

(define (init)
  (ui-init)
  (form-bind "type" "change" read-config)
  (form-bind-upload "upload_button" "click" "ca_cert" on-upload)
  (form-bind "server_networks" "change" (lambda() (update-server-net-values (form-value "server_networks"))))
  (form-bind "add_network" "click" add-network)
  (form-bind "remove_network" "click" remove-network)
  (form-bind "clients_managment" "click" clients-networks-interface)
  (form-bind "reset" "click" reset-config)
  (form-bind "apply" "click" apply-config)
  (form-bind "certificate" "click" certificate-interface))

