(document:surround "/std/frame")

;;; Helpers for checklistbox

(define *prev-current-rows* (make-cell #f))

(define (checklistbox-current)
  (let* ((cr (packages current-rows))
	 (pcr (cell-ref *prev-current-rows*)))
    (cond-car (if pcr (lset-difference = cr pcr) cr))))

(define (checklistbox-update-current)
  (cell-set! *prev-current-rows* (packages current-rows)))

(define (checklistbox-revert-current)
  (packages current-rows (cell-ref *prev-current-rows*)))

(define (checklistbox-state num)
  (list-ref (packages current-rows) num))

;;; Features: list of packages

(define *pkg-lists* (make-cell '()))

(define (make-pkg-lists pkg-groups)
  (cell-set! *pkg-lists* (map (lambda(x) (woo-get-option x 'lists)) pkg-groups)))

; Lookup field value by id and field name
(define (pkg-lookup-field pkg-groups id field)
  (if (not-empty-string? id)
    (woo-get-option 
      (car 
        (filter (lambda (x) (string=? (woo-get-option x 'name) id)) pkg-groups))
	  field)))

;;; Features: Lists

; Return list of checked group lists
(define *groups-list* (make-cell #f))

(define (pkg-install-list)
  (if 
    (proper-list? (packages current-rows))
 	(string-join 
 		(map 
 		(lambda (x) (pkg-lookup-field (cell-ref *groups-list*) x 'lists))
 		(packages current-rows))
 		" ")))

; Check if package with specified name is exists
(define (pkg-exists? name)
  (and
    (not-empty-string? name)
    (any 
      (lambda (x) (string=? (woo-get-option x 'name) name))
      (cell-ref *groups-list*))))

; Mark all depended item for current checked
; Need to check wrong package names in dependency list
(define (pkg-depends x)
  (filter (lambda (y) (pkg-exists? y))
    (string-split
      (pkg-lookup-field (cell-ref *groups-list*) x 'depends)
      #\ )))

; Mark checked all items depended on currently checked
(define (pkg-marks-depended)
  (if 
    (proper-list? (packages current-rows))
    (packages current-rows
    ;(format #t "CHECKED: ~S\n"
       (filter (lambda (x) (not-empty-string? x))
         (apply append 
           (map (lambda (pkg) (append (list pkg) (pkg-depends pkg)))
             (packages current-rows))))
    )))

; Generate rows content and hierarchy for tree. Prepare data tuple in CheckTree 
; format of rows method. See http://www.altlinux.org/Alterator/CheckTree#API 
; for details
(define (row-content x)
  (list 
	(woo-get-option x 'name) 
	(woo-get-option x 'parent)
	(woo-get-option x 'label) 	
  ))

; Check item state. 
; Return flag if group is explicitly marked as required in direcrory file.
(define (is-checked x)
  (or
    (string=? (woo-get-option x 'required) "yes")
    (string=? (woo-get-option x 'required) "true")))

; Generate checked item list
(define (row-checked x)
  (woo-get-option x 'name))


;;; Features: Conflicts
(define *pkg-conflicts* (make-cell '()))

(define (name+conflicts x)
  (cons (woo-get-option x 'name)
	(woo-get-option x 'conflicts)))

(define (get-index lst)
  (lambda(x)
    (list-index (lambda (y) (string=? x y)) lst)))

; Build conflict relations
(define (make-pkg-conflicts pkg-groups)
  (let* ((lst (map name+conflicts pkg-groups))
	 (names (map car lst))
	 (conflicts (map cdr lst)))
	 (cell-set! *pkg-conflicts* (map (get-index names) conflicts))))

(define (pkg-conflicts-num num)
  (list-ref (cell-ref *pkg-conflicts*) num))

(define (pkg-conflicts-name num)
  (car (list-ref (packages checklist-rows) num)))

(define (check-pkg-conflicts num)
  (let* ((c-num (pkg-conflicts-num num)))
    (not (and (number? c-num)
	      (checklistbox-state c-num)
	      (begin (document:popup-critical
		       (format #f (_ "Has conflict with ~S") (pkg-conflicts-name c-num)))
		     #t)))))

; Check single checked item
;(define (check-group-conflicts name)
;  (let conflicts (pkg-lookup-field (pkg-groups id field))
;)

; Look for conflicts in all checked items
;(define (check-pkg-conflicts)
;  (map check-group-conflicts (packages current-rows)))

;;; Features: Disk usage

(define (string-prefix-at-pos str prefix pos)
  (and (string-prefix? prefix str)
       (list-ref (string-cut str #\:) pos)))

; Generate installed software size in human readable format
; TODO need to add space between number and units
(define (pkg-size-string msg)
  (if (string-null? msg)
    msg
    (let* ((len (- (string-length msg) 1))
	   (first-msg (substring msg 0 len))
	   (last-msg (string-ref msg len)))
      (string-append
	first-msg
	(case last-msg
	  ((#\k) (_ " kB"))
	  ((#\M) (_ " MB"))
	  ((#\G) (_ " GB"))
	  ((#\T) (_ " TB"))
	  (else (string last-msg)))))))
	
(define (pkg-size-label msg)
  (size text (format #f "~A: ~A" (_ "Required disk space") msg)))

(define (pkg-message-handler msg)
  (and-let* ((msg (cond-plistq 'message msg #f)))
	    (cond
	      ((string=? msg "pkg-size:start")
	       (pkg-size-label (_ "calculating...")))
	      ((string-prefix-at-pos msg "apt-get:status:disk-size:" 3)
	        => (lambda(msg) (pkg-size-label (pkg-size-string msg))))
	      ((string-prefix-at-pos msg "pkg-size:finish" 2) =>
	        (lambda(result)
		  (if (string=? result "100") 
		    (begin 
		      (pkg-size-label (_ "unable to calculate"))
		      (frame:next-activity #f)
		    )
		    (frame:next-activity #t)))))))


(define (pkg-error-handler msg)
  (and-let* ((error-text (cond-plistq 'error msg #f)))
	    (format #t "alterator-pkg: ~S~%" error-text)))

(define (make-pkg-size)
  (add-mailbox pkg-message-handler)
  (add-mailbox pkg-error-handler)
  (pkg-size-label "0 kB"))

(define (remove-pkg-size)
  (remove-mailbox pkg-message-handler)
  (remove-mailbox pkg-error-handler)
  (woo-write "/pkg-size" 'lists ""))

(define (check-pkg-size)
  ;(format #t "INSTALLED: ~S\n" (pkg-install-list)))
  (woo-write "/pkg-size" 'lists (pkg-install-list)))


;;; UI
(gridbox
  columns "20;50;20"

  ;;
  (spacer)
  (label text (_ "Additional applications:"))
  (spacer)
  ;;
  (spacer)

  (document:id packages (checktree
			  (when changed
				;(check-pkg-conflicts (packages current))
				(pkg-marks-depended)
				(check-pkg-size))))

  (spacer)
  ;;
  (spacer)
  (document:id size (label text (_ "Preparing...")))
  (spacer))

;;; Logic

(frame:on-next (thunk (remove-pkg-size)
		      (frame:replace "/pkg/install" 'lists (pkg-install-list))
		      'cancel))

(frame:on-back (thunk (remove-pkg-size)))

(document:root
  (when loaded
    (woo-catch/message
      (thunk
        (frame:back-activity #f)
		(woo-write "pkg-init")
	; Read list data from /var/lib/install3/groups/*.directory
	(let* ((pkg-groups (woo-list "/pkg-groups")))
     (if (null? pkg-groups)
        (frame:replace "/pkg/install")
	(begin
		  
	  ; Fill packages list
	  (make-pkg-lists pkg-groups)
	  (cell-set! *groups-list*  pkg-groups)
		  
	  ; Build conflict relations
	  (make-pkg-conflicts pkg-groups)
	      
	  ; Calculate installed software size
	  (make-pkg-size)
	      
	  ; Fill group tree rows
	  (packages rows (map row-content pkg-groups))
	  ; Set checked item state
	  (packages current-rows (map row-checked (filter is-checked pkg-groups)))

          ; Mark all depended group
          (pkg-marks-depended)
          
          ; Inital size of installation
          (check-pkg-size)
)))))))
