(define-module (ui snort rules ajax)
    :use-module (alterator ajax)
    :use-module (alterator woo)
    :use-module (alterator effect)
    :use-module (srfi srfi-1)
    :use-module (srfi srfi-2)
    :use-module (srfi srfi-11)
    :export (init))

(define *rules-list* '())
(define *rules-unused-list* '())
(define *changed* #f)

(define (read-state)
  (catch/message
    (lambda()
      (form-update-value-list '("state_enabled")
                              (woo-read-first "/snort/state_enabled"
                                              'language (form-value "language"))))))

(define (update-rules-list rlist)
  (set! *rules-list* rlist)
  (form-update-enum "rules" rlist))

(define (update-rules-unused-list rlist)
  (set! *rules-unused-list* rlist)
  (form-update-enum "rules_unused" rlist)
  (form-update-activity "add_button" (not (null? rlist))))

(define (read-rules-list)
  (catch/message
    (lambda()
      (update-rules-list (woo-list "/snort/avail_rules" 'language (form-value "language"))))))

(define (read-rules-unused-list)
  (catch/message
    (lambda()
      (update-rules-unused-list (woo-list "/snort/avail_unused_rules" 'language (form-value "language"))))))

(define (rules-list)
  (map 
    (lambda(x)
      (second (member 'name x)))
    *rules-list*))

(define (rules-list-changed changed)
  (set! *changed* changed)
  (form-update-activity "apply_button" changed))

(define (rules-list-elem-name rlist)
  (second (member 'name rlist)))

(define (rules-list-elem-index name rlist)
  (and (string? name)
       (not (string=? name ""))
       (list-index (lambda(x)
                     (string=? name (rules-list-elem-name x)))
                   rlist)))

(define (rules-list-elem name rlist)
  (find (lambda(x)
          (string=? name (rules-list-elem-name x)))
        rlist))

(define (rules-list-elem=? el1 el2)
  (string=? (rules-list-elem-name el1) (rules-list-elem-name el2)))

(define (swap-elements n rlist)
  (let*-values (((head temp-tail) (split-at rlist n))
                ((pair tail) (split-at temp-tail 2)))
               (append head (cdr pair) (list (car pair)) tail)))

(define (move-element n)
  (and-let* ((name  (form-value "rules"))
             (index (rules-list-elem-index name *rules-list*))
             (shift (- index n)))
            (and (>= shift 0)
                 (< (+ shift 1)(length *rules-list*))
                 (begin
                   (update-rules-list (swap-elements shift *rules-list*))
                   (form-update-value "rules" name)))))

(define (update-activity)
  (let ((val (form-value "auto_update")))
    (form-update-activity "weekday" (and val
                                         (string=? "weekly" (form-value "period"))))
    (form-update-activity "monthday" (and val
                                          (string=? "monthly" (form-value "period"))))))

(define (ui-read)
  (read-rules-list)
  (read-rules-unused-list)
  (rules-list-changed #f))

(define (read-download-data)
  (form-update-value-list '("rules_url" "oinkcode" "custom_url" "auto_update" "period" "weekday" "monthday" "time")
                          (woo-read-first "/snort/download-data" 'language (form-value "language")))
  (update-effect)
  (update-activity))

(define (ui-init)
  (if (and (null? *rules-list*)
           (null? *rules-unused-list*))
    (ui-read)
    (begin (update-rules-list *rules-list*)
           (update-rules-unused-list *rules-unused-list*)
           (rules-list-changed *changed*)))
  (form-update-enum "weekday" (woo-list "/snort/avail_weekday" 'language (form-value "language")))
  (read-download-data))

(define (on-add)
  (let*-values (((add-list-names) (string-split (or (form-value "rules_unused")
                                                    "")
                                                #\;))
                ((add-list unused-list) (partition
                                          (lambda(x)
                                            (any (lambda(y)
                                                   (string=? y (rules-list-elem-name x)))
                                                 add-list-names))
                                          *rules-unused-list*)))
               (and (not (null? add-list-names))
                    (begin
                      (let* ((name (form-value "rules"))
                             (index (rules-list-elem-index name *rules-list*))
                             (n (if index
                                  (+ index 1)
                                  (length *rules-list*))))
                        (update-rules-list (append (take *rules-list* n) add-list (drop *rules-list* n)))
                        (form-update-value "rules" name)
                        (update-rules-unused-list unused-list)
                        (rules-list-changed #t))))))

(define (on-remove)
  (let ((elem (rules-list-elem (form-value "rules") *rules-list*)))
    (and elem
         (not (null? *rules-list*))
         (begin
           (update-rules-list (remove 
                                (lambda(x)
                                  (rules-list-elem=? x elem))
                                *rules-list*))
           (update-rules-unused-list (append *rules-unused-list* (list elem)))
           (rules-list-changed #t)))))

(define (on-up)
  (move-element 1))

(define (on-down)
  (move-element 0))

(define (on-apply)
  (woo-write "/snort/rules-list" 
             'rules_list (rules-list))
  (ui-read))

(define (on-reset)
  (ui-read))

(define (read-rule-description)
  (catch/message
    (lambda()
      (let ((rule (form-value "rules")))
        (and (string? rule)
             (form-update-value-list '("rule_description")
                                     (woo-read-first "/snort/rule-description"
                                                     'rules rule
                                                     'language (form-value "language"))))))))

(define (on-download-apply)
  (catch/message
    (lambda()
      (apply woo-write "/snort/download-data"
             (form-value-list
               '("rules_url" "oinkcode" "custom_url" "auto_update" "period" "weekday" "monthday" "time" "language")))))
  (read-download-data))

(define (on-download)
  (catch/message
    (lambda()
      (woo-write "/snort/download-now"
                 'language (form-value "language"))))
  (ui-read))

(define (init)
  (effect-enable "period" "auto_update" #t)
  (effect-enable "time" "auto_update" #t)
  (effect-enable "weekday" "period" "weekly")
  (effect-enable "monthday" "period" "monthly")
  (effect-enable "oinkcode" "rules_url" "oinkcode")
  (effect-enable "custom_url" "rules_url" "custom")
  (effect-disable "auto_update" "rules_url" #f) 
  (init-effect)
  (form-bind "auto_update" "change" update-activity)
  (form-bind "add_button" "click" on-add)
  (form-bind "remove_button" "click" on-remove)
  (form-bind "up_button" "click" on-up)
  (form-bind "down_button" "click" on-down)
  (form-bind "apply_button" "click" on-apply)
  (form-bind "reset_button" "click" on-reset)
  (form-bind "rules" "change" read-rule-description)
  (form-bind "download_apply_button" "click" on-download-apply)
  (form-bind "download_button" "click" on-download)
  (ui-init))
