=begin

= Synopsis

   require "tempfile"

   tf = Tempfile.new("afile")
   tf.path
   tf.puts("Cosi Fan Tutte")
   tf.close
   tf.open
   tf.gets
   tf.close(true)

= Description

Class (({Tempfile})) creates managed temporary files. Although they
behave like any other (({IO})) object, temporary files are automatically 
deleted when the Ruby program terminates. Once a (({Tempfile})) object
has been created, the underlying file may be opened and closed a
number of times in succession.

(({Tempfile})) does not directly inherit from (({IO})). Instead it
delegates calls to a (({File})) object. From the programmer's
perspective, apart from the unusual ((<(({new}))|Tempfile.new>)),
((<(({open}))|Tempfile#open>)), and ((<(({close}))|Tempfile#close>))
semantics, a (({Tempfile})) object behaves as if it were an (({IO}))
object.  = Class Methods

--- Tempfile.new( basename, tmpdir=see below )
      Constructs a temporary file in the given directory. The file
      name is build by concatenating ((|basename|)), the current
      process id and (as an extension) a unique sequence number. If
      the ((|tmpdir|)) parameter is not supplied, it defaults to the
      value of one of the environment variables (({TMPDIR})), (({TMP})), 
      or (({TEMP})), or to the directory (({/tmp})).
      
      The file is then opened using mode ``w+'', which allows reading
      and writing, and which deletes any existing content.

--- Tempfile.open( basename, tmpdir )
      Synonym for ((<Tempfile.new>)).

= Instance Methods

--- Tempfile#open
      Reopens ((|aTempfile|)) using mode ``r+'', which allows reading and
      writing, but does not delete existing content.

  
--- Tempfile#close( final=false )
      Closes ((|aTempfile|)). If ((|final|)) is true, deletes the underlying
      real file. If ((|final|)) is false, ((|aTempfile|)) may be
      subsequently reopened. In all cases, the underlying file will be 
      deleted when the program terminates.
  
--- Tempfile#path
      Returns the full path of the underlying file.

= History

    $Id: tempfile.rb,v 1.1 2001/01/23 21:58:45 dave Exp $
=end

require 'delegate'
require 'final'

class Tempfile < SimpleDelegator
  Max_try = 10

  def Tempfile.callback(path, data)
    lambda{
      print "removing ", path, "..." if $DEBUG
      data[0].close if data[0]
      if File.exist?(path)
	File.unlink(path) 
      end
      if File.exist?(path + '.lock')
	Dir.rmdir(path + '.lock')
      end
      print "done\n" if $DEBUG
    }
  end

  def initialize(basename, tmpdir=ENV['TMPDIR']||ENV['TMP']||ENV['TEMP']||'/tmp')
    umask = File.umask(0177)
    begin
      n = 0
      while true
	begin
	  tmpname = sprintf('%s/%s%d.%d', tmpdir, basename, $$, n)
	  lock = tmpname + '.lock'
	  unless File.exist?(tmpname) or File.exist?(lock)
	    Dir.mkdir(lock)
	    break
	  end
	rescue
	  raise "cannot generate tmpfile `%s'" % tmpname if n >= Max_try
	  #sleep(1)
	end
	n += 1
      end

      @protect = []
      @clean_files = Tempfile.callback(tmpname, @protect)
      ObjectSpace.define_finalizer(self, @clean_files)

      @tmpfile = File.open(tmpname, 'w+')
      @protect[0] = @tmpfile
      @tmpname = tmpname
      super(@tmpfile)
      Dir.rmdir(lock)
    ensure
      File.umask(umask)
    end
  end

  def Tempfile.open(*args)
    Tempfile.new(*args)
  end

  def open
    @tmpfile.close if @tmpfile
    @tmpfile = File.open(@tmpname, 'r+')
    @protect[0] = @tmpfile
    __setobj__(@tmpfile)
  end

  def close(real=false)
    @tmpfile.close if @tmpfile
    @protect[0] = @tmpfile = nil
    if real
      @clean_files.call
      ObjectSpace.undefine_finalizer(self)
    end
  end

  def path
    @tmpname
  end
end

if __FILE__ == $0
#  $DEBUG = true
  f = Tempfile.new("foo")
  f.print("foo\n")
  f.close
  f.open
  p f.gets # => "foo\n"
  f.close(true)
end
