#!/usr/bin/env ruby

require 'xmltreebuilder'
require 'ri/refdoc.rb'

DATADIR = "srcdesc"
INDEX_FILE = File.join(DATADIR, "index")

##
# Return the fully qualified name of a method (either Class::meth or Class#meth
#

def fqn(cl, meth)
  sep =  (meth.type == "class") ? "::" : "#"
  cl.name + sep + meth.name
end

def deXML(str)
  str.
    gsub(/&lt;/,  '<').
    gsub(/&gt;/,  '>').
    gsub(/&amp;/, '&').
    sub(/ \(backquote\)/, '')
end

def flatten(node)
  res = ""
  node.childNodes do |child|
    case child
    when Element
      name = child.nodeName
      pre =  "<#{name}>"
      post = "</#{name}>"
      res << pre + flatten(child) + post
    when Text
      res << child.nodeValue.tr_s("\n ", "  ")
    else
      raise child.nodeName
    end
  end
  res.strip
end

def handleVerbatim(node)
  txt = ""
  node.childNodes do |child|
    case child
    when Element
      name = child.nodeName
      case name
      when "code"
        txt << handleVerbatim(child)
      when "m1"
        txt << handleVerbatim(child) << COLUMN_SEP
      when "m2"
        txt << handleVerbatim(child)
      when "MULTI"
        txt << handleVerbatim(child)
      when "p"
        txt << "\n"
      else
        raise name
      end
    when Text
      txt << child.nodeValue.gsub(/--@/, '-@')
    else
      raise child.nodeName
    end
  end
  txt
end

def handleDescription(m, desc, stopAt="")
  txt = ""
  desc.childNodes do |child|
    case child
    when Element
      name = child.nodeName
      break if name == stopAt
      case name
      when "i", "code", "b", "em"
        txt << "<#{name}>" << flatten(child) << "</#{name}>"
      when "p", "verbatim"
        txt.squeeze!("\n")
        m.addFragment(Paragraph.new(txt)) unless txt =~ /\A[\n\s]*\Z/m
        txt = ""
        if name == "verbatim"
          m.addFragment(Verbatim.new(handleVerbatim(child)))
        end
      when "subclasses"
        ;
      else
        ;
      end
    when Text
      txt << child.nodeValue unless txt == "\n"
    else
      raise child.nodeName
    end
  end
  txt.squeeze!("\n")
  m.addFragment(Paragraph.new(txt)) unless txt =~ /\A[\n\s]*\Z/m
end

def processMethod(cname, type, method)
  mname = deXML(method.attributes['name'].nodeValue).
    gsub(/\s/, '').
    sub(/--/, '-')

  callseq = method.getElementsByTagName("callseq")[0]
  desc    = method.getElementsByTagName("desc")[0]
  m = MethodDesc.new(mname, cname, type, flatten(callseq))

  handleDescription(m, desc)
  m
end

def processMethodList(cl, ml)
  type = ml.attributes['type'].nodeValue
  ml.getElementsByTagName("method").each do |method|
    cl.addMethod(processMethod(cl.name, type, method))
  end
end

def processClass(aClass)
  cname = aClass.attributes['name'].nodeValue
  csuper = aClass.attributes['super'].nodeValue
  ctype  = aClass.attributes['type'].nodeValue

  cl = ClassModule.new(cname, csuper, ctype)

  # look for 'subclasses' element
  subclasses = nil
  n = aClass.firstChild
  while n
    if XML::SimpleTree::Element === n and n.nodeName == "subclasses"
      cl.addSubclasses(flatten(n).split(/\s*,\s*/))
      break
    end
    n = n.nextSibling
  end

  handleDescription(cl, aClass, "methods")

  aClass.getElementsByTagName("methods").each do |methodList|
    processMethodList(cl, methodList)
  end

  cl
end

def serializeSource(object, fileName)
  File.open(fileName, "w") do |f|

    f.puts "# Automatically generated....\n"
    f.puts "raise 'Must be invoked by installation process' unless \$opfile"
    f.puts object.to_src("aClass")
    
    f.puts %{File.open(\$opfile, "w") {|f| Marshal.dump(aClass, f) }}
  end
end

###############################################################33

# read in existing index (or create a new one)

index = nil
begin
  File.open(INDEX_FILE, "r") do |f|
    index = Marshal.load(f)
  end
rescue
  index = MethodIndex.new
end

xml = $<.read

builder = XML::SimpleTreeBuilder.new

tree = builder.parse(xml)

root = tree.documentElement
root.normalize

classes = root.getElementsByTagName("class")

#
# Now we write out a serialization of each class, but in
# Ruby source code format
#

classes.each do |aClass|
  cl = processClass(aClass)

  fileName = cl.name.tr(':','_') + ".rb"
  puts fileName

  serializeSource(cl, File.join(DATADIR, fileName))

  # and update the index

  index.updateWith(cl)

end


# Finally write the index back out, both in binary and in source

File.open(INDEX_FILE, "w") do |f|
  Marshal.dump(index, f)
end

serializeSource(index, INDEX_FILE + ".rb")


