#!/usr/bin/env ruby -w
#
# Scan an input file, looking for fragments of Ruby code between
#
#    \begin{ruby}[options]
#        :   :
#    \end{ruby}
#
# Each fragment is a standalone Ruby program. The default behaviour is
# to render the fragment as printable by LaTeX, run it, and include its 
# output.
#
#    \begin{ruby}
#       for i in 1..3
#         p i
#       end
#    \end{ruby}
#
# Will result in the typeset output
#
# |     for i in 1..3
# |        p i
# |     end
# |
# |  Produces:
# |
# |     1
# |     2
# |     3
#
# If the option list includes 'comment', then the result of
# each line will be shown as a pseudo comment next to that line:
#
#    \begin{ruby}[comment]
#       a = 2
#       b = 3
#       a + b
#       a * b
#    \end{ruby}
#
# Will result in the typeset output
#
# |     a = 2
# |     b = 3
# |     a + b   ->  5
# |     a * b   ->  6
#
# The [norun] option typesets the code but does not run it. The [synerr] 
# option says that a syntax (or runtime) error is acceptable, and outputs
# that error
#
#
# Within the fragment of code, the following flags are allowed
#
#   !-<code>          - include code in program, but don't typeset
#   !+<stuff>         - include stuff in typeset output, but not in code
#   ![\d\w]           - include in output only if corresponding flag is
#                       set in options 'include=135'
#   !!<filename>      - include contents of filename
#   !!-<filename>     - include but don't output filename
#   !!+<filename>     - include filename for output (but don't compile)


#################################################################
#
# Kinda like the real gets, but if we come across a line that looks
# like !!<name>, start returning lines from that file, then at end of
# file continue from the original. 

require 'rbconfig'

SAMPLESDIR = 'samples'
$fileName = nil

class NestedIO

  def initialize(stream)
    @ip = stream
    @ipStack = []

    @prefix = nil
  end

  # Yield successive lines, making !!<file> transparent
  def each
    loop do
      while (line = @ip.gets).nil?    # unstack at end of file
        @ip.close
        return nil if @ipStack.empty?
        @ip, @prefix = @ipStack.pop
      end

      # otherwise look for file inclusion. If the !! command has a flag, we include
      # in on all lines that come out of that file

      while line =~ /^!!([-+0-9A-Z])?(.*)/ 
        
        @ipStack.push [ @ip, @prefix ]
        @ip = File.new($2.strip)
        
        if ($1.nil?)
          @prefix = nil
        else
          @prefix = "!" + $1
        end
        
        line = @ip.gets

        # Expand tabs
        1 while line.gsub!(/\t+/) { ' ' * (8*$&.length - $`.length % 8)}  && $~ #`
        
      end
      
      yield @prefix, line
    end
  end
end

#################################################################
#
# Read in a block of code. Flag sections between \begin{hidden}
# and \begin{latex} appropriately
#

def readBlock
  io = NestedIO.new($stdin)
  chunk = []
  
  override = nil
  
  io.each do |prefix, $_|
    
    chomp
    
    break if /^\s*\\end{ruby}/
    
    prefix = override if override and !prefix
    
    if /^\s*\\begin{(.*?)}/
      override = "!" + ($1 == "hidden" ? "-" : "+")
    elsif /^\s*\\end{(hidden|latex)}/
      override = nil
    else
      if prefix
        $_[0,2] = "" if /^!(.)/
        chunk.push prefix + $_
      else
        chunk.push $_
      end
    end
  end
  chunk
end

#################################################################
#
# Build two arrays, one of lines to be written to
# latex, the other code to be executed by ruby

def partitionCode(chunk, options)
  code = []
  latex = []

  chunk.each do |line|
    flag = ""
    if line =~ /^!(.)/
      flag = $1
      line[0,2] = ""
    end
    
    case flag
    when ""
      code.push line
      latex.push line

    when "-"
      code.push line
      latex.push nil
      
    when "+"
      code.push ""
      latex.push line
      
    when /[0-9A-Z]/
      code.push line
      latex.push options[flag] ? line : nil
    end
  end
  
  return code, latex
end

#################################################################
#
# If we're putting the result of statements in comments, then
# we need to go through the actual code we execute and
# change
#     expr
# to
#     print "<<#{lineNum}>>#{(expr).inspect}\n"

def annotateCode(code, latex, doAssign)

  defining = 0

  for lineNum in 0...code.length
#     $stderr.print lineNum, ": ", line, "\n"
    
    line = code[lineNum].dup
    # Remove comments, but first look for the !sh! flag, which
    # tells us not to annotat this line

    skipThis = line =~ /#.*!sh!/

    line.strip!
    line.sub!(/#[^{@$].*$/, '')

    # We don't output values for code blocks and loops
    if line =~ /^\s*(def|class|module|while|if|for|case)\s/ or 
       line =~ / \{[^}]*$/ or
       (line =~ /\bdo\b/ and line !~ /\bdo\b.*\bend\b/)
      defining += 1  
    end

#    $stderr.puts "#{defining}: #{line}"

    if defining == 0            and
        not skipThis            and
        latex[lineNum]          and
        line !~ /^\s*include\s/ and
        line !~ /^\s*require\s/ and
        line !~ /^\s*$/         and
        (doAssign or line !~ /^.+\s+=\s+/)

      code[lineNum] = "print '<<#{lineNum}>>', (#{line}).inspect, \"\\n\""
    end

    defining -= 1 if line =~ /^(!-)?\s*\bend\b/ or 
      line =~ /^\}/ or (line =~ / \}/ and line !~ /\{.*\}/)

    lineNum += 1
  end
#  $stderr.print "Code: ", code.join("\n"), "\n"
end

#################################################################
#
# If we get an exception, it contains the line number in the
# code that was ru. However, if we chose not to show all that
# code in the latex output (using !-), then the line numbers
# will be wrong. This hack corrects that

def massageLineNumber(actual, code, latex)
  # look for the latex line number that corresponds to actual in
  # the code

  latexLine = 0
  actual.times {|i| latexLine += 1 if latex[i]}

  latexLine
end

#################################################################
#
# Run the given code and collect the output
#

def runCode(code, latex, errOk, rubyopts)
  lineNo = $.
#  $stderr.puts lineNo
  
  ruby = open("|#{Config::CONFIG['bindir']}/ruby -I #{$:.join(':')} #{rubyopts} 2>&1", "w+")
  ruby.print '$defout.sync=1; '
  ruby.puts code.join("\n")
  ruby.close_write


  result = ruby.readlines
  ruby.close

#  if  $? != 0
    if errOk
      # Move the errors to the end
      rnormal = []
      rerr = []
      for l in result
        l.sub!(%r{/ruby161}, '')
        if l =~ /-:(\d+)/
          actualLine = massageLineNumber($1.to_i, code, latex)
          rerr << l.sub(/-:\d+/, "prog.rb:#{actualLine}")
        else
          rnormal << l
        end
      end
      result = rnormal + rerr
    elsif $? != 0
      $stderr.print "\n\n#{$FILENAME}:#{$.} " +
      "Error in a ruby code fragment on line #{lineNo}\n"
      $stderr.print result
      exit($?)
    end
#  end
  result
end

#################################################################
#
# Remove leading whitespace from a set up lines
#
#  |  def fred
#  |    a = 1
#  |  end
# 
# ->
#
#  |def fred
#  |  a = 1
#  |end
#

def fixIndent(lines)
  indent = 9999

  for line in lines
    next if line.nil?
    if line =~ /^(\s*)\S/
      thisIndent = $1.length 
      indent = thisIndent if thisIndent < indent
    end
  end
  
  return if indent == 9999 or indent == 0

  pattern = Regexp.new('^' + ' '*indent)

  for i in 0...lines.size
    line = lines[i]
    next if line.nil?
    lines[i] = line.sub(pattern, '')
  end
    
end
#################################################################
#
# Escape latex sequences in the output
#

def escape(line, space)
# $stderr.print "FE: ", line
  s = line.sub(/\s+$/, '').
    gsub(/\\/, "\\bs\?C-q").
      gsub(/([_\${}&%#])/, '\\\\\1').
      gsub(/\?C-q/, "{}").
      gsub(/\^/, "\\up{}").
      gsub(/~/, "\\sd{}").
      gsub("<<", "<{}<").
      gsub(">>", ">{}>").
      gsub(",,", ",{},").
      gsub("`",  "\\bq{}").
      gsub(/ /, space)
# $stderr.print "-> ", s, "\n"
  s
end

#################################################################
#
# Given an array of latex lines and some output, write
# the two side-by-side
#

def formatAsComment(latex, output, showSpaces)

#  $stderr.puts "Latex:\n" + latex.join("\n")
#  $stderr.puts "Output:\n" + output.join("\n")

  comment = Array.new
  sp = showSpaces ? '\\char32{}' : '\\ '
  
  puts "\\begingroup\\VerbatimFont"
  puts "\\LogCodeRef{#$fileName}{#$codeCount}\\relax{}%"
  puts "\\begin{#$table_name}{\\codewidth}{@{}ll>{\\raggedright\\arraybackslash}X@{}}"

  
  lineNo = 0
  for op in output
    if op =~ /^<<(\d+)>>/
      lineNo = $1.to_i
      op.gsub!(/^<<\d+>>/, '')
    end
    if comment[lineNo]
      comment[lineNo] << '\\n' << op.chomp
    else
      comment[lineNo] = op.chomp
    end
  end
  
  # we now have latex and comments in parallel arrays
  
  res = ""
  for i in 0...latex.length
    next if latex[i].nil?
    codeLine = escape(latex[i].sub(/#.*!sh!/, ''), '\\ ')
    if comment[i].nil?
      print "\\multicolumn{3}{@{}l@{}}{\\CF{#{codeLine}}}"
    else
      print "\\CF{#{codeLine}} & $\\rightarrow$ & "
      print "\\CF{" + escape(comment[i], sp) + "}"
      end
    puts "\\\\"
  end
  
  puts "\\end{#$table_name}"
  puts"\\endgroup\n"
end

#################################################################
#
# Given an set of lines and output, write the lines and then
# the output
#

def formatWithOutput(latex, output, wrapoutput)

  puts "\\begin{alltt}\\LogCodeRef{#$fileName}{#$codeCount}\\relax{}"
      
  for line in latex
    next unless line
    l = line.sub(/#.*!sh!/, '')
    if line =~ /^\s*$/
      puts "\\end{alltt}"
      puts "\\vspace{-4pt}"
      puts "\\begin{alltt}"
    else
      puts escape(l, '\\ ')
    end
  end
      
  puts "\\end{alltt}"
      
      
  unless output.nil? || output.size == 0
    puts  "\\end{codefragment}"
    print "\\vspace{-.2\\baselineskip}" 
    puts  "\\emph{produces:}" 
    puts  "\\vspace{-.2\\baselineskip}\\begin{codefragment}\\begin{alltt}"
        
    for l in output
      opline = escape(l, wrapoutput ? ' ' : '\\ ')
      if wrapoutput && opline.length > 60
        opline.sub!(/(.{50}.*?) /) { "#$1\\\\" }
      end
      puts opline
    end
    puts "\\end{alltt}"
  end
end

#################################################################
#
# Handle a fragment of Ruby code.
#
    
def handleCodeBlock(optionString)
  options = Hash.new
  optionString.split(/\s*,\s*/).each do |o|
    if o =~ /rubyopts=(.*)/
      options['rubyopts'] = $1
    else
      options[o] = 1
    end
  end

  # Read in this code fragment
  chunk = readBlock

  if chunk.size == 0
    $stderr.print "\n\n#{ARGV} #$.: empty code block\n\n"
    return
  end

  # Write the whole block out to a file in the samples directory

  if $fileName
    File.open(File.join(SAMPLESDIR, "#$fileName:#$codeCount"), "w") do |f|
      chunk.each do |cline|
        cline = cline.dup
        if cline[0] == ?!
          flag = cline[1].chr
          case flag
          when '-'
            ;
          when '+'
            cline[0..1] = '#'
          when /[0-9A-Z]/
            if options[flag]
              cline[0..1] = ''
            else
              cline[1] = '-'
            end
          end
        end
        f.puts cline
      end
    end
  end

  # partition into the stuff to be executed and the stuff to be output
  code, latex = partitionCode(chunk, options)

  # If we're displaying the value of a statment next to that statement,
  # then we need to change the code to output the values with flags

  if options['comment'] or options['comment*']
    annotateCode(code, latex, options['comment*'])
  end

  # Run the code and collect the output

  if options['rubyopts'] != nil
    rubyopts=options['rubyopts']
  else
    rubyopts=""
  end

  output = []
  unless options['norun']
    output = runCode(code, latex, options['synerr'], rubyopts) 

    # seems crazy, but we want to catch syntax errors
    # even if there's no output
    output = [] if options['nooutput']
  end

  # The way we massage the output depends on the options

#  return if options['nooutput']

  # Rawoutput simply gets written out with no conversion

  if options['rawop']
    puts output
    return
  end


  # Otherwise we're doing some kind of formatted output

  fixIndent(latex)

  puts
  puts '\begin{codefragment}'

  if options['comment'] or options['comment*']
    formatAsComment(latex, output, options['showspace'])
  else
    formatWithOutput(latex, output, options['wrapoutput'])
  end
  
  puts '\end{codefragment}'
end


#################################################################
#
# Read in a syntax block and strip off leading whitespace such
# that the first non-whitespace column in the block is the new margin

def handleSyntax
  lines = []
  while $stdin.gets
    chomp
    if /\\end{syntax}/
      last = $_
      break
    end
    lines << $_
  end

  fixIndent(lines)

  puts lines

  puts last
end

#################################################################
#
#
# Read in the file, copying to output until we reach \begin{ruby}.
#

$table_name = "tabularx"

if ARGV[0] == "-for-xml"
  $table_name = "tabularruby"
  ARGV.shift
end
    
if (ARGV.size == 2) and (ARGV[0] == "-name")
  $fileName = File.basename(ARGV[1]).gsub(/_/, '').sub(/\.tip/, '')
end

$codeCount = 0

if !test(?d, SAMPLESDIR)
  Dir.mkdir(SAMPLESDIR)
end

while $stdin.gets
  if /^\s*\\begin{ruby}(?:\[(.*?)\])?/
    $codeCount += 1
    handleCodeBlock($1 || "")
  elsif /^\s*\\begin{syntax}/
    print
    handleSyntax
  else
    print
  end
end



