=begin
= rd2html-lib.rb
=end

require "rd/rdvisitor"

module RD
  class RD2HTMLVisitor < RDVisitor
    include AutoLabel
    include MethodParse
    
    # must-have constants
    OUTPUT_SUFFIX = "html"
    INCLUDE_SUFFIX = ["html"]
    
    METACHAR = { "<" => "&lt;", ">" => "&gt;", "&" => "&amp;" }
    
    attr(:include_suffix, true)
    attr(:css, true)
    attr(:charset, true)
    alias charcode charset
    alias charcode= charset=
    attr(:title, true)
    attr(:html_link_rel, nil)
    attr(:html_link_rev, nil)
    
    def initialize
      @include_suffix = INCLUDE_SUFFIX.clone
      @css = nil
      @charset = nil
      @title = ""
      @html_link_rel = {}
      @html_link_rev = {}

      @footnotes = []
      @index = {}
      super
    end
    
    def visit(tree)
      prepare_labels(tree, "label:")
      super(tree)
    end

    def apply_to_DocumentElement(element, content)
      content = content.join("")
      index = make_index()
      foottext = make_foottext()
      title = @title || @filename || ARGF.filename || "Untitled"
      ret = %Q[<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0 Transitional//EN">\n]
      ret << %Q[<HTML>\n<HEAD>\n<TITLE>#{title}</TITLE>\n]
      if @charset
	ret << %Q[<META HTTP-EQUIV="Content-type"] +
	  %Q[content="text/html; charset=#{@charset}">\n]
      end
      if @css
	ret << %Q[<LINK HREF="#{@css}" TYPE="text/css" REL="stylesheet">\n]
      end

      @html_link_rel.each do |rel, href|
	ret << %Q[<LINK REL="#{rel}" HREF="#{href}">\n]
      end
      @html_link_rev.each do |rev, href|
	ret << %Q[<LINK REV="#{rev}" HREF="#{href}">\n]
      end

      ret << %Q[</HEAD>\n]
#      ret << %Q[<BODY>\n#{content}#{index}#{foottext}\n</BODY>\n</HTML>]
      ret << %Q[<BODY>\n#{content}#{foottext}\n</BODY>\n</HTML>\n]
      ret
    end
    
    def apply_to_Headline(element, title)
      anchor = refer(element)
      %Q[<H#{element.level}><A NAME="#{anchor}">#{title}</A>] +
      %Q[</H#{element.level}><!-- RDLabel: "#{element.label}" -->\n]
    end

    # RDVisitor#apply_to_Include 

    def apply_to_TextBlock(element, content)
      content = content.join("")
      %Q[<P>\n#{content}</P>\n]
    end
  
    def apply_to_Verbatim(element)
      content = []
      element.each_line do |i|
	content.push(apply_to_String(i))
      end
      %Q[<PRE>\n#{content.join("")}</PRE>\n]
    end
  
    def apply_to_ItemList(element, items)
      %Q[<UL>\n#{items.join("")}</UL>\n]
    end
  
    def apply_to_EnumList(element, items)
      %Q[<OL>\n#{items.join("")}</OL>\n]
    end
    
    def apply_to_DescList(element, items)
      %Q[<DL>\n#{items.join("\n")}\n</DL>\n]
    end

    def apply_to_MethodList(element, items)
      %Q[<DL>\n#{items.join("\n")}\n</DL>\n]
    end
    
    def apply_to_ItemListItem(element, content)
      %Q[<LI>#{content.join("\n")}]
    end
    
    def apply_to_EnumListItem(element, content)
      %Q[<LI>#{content.join("\n")}]
    end
    
    def apply_to_DescListItem(element, term, description)
      anchor = refer(element)
      if description.empty?
	%Q[<DT><A NAME="#{anchor}">#{term}</A>] +
	%Q[<!-- RDLabel: "#{element.label}" -->]
      else
        %Q[<DT><A NAME="#{anchor}">#{term}</A>] +
        %Q[<!-- RDLabel: "#{element.label}" -->\n] +
        %Q[<DD>\n#{description.join("\n")}</DD>]
      end
    end

    def apply_to_MethodListItem(element, term, description)
      term = parse_method(term)  # maybe: term -> element.term
      anchor = refer(element)
      if description.empty?
	%Q[<DT><A NAME="#{anchor}"><CODE>#{term}] +
        %Q[</CODE></A><!-- RDLabel: "#{element.label}" -->]
      else
        %Q[<DT><A NAME="#{anchor}"><CODE>#{term}] +
	%Q[</CODE></A><!-- RDLabel: "#{element.label}" -->\n] +
	%Q[<DD>\n#{description.join("\n")}</DD>]
      end
    end
  
    def apply_to_StringElement(element)
      apply_to_String(element.content)
    end
    
    def apply_to_Emphasis(element, content)
      %Q[<EM>#{content.join("")}</EM>]
    end
  
    def apply_to_Code(element, content)
      %Q[<CODE>#{content.join("")}</CODE>]
    end
  
    def apply_to_Var(element, content)
      %Q[<VAR>#{content.join("")}</VAR>]
    end
  
    def apply_to_Keyboard(element, content)
      %Q[<KBD>#{content.join("")}</KBD>]
    end
  
    def apply_to_Index(element, content)
      tmp = []
      element.each do |i|
	tmp.push(i) if i.is_a?(String)
      end
      key = meta_char_escape(tmp.join(""))
      if @index.has_key?(key)
	# warning?
	%Q[<!-- Index, but conflict -->#{content.join("")}<!-- Index end -->]
      else
	num = @index[key] = @index.size
	%Q[<A NAME="index:#{num}">#{content.join("")}</A>]
      end
    end

    def apply_to_Reference(element, content)
      case element.label
      when Reference::URL
	apply_to_RefToURL(element, content)
      when Reference::RDLabel
	if element.label.filename
	  apply_to_RefToOtherFile(element, content)
	else
	  apply_to_RefToElement(element, content)
	end
      end
    end

    def apply_to_RefToElement(element, content)
      content = content.join("")
      if anchor = refer(element)
	content = content.sub(/^function#/, "")
	%Q[<A HREF="\##{anchor}">#{content}</A>]
      else
	# warning?
	%Q[<!-- Reference, RDLabel "#{element.to_label}" doesn't exist -->] +
	%Q[<I>#{content}</I><!-- Reference end -->] #' 
      end
    end

    def apply_to_RefToOtherFile(element, content)
      content = content.join("")
      filename = element.label.filename.sub(/\.(?:rd|rb)$/, ".#{OUTPUT_SUFFIX}")
      %Q[<A HREF="#{filename}">#{content}</A>]
    end
  
    def apply_to_RefToURL(element, content)
      %Q[<A HREF="#{element.label.url}">#{content.join("")}</A>]
    end
    
    def apply_to_Footnote(element, content)
      @footnotes.push(content)
      num = @footnotes.size
      %Q[<A NAME="footnote:#{num}" HREF="#foottext:#{num}"><SUP><SMALL>] +
      %Q[*#{num}</SMALL></SUP></A>]
    end
    
    def apply_to_Verb(element)
      content = apply_to_String(element.content)
      %Q[#{content}]
    end
    
    def apply_to_String(element)
      meta_char_escape(element)
    end
    
    def parse_method(method)
      klass, kind, method, args = MethodParse.analize_method(method)
      
      if kind == :function
	klass = kind = nil
      else
	kind = MethodParse.kind2str(kind)
      end
      
      args.gsub!(/\w+/, '<var>\\&</var>')

      case method
      when "[]"
	args.strip!
	args.sub!(/^\((.*)\)$/, '\\1')
	"#{klass}#{kind}[#{args}]"
      when "[]="
	args.strip!
	args.sub!(/^\((.*)\)$/, '\\1')
	args, val = /^(.*),([^,]*)$/.match(args)[1,2]
	args.strip!
	val.strip!

	"#{klass}#{kind}[#{args}] = #{val}"
      else
	"#{klass}#{kind}#{method}#{args}"
      end
    end
    private :parse_method

    def meta_char_escape(str)
      str.gsub(/[<>&]/) {
	METACHAR[$&]
      }
    end
    private :meta_char_escape
    
    def make_index
      unless @index.empty?
	tmp = []
	@index.keys.sort.each do |entry|
	  num = @index[entry]
	  tmp.push(%[<LI><A HREF="#index:#{num}">#{entry}</A>])
	end
	tmp = tmp.join("\n")
	%[\n<HR>\n<UL>\n#{tmp}\n</UL>]
      end
    end
    private :make_index
    
    def make_foottext
      if @footnotes[0]
	ret = "\n<HR>\n<P>\n"
	num = 0
	while i = @footnotes.shift
	  num += 1
	  ret << %Q[<A NAME="foottext:#{num}" HREF="#footnote:#{num}"><SUP>] +
	  %Q[<SMALL>*#{num}</SMALL></SUP></A>] +
	  %Q[<SMALL>#{i}</SMALL><BR>\n]
	end
	ret << "</P>"
	ret
      else
	""
      end
    end
    private :make_foottext
    
  end # RD2HTMLVisitor
end # RD

$Visitor_Class = RD::RD2HTMLVisitor
$RD2_Sub_OptionParser = "rd/rd2html-opt"

=begin
== script info.
 RD to HTML translate library for rdfmt.rb
 $Id: rd2html-lib.rb,v 1.1 2001/01/23 22:10:48 dave Exp $

== changes
:0.6.3
  * HTML TITLE, LINK options.
:0.6.1
  * <BLOCKQUOTE> for Verbatim is removed.
:0.6.0
  * MethodList. thanks. > Arai
  * method index (-> rd/rd2rmi-lib.rb)
  * changes about changes of (({Reference}))
  * <VERB>?? what is it? W3C hasn't defined such a element! ;-p
  * temporary disable Index.
:0.5.4
  * open (({@include_suffix}))
:0.5.3
  * Include.
  * empty description of DescListItem.
  * BLOCKQUOTE are also used for Verbatim.
:0.5.2
  * into RD module.
  * label processing is now into RDVisitor.
:0.5.1
  * when target isn't finded, Reference come to be translate into "<I>...</I>".
:0.5.0
  * reborned for new RD.
:0.2.2
  * bug in term part of DescListItem.
:0.2.1 (1999/08/15)
  * Index is now available.(EXPERIMENTAL)
:0.2.0 (1999/08/15)
  * insert "\n" into the end of line of "<!DOCTYPE ..."
  * Bug about HTML meta-char fixed.
:0.1.1 (1999/08/13)
  * patch for HTML format mistake.(rd2html-lib.rb) Thanks > gotoken
:0.1.0 (1999/08/12)
  * first release version
=end
