=begin
= rd2html-lib.rb
=end

require "rd/rdvisitor"

module RD
  class RD2LaTeXVisitor < RDVisitor
    include AutoLabel
    include MethodParse
    
    # must-have constants
    OUTPUT_SUFFIX = "tex"
    INCLUDE_SUFFIX = ["tex"]
    
    METACHAR = { 
      "_" => "\\_", 
      "&" => "\\&",
      "#" => "\\#",
      "\\" => "\\bs{}",
      "^" => "\\up{}",
      "~" => "\\sd{}",
      "%" => "\\%",
      "{" => "\\{",
      "}" => "\\}",
      "|" => "$|$",
      ">" => "$>$",
      "<" => "$<$",
    }
    
    attr(:include_suffix, true)
    attr(:css, true)
    attr(:charset, true)
    alias charcode charset
    alias charcode= charset=
    
    def initialize
      @include_suffix = INCLUDE_SUFFIX.clone
      @footnotes = []
      @index = {}
      super
    end
    
    def visit(tree)
      prepare_labels(tree, "label:")
      super(tree)
    end

    def apply_to_DocumentElement(element, content)
      content = content.join("")
      index = make_index()
      title = @filename || ARGF.filename || "Untitled"
      ret = %Q[\\documentclass{article}\n]
      ret << %Q[\\usepackage{url}\n]
      ret << %Q[\\setlength{\\topmargin}{0pt}\n]
      ret << %Q[\\setlength{\\headsep}{0pt}\n]
      ret << %Q[\\setlength{\\textheight}{800pt}\n]
      ret << %Q[\\newcommand{\\bs}{\\symbol{\'134}}\n]
      ret << %Q[\\newcommand{\\up}{\\symbol{\'136}}\n]
      ret << %Q[\\newcommand{\\sd}{\\symbol{\'176}}\n]
      ret << %Q[\\begin{document}\n]
      ret << %Q[#{content}\n]
      ret << %Q[\\end{document}\n]
      ret
    end

    HEADINGS = [ '\\section', '\\subsection',
      '\\subsubsection', '\\paragraph' ];

    def apply_to_Headline(element, title)
      anchor = refer(element)

      index = element.level
      index = HEADINGS.size if index > HEADINGS.size

      %Q[\n#{HEADINGS[index-1]}*{#{title.join("").strip}}\\label{#{anchor}}\n\n]
    end

    # RDVisitor#apply_to_Include 

    def apply_to_TextBlock(element, content)
      content = content.join("")
      %Q[\n#{content}\n]
    end
  
    def apply_to_Verbatim(element)
      content = []
      element.each_line do |i|
	content.push(i)
#	content.push(apply_to_String(i))
      end
      %Q[\\begin{verbatim}\n#{content.join("")}\\end{verbatim}\n]
    end
  
    def apply_to_ItemList(element, items)
      %Q[\\begin{itemize}\n#{items.join("")}\n\\end{itemize}\n]
    end
  
    def apply_to_EnumList(element, items)
      %Q[\\begin{enumerate}\n#{items.join("")}\n\\end{enumerate}\n]
    end
    
    def apply_to_DescList(element, items)
      %Q[\\begin{description}\n#{items.join("")}\n\\end{description}\n]
    end

    def apply_to_MethodList(element, items)
      %Q[\\begin{description}\n#{items.join("")}\n\\end{description}\n]
    end
    
    def apply_to_ItemListItem(element, content)
      %Q[\\item #{content.join("\n")}]
    end
    
    def apply_to_EnumListItem(element, content)
      %Q[\\item #{content.join("\n")}]
    end
    
    def apply_to_DescListItem(element, term, description)
      anchor = refer(element)
      %Q[\\item[#{term.join("").strip}]\\label{#{anchor}}\\mbox{}\\\\] +
      (description.empty? ? "" : (description.join("\n") + "\n"))
    end

    def apply_to_MethodListItem(element, term, description)
      term = parse_method(term)  # maybe: term -> element.term
      anchor = refer(element)
      %Q[\\item[\\texttt{#{term}}]\\label{#{anchor}}\\mbox{}\\\\] +
      (description.empty? ? "" : (description.join("\n") + "\n"))
    end
  
    def apply_to_StringElement(element)
      apply_to_String(element.content)
    end
    
    def apply_to_Emphasis(element, content)
      %Q[\\emph{#{content.join("")}}]
    end
  
    def apply_to_Code(element, content)
      %Q[\\texttt{#{content.join("")}}]
    end
  
    def apply_to_Var(element, content)
      %Q[\\textsl{#{content.join("")}}]
    end
  
    def apply_to_Keyboard(element, content)
      %Q[\\texttt{#{content.join("")}}]
    end
  
    def apply_to_Index(element, content)
      tmp = []
      element.each do |i|
	tmp.push(i) if i.is_a?(String)
      end
      key = meta_char_escape(tmp.join(""))
      if @index.has_key?(key)
	stderr.print "Duplicate index entry: #{content.join("")}\n"
        ""
      else
	num = @index[key] = @index.size
        ind = "index:" + num.to_s
	%Q[#{content.join("")}label{#{ind}}]
      end
    end

    def apply_to_Reference(element, content)
      case element.label
      when Reference::URL
	apply_to_RefToURL(element, content)
      when Reference::RDLabel
	if element.label.filename
	  apply_to_RefToOtherFile(element, content)
	else
	  apply_to_RefToElement(element, content)
	end
      end
    end

    def apply_to_RefToElement(element, content)
      content = content.join("")
      if anchor = refer(element)
	content = content.sub(/^function#/, "")
#	%Q[#{content} (\\ref{#{anchor}} on page \\pageref{#{anchor}})]
	%Q[#{content}]
      else
	$stderr.print "Reference to undefined label: #{element.to_label}\n";
	"\\emph{#{content}}"
      end
    end

    def apply_to_RefToOtherFile(element, content)
      content = content.join("")
      filename = element.label.filename.sub(/\.(?:rd|rb)$/, ".#{OUTPUT_SUFFIX}")
      %Q[#{content} (see file #{filename})]
    end
  
    def apply_to_RefToURL(element, content)
#      %Q[#{content.join("")}] +
      txt = element.label.url.split(/#/).collect {|i| "\\url{#{i}}" }
      txt = txt.join("\\texttt{\\#}")
      
      %Q[(see #{txt})]
    end
    
    def apply_to_Footnote(element, content)
      %Q[\\footnote{#{content}}]
    end
    
    def apply_to_Verb(element)
      content = apply_to_String(element.content)
      %Q[#{content}]
    end
    
    def apply_to_String(element)
      meta_char_escape(element)
    end
    
    def parse_method(method)
      klass, kind, method, args = MethodParse.analize_method(method)
      
      if kind == :function
	klass = kind = nil
      else
	kind = MethodParse.kind2str(kind)
      end
      
      args.gsub!(/\w+/, '\\emph{\\&}')

      case method
      when "[]"
	args.strip!
	args.sub!(/^\((.*)\)$/, '\\1')
	"#{klass}#{kind}[#{args}]"
      when "[]="
	args.strip!
	args.sub!(/^\((.*)\)$/, '\\1')
	args, val = /^(.*),([^,]*)$/.match(args)[1,2]
	args.strip!
	val.strip!

	"#{klass}#{kind}[#{args}] = #{val}"
      else
	"#{klass}#{kind}#{method}#{args}"
      end
    end
    private :parse_method

    def meta_char_escape(str)
      str.gsub(/[&_#\\^~%{}|<>]/) {
	METACHAR[$&]
      }
    end
    private :meta_char_escape
    
    def make_index
      unless @index.empty?
	tmp = []
	@index.keys.sort.each do |entry|
	  num = @index[entry]
	  tmp.push(%[<LI><A HREF="#index:#{num}">#{entry}</A>])
	end
	tmp = tmp.join("\n")
	%[\n<HR>\n<UL>\n#{tmp}\n</UL>]
      end
    end
    private :make_index
    
  end # RD2HTMLVisitor
end # RD

$Visitor_Class = RD::RD2LaTeXVisitor

=begin
== script info.
 $Id: rd2latex-lib.rb,v 1.1 2001/01/23 22:10:48 dave Exp $

== changes
=end
