=begin
= What is rdfmt.rb
Basic stracture of RDtool is defined in rdfmt.rb. This "basic stracture"
is mainly classes of nodes of RD document tree, "Element". 

== Classes in rdfmt.rb
following classes are defined in rdfmt.rb.
*RDElement
  *BlockElement
    *TextBlock
    *Verbatim
    *Headline
    *Include
    *List
      *ItemList
      *EnumList
      *DescList
      *MethodList
    *ListItem
      *ItemListItem
      *EnumListItem
      *DescListItem
      *MethodListItem
  *InlineElement
    *Emphasis
    *Code
    *Var
    *Keyboard
    *Index
    *Reference
    *Footnote
    *Verb
*RDTree
=end
require "rd/rdblockparser.tab"
require "rd/filter"

module RD
  class RDElement
    include Enumerable
    attr(:parent, true)
    
    def tree
      if @parent.is_a?(RDTree)
	@parent
      else
	@parent.tree
      end
    end
    
    private
    def tell_parent_to(id)
      eval(id.id2name).each do |i|
	i.parent = self
      end
    end
  end
  
  class DocumentElement < RDElement
    attr(:blocks, true)
    def initialize(blocks)
      @blocks = blocks          # Array of Block
      tell_parent_to(:@blocks)
    end
    
    def accept(visitor)
      visitor.visit_DocumentElement(self)
    end
    
    def each_element(&block)
      yield(self)
      @blocks.each do |i|
	i.each_element(&block)
      end
    end
    alias each each_element
    
    def each_block
      @blocks.each do |i|
	yield(i)
      end
    end
    
    def children
      @blocks
    end
    def children=(value)
      @blocks = value
    end
  end
  
  class BlockElement < RDElement
  end
  
  class Headline < BlockElement
    attr(:level, true)
    attr(:title, true)
    def initialize(level, title)
      @level = level         # Integer
      @title = title         # Array of elements
      #    @title.parent = self
    end
    
    def accept(visitor)
      visitor.visit_Headline(self)
    end
    
    def each_element(&block)
      yield(self)
      @title.each do |i|
	i.each_element(&block)
      end
    end
    alias each each_element
    
    def to_label
      ret = ""
      @title.each do |i|
	ret << i.to_label
      end
      ret
    end
    alias label to_label
    
    def Headline.mark_to_level(mark)
      case mark
      when "="
	1
      when "=="
	2
      when "==="
	3
      when "===="
	4
      when "+"
	5
      when "++"
	6
      end
    end
  end
  
  class Include < BlockElement
    attr(:filename, true)
    
    def initialize(filename)
      @filename = filename
    end
    
    def accept(visitor)
      visitor.visit_Include(self)
    end
    
    def each_element(&block)
      yield(self)
    end
    alias each each_element
  end
  
  class TextBlock < BlockElement
    attr(:content, true)
    def initialize(content)
      @content = content         # Array of Inline and String
      tell_parent_to(:@content)
    end
    
    def accept(visitor)
      visitor.visit_TextBlock(self)
    end
    
    def each_element(&block)
      yield(self)
      @content.each do |i|
	i.each_element(&block)
      end
    end
    alias each each_element
    
    def each_child
      @content.each do |i|
	yield(i)
      end
    end
    
    def children
      @content
    end
    def children=(value)
      @content = value
    end
  end
  
  class Verbatim < BlockElement
    attr(:content, true)
    def initialize(content)
      @content = content  # Array of String
    end
    
    def accept(visitor)
      visitor.visit_Verbatim(self)
    end
    
    def each_element(&block)
      yield(self)
#      @content.each do |i|
#	yield(i)
#      end
    end
    alias each each_element
    
    def each_line
      @content.each do |i|
	yield(i)
      end
    end
    
    def children
      @content
    end
    def children=(value)
      @content = value
    end
  end

  class List < BlockElement
    attr(:items, true)
    def initialize(items)
      @items = items          # Array of ListItem
      tell_parent_to(:@items)
    end
    
    def each_element(&block)
      yield(self)
      @items.each do |i|
	i.each_element(&block)
      end
    end
    alias each each_element
    
    def each_item
      @items.each do |i|
	yield(i)
      end
    end
    alias each_child each_item
  
    def children
      @items
    end
    def children=(value)
      @items = value
    end
  end
  
  class ListItem < BlockElement
    attr(:content, true)
    def initialize(content)
      @content = content        # Array of BlockElement
      tell_parent_to(:@content)
    end
    
    def each_element(&block)
      yield(self)
      @content.each do |i|
	i.each_element(&block)
      end
    end
    alias each each_element
    
    def each_block
      @content.each do |i|
	yield(i)
      end
    end
    alias each_child each_block
    
    def children
      @content
    end
    def children=(value)
      @content = value
    end
    alias blocks children
    alias blocks= children=
  end
  
  class ItemList < List
    def accept(visitor)
      visitor.visit_ItemList(self)
    end
  end
  class ItemListItem < ListItem
    def accept(visitor)
      visitor.visit_ItemListItem(self)
    end
  end
  
  class EnumList < List
    def accept(visitor)
      visitor.visit_EnumList(self)
    end
  end
  class EnumListItem < ListItem
    def accept(visitor)
      visitor.visit_EnumListItem(self)
    end
  end
  
  class DescList < List
    def accept(visitor)
      visitor.visit_DescList(self)
    end
  end
  
  class DescListItem < ListItem
    attr(:term, true)
    attr(:description, true)
    
    def initialize(term, description)
      @term = term                # Array of Inline
      @description = description  # Array of Block
      tell_parent_to(:@term)
      tell_parent_to(:@description)
    end
    
    def accept(visitor)
      visitor.visit_DescListItem(self)
    end
    
    def each_element(&block)
      yield(self)
      @term.each do |i|
	i.each_element(&block)
      end
      @description.each do |i|
	i.each_element(&block)
      end
    end
    alias each each_element
    
    def each_block_in_description
      @description.each do |i|
	yield(i)
      end
    end
    alias each_block each_block_in_description
    alias each_child each_block_in_description
    
    def children
      @description
    end
    def children=(value)
      @description = value
    end
    alias blocks children
    alias blocks= children=
      
    def to_label
      ret = ""
      @term.each do |i|
	ret << i.to_label
      end
      ret
    end
    alias label to_label
  end
  
  class MethodList < List
    def accept(visitor)
      visitor.visit_MethodList(self)
    end
  end
  
  class MethodListItem < ListItem
    attr(:term, true)
    attr(:label, true)
    alias to_label label
    attr(:description, true)
    
    def initialize(term, description)
      @term = term                # String
      if /[^{( \t]+/ =~ @term
	@label = $&
      else
	@label = @term.dup
      end
      @description = description  # Array of Block
#      tell_parent_to(:@term)
      tell_parent_to(:@description)
    end
    
    def accept(visitor)
      visitor.visit_MethodListItem(self)
    end
    
    def each_element(&block)
      yield(self)
      yield(@term)
      @description.each do |i|
	i.each_element(&block)
      end
    end
    alias each each_element
    
    def each_block_in_description
      @description.each do |i|
	yield(i)
      end
    end
    alias each_block each_block_in_description
    alias each_child each_block_in_description
    
    def children
      @description
    end
    def children=(value)
      @description = value
    end
    alias blocks children
    alias blocks= children=
  end

  
  class InlineElement < RDElement
    attr(:content, true)
    def initialize(content)
      @content = content         # Array of Inline and String
      tell_parent_to(:@content)
    end
    
    def each_element(&block)
      yield(self)
      @content.each do |i|
	i.each_element(&block)
      end
    end
    alias each each_element
    
    def each_child
      @content.each do |i|
	yield(i)
      end
    end
    
    def children
      @content
    end
    def children=(value)
      @content = value
    end
    
    def to_label
      ret = ""
      @content.each do |i|
	ret << i.to_label
      end
      ret
    end
  end
  
  class StringElement < InlineElement
    attr(:content, true)
    attr(:parent, true)
    
    def initialize(content)
      @content = content
      @parent = nil
    end
    
    def to_label
      @content.strip
    end
    
    def each_element
      yield(self)
    end
    alias each each_element
    
    def accept(visitor)
      visitor.visit_StringElement(self)
    end
  end
  
  class Emphasis < InlineElement
    def accept(visitor)
      visitor.visit_Emphasis(self)
    end
  end
  
  class Code < InlineElement
    def accept(visitor)
      visitor.visit_Code(self)
    end
  end
  
  class Var < InlineElement
    def accept(visitor)
      visitor.visit_Var(self)
    end
  end
  
  class Keyboard < InlineElement
    def accept(visitor)
      visitor.visit_Keyboard(self)
    end
  end
  
  class Index < InlineElement
    def accept(visitor)
      visitor.visit_Index(self)
    end
  end
  
  class Reference < InlineElement
    attr(:label, true)   # Reference::Label
    attr(:content, true)
    
    def initialize(label, substitute = nil)
      @label = label
      if substitute
	@content = substitute
      else
	if label.type == URL
	  @content = [StringElement.new("<URL:#{label.url}>")]
	else
	  @content = label.element_label
	end
      end
      tell_parent_to(:@content)
    end

    def accept(visitor)
      visitor.visit_Reference(self)
    end
    
    def to_label
      if @label.type == RDLabel
	@label.to_label
      else
	nil
      end
    end
    # Label for Reference 
    class Label
    end

    class URL < Label
      attr(:url, true)
      
      def initialize(url)
	@url = url
      end
    end

    class RDLabel < Label
      attr(:element_label, true)
      attr(:filename, true)
      
      def initialize(element_label, filename = nil)
	@filename = filename
	@element_label = element_label
      end
      
      def to_label
	ret = ""
	@element_label.each do |i|
	  ret << i.to_label
	end
	ret
      end
    end
  end

=begin  ###
  class RefToElement < Reference
    def accept(visitor)
      visitor.visit_RefToElement(self)
    end
    # MUST CHANGE!!
    def initialize(content)
      @content = content
      @parent = nil
    end
    
    def to_label
      @content.strip
    end
    
    def each_element
      yield(self)
    end
    alias each each_element
  end
  
  class RefToURL < Reference
    def accept(visitor)
      visitor.visit_RefToURL(self)
    end
    
    def initialize(content)
      @content = content
      @parent = nil
    end
    
    def to_label
      @content.strip
    end
    
    def each_element
      yield(self)
    end
    alias each each_element
  end
=end ###

  class Footnote < InlineElement
    def accept(visitor)
      visitor.visit_Footnote(self)
    end
  end
  
  class Verb < InlineElement
    def accept(visitor)
      visitor.visit_Verb(self)
    end
    
    def initialize(content)
      @content = content
      @parent = nil
    end
    
    def to_label
      @content.strip
    end
    
    def each_element
      yield(self)
    end
    alias each each_element
  end

=begin
=== RDTree
RDTree defines interfaces for client. RDTree has a tree of RDElements.
A client makes/access/translates RD document tree throught RDTree.

=end

  class RDTree
    include Enumerable
    
    TMP_DIR = "/tmp"
    
    attr(:root, true)
    attr(:include_path, true)
    attr(:filter, true)
    attr(:tmp_dir, true)
    
    def initialize(src, path = [], do_parse = true)
      @src = src
      @include_path = path
      @filter = Hash.new()
      @tmp_dir = TMP_DIR
      if do_parse
	parse
      end
    end
    
    def parse
      parser = RDParser.new
      @root = parser.parse(@src.to_a, self)
      @root.parent = self
    end

    def accept(visitor)
      @root.accept(visitor)
    end
    
    def each_element(&block)
      @root.each(&block)
    end
    alias each each_element
    
    def RDTree.new_from_rdo(*rdos) # rdos: IOs
      tree = RDTree.new("", [], nil)
      tree_content = []
      rdos.each do |i|
	subtree = Marshal.load(i)
	tree_content.concat(subtree.root.blocks)
      end
      tree.root = DocumentElement.new(tree_content)
      tree
    end
  end
end

=begin
== script info.
 a library to make a tree of document structure
 from RD (Ruby Document).
 $Id: rdfmt.rb,v 1.1 2001/01/23 22:10:48 dave Exp $

== changes
:0.6.0
  * MethodList. thanks. > Arai
  * (({RDTree#preprocess})) -> (({RDParser#next_token}))
  * (({RDTree.new_from_rdo}))
  * (({RefToElement})) and (({RefToURL})) disappeared. by now, (({Reference}))
    represent both type of reference. on the other hand, (({Reference::Label}))
    and its subclass are defined.
:0.5.7
  * bug fix. thanks. > Nakada
:0.5.4
  * filter.
:0.5.3
  * Include other RD files.
:0.5.2
  * into RD module.
  * include Enumerable.
:0.5.0
  * reborened for new RD
:0.2.2
  * bug in parce of Verbatim is fixed.
  * bug in parce of term part of DescListItem is fixed.
:0.2.0 (1999/08/15)
  * RD class disappeared.
  * rdfmt.rb is now no more than basic part of RDtool
  * some little bug fixed.
=end
