#include "stdafx.h"
#include "openttd.h"
#include "debug.h"
#include "strings.h"
#include "table/strings.h"
#include "window.h"
#include "gui.h"
#include "station.h"
#include "gfx.h"
#include "player.h"
#include "town.h"
#include "command.h"
#include "engine.h"
#include "depot.h"

extern int _traininfo_vehicle_pitch;

static void StationsWndShowStationRating(int x, int y, int type, uint acceptance, int rating)
{
	static const byte _rating_colors[NUM_CARGO] = {152,32,15,174,208,194,191,55,184,10,191,48};
	int color = _rating_colors[type];
	uint w;

	if (acceptance > 575)
		acceptance = 575;

	acceptance = (acceptance + 7) >> 3;

	/* draw cargo */
	if ( (w=acceptance>>3) != 0) {
		GfxFillRect(x, y, x+w-1, y+6, color);
		x += w;
	}

	if ( (w=acceptance&7) != 0) {
		if (w==7) w--;
		GfxFillRect(x, y+(w-1), x, y+6, color);
	}

	x -= (acceptance>>3);

	DrawString(x+1, y, _cargoc.names_short[type], 0x10);

	/* draw green/red ratings bar */
	GfxFillRect(x+1, y+8, x+7, y+8, 0xB8);

	rating = (rating >> 5);

	if (rating != 0) {
		GfxFillRect(x+1, y+8, x+rating, y+8, 0xD0);
	}
}

static uint16 _num_station_sort[MAX_PLAYERS];

static char _bufcache[64];
static uint16 _last_station_idx;

static int CDECL StationNameSorter(const void *a, const void *b)
{
	char buf1[64];
	Station *st;
	const SortStruct *cmp1 = (const SortStruct*)a;
	const SortStruct *cmp2 = (const SortStruct*)b;

	st = GetStation(cmp1->index);
	SetDParam(0, st->town->townnametype);
	SetDParam(1, st->town->townnameparts);
	GetString(buf1, st->string_id);

	if ( cmp2->index != _last_station_idx) {
		_last_station_idx = cmp2->index;
		st = GetStation(cmp2->index);
		SetDParam(0, st->town->townnametype);
		SetDParam(1, st->town->townnameparts);
		GetString(_bufcache, st->string_id);
	}

	return strcmp(buf1, _bufcache);	// sort by name
}

static void GlobalSortStationList(void)
{
	const Station *st;
	uint32 n = 0;
	uint16 *i;

	// reset #-of stations to 0 because ++ is used for value-assignment
	memset(_num_station_sort, 0, sizeof(_num_station_sort));

	/* Create array for sorting */
	_station_sort = realloc(_station_sort, GetStationPoolSize() * sizeof(_station_sort[0]));
	if (_station_sort == NULL)
		error("Could not allocate memory for the station-sorting-list");

	FOR_ALL_STATIONS(st) {
		if(st->xy && st->owner != OWNER_NONE) {
			_station_sort[n].index = st->index;
			_station_sort[n++].owner = st->owner;
			_num_station_sort[st->owner]++;	// add number of stations of player
		}
	}

	// create cumulative station-ownership
	// stations are stored as a cummulative index, eg 25, 41, 43. This means
	// Player0: 25; Player1: (41-25) 16; Player2: (43-41) 2
	for (i = &_num_station_sort[1]; i != endof(_num_station_sort); i++) {*i += *(i-1);}

	qsort(_station_sort, n, sizeof(_station_sort[0]), GeneralOwnerSorter); // sort by owner

	// since indexes are messed up after adding/removing a station, mark all lists dirty
	memset(_station_sort_dirty, true, sizeof(_station_sort_dirty));
	_global_station_sort_dirty = false;

	DEBUG(misc, 1) ("Resorting global station list...");
}

static void MakeSortedStationList(byte owner)
{
	SortStruct *firstelement;
	uint32 n = 0;

	if (owner == 0) { // first element starts at 0th element and has n elements as described above
		firstelement =	&_station_sort[0];
		n =							_num_station_sort[0];
	}	else { // nth element starts at the end of the previous one, and has n elements as described above
		firstelement =	&_station_sort[_num_station_sort[owner-1]];
		n =							_num_station_sort[owner] - _num_station_sort[owner-1];
	}

	_last_station_idx = 0; // used for "cache" in namesorting
	qsort(firstelement, n, sizeof(_station_sort[0]), StationNameSorter); // sort by name

	_station_sort_dirty[owner] = false;

	DEBUG(misc, 1) ("Resorting Stations list player %d...", owner+1);
}

static void PlayerStationsWndProc(Window *w, WindowEvent *e)
{
	switch(e->event) {
	case WE_PAINT: {
		uint32 i;
		const byte window_number = (byte)w->window_number;

		// resort station window if stations have been added/removed
		if (_global_station_sort_dirty)
			GlobalSortStationList();

		if (_station_sort_dirty[window_number]) { // resort in case of a station rename.
			MakeSortedStationList(window_number);
		}

		// stations are stored as a cummulative index, eg 25, 41, 43. This means
		// Player0: 25; Player1: (41-25) 16; Player2: (43-41) 2 stations
		i = (window_number == 0) ? 0 : _num_station_sort[window_number-1];
		SetVScrollCount(w, _num_station_sort[window_number] - i);

		/* draw widgets, with player's name in the caption */
		{
			Player *p = GetPlayer(window_number);
			SetDParam(0, p->name_1);
			SetDParam(1, p->name_2);
			SetDParam(2, w->vscroll.count);
			DrawWindowWidgets(w);
		}

		{
			byte p = 0;
			Station *st;
			int x,xb = 2;
			int y = 16;	// offset from top of widget
			int j;

			if (w->vscroll.count == 0) {	// player has no stations
				DrawString(xb, y, STR_304A_NONE, 0);
				return;
			}

			i += w->vscroll.pos;	// offset from sorted station list of current player
			assert(i < _num_station_sort[window_number]); // at least one station must exist

			while (i < _num_station_sort[window_number]) {	// do until max number of stations of owner
				st = GetStation(_station_sort[i].index);

				assert(st->xy && st->owner == window_number);

				SetDParam(0, st->index);
				SetDParam(1, st->facilities);
				x = DrawString(xb, y, STR_3049_0, 0) + 5;

				// show cargo waiting and station ratings
				for(j=0; j!=NUM_CARGO; j++) {
					int acc = (st->goods[j].waiting_acceptance & 0xFFF);
					if (acc != 0) {
						StationsWndShowStationRating(x, y, j, acc, st->goods[j].rating);
						x += 10;
					}
				}
				y += 10;
				i++;	// next station
				if (++p == w->vscroll.cap) { break;} // max number of stations in 1 window
			}
		}
	} break;
	case WE_CLICK: {
		switch(e->click.widget) {
		case 3: {
			uint32 id_v = (e->click.pt.y - 15) / 10;

			if (id_v >= w->vscroll.cap) { return;} // click out of bounds

			id_v += w->vscroll.pos;

			{
				const byte owner = (byte)w->window_number;
				Station *st;
				id_v	+= (owner == 0) ? 0 : _num_station_sort[owner - 1]; // first element in list

				if (id_v >= _num_station_sort[owner]) { return;} // click out of station bound

				st = GetStation(_station_sort[id_v].index);

				assert(st->xy && st->owner == owner);

				ScrollMainWindowToTile(st->xy);
			}
		} break;
		}
	} break;

	case WE_4:
		WP(w,plstations_d).refresh_counter++;
		if (WP(w,plstations_d).refresh_counter==5) {
			WP(w,plstations_d).refresh_counter = 0;
			SetWindowDirty(w);
		}
		break;

	case WE_RESIZE:
		w->vscroll.cap += e->sizing.diff.y / 10;
		break;
	}
}

static const Widget _player_stations_widgets[] = {
{   WWT_CLOSEBOX,   RESIZE_NONE,    14,     0,    10,     0,    13, STR_00C5,									STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,  RESIZE_RIGHT,    14,    11,   345,     0,    13, STR_3048_STATIONS,				STR_018C_WINDOW_TITLE_DRAG_THIS},
{  WWT_STICKYBOX,     RESIZE_LR,    14,   346,   357,     0,    13, 0x0,											STR_STICKY_BUTTON},
{      WWT_PANEL,     RESIZE_RB,    14,     0,   345,    14,   137, 0x0,											STR_3057_STATION_NAMES_CLICK_ON},
{  WWT_SCROLLBAR,    RESIZE_LRB,    14,   346,   357,    14,   125, 0x0,											STR_0190_SCROLL_BAR_SCROLLS_LIST},
{  WWT_RESIZEBOX,   RESIZE_LRTB,    14,   346,   357,   126,   137, 0x0,											STR_RESIZE_BUTTON},
{   WIDGETS_END},
};

static const WindowDesc _player_stations_desc = {
	-1, -1, 358, 138,
	WC_STATION_LIST,0,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET | WDF_STICKY_BUTTON | WDF_RESIZABLE,
	_player_stations_widgets,
	PlayerStationsWndProc
};


void ShowPlayerStations(int player)
{
	Window *w;

	w = AllocateWindowDescFront(&_player_stations_desc, player);
	if (w) {
		w->caption_color = (byte)w->window_number;
		w->vscroll.cap = 12;
		w->resize.step_height = 10;
		w->resize.height = w->height - 10 * 7; // minimum if 5 in the list
	}
}

static const Widget _station_view_expanded_widgets[] = {
{    WWT_TEXTBTN,   RESIZE_NONE,    14,     0,    10,     0,    13, STR_00C5,		STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,   RESIZE_NONE,    14,    11,   236,     0,    13, STR_300A_0,	STR_018C_WINDOW_TITLE_DRAG_THIS},
{  WWT_STICKYBOX,   RESIZE_NONE,    14,   237,   248,     0,    13, 0x0,         STR_STICKY_BUTTON},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   236,    14,    65, 0x0,					STR_NULL},
{  WWT_SCROLLBAR,   RESIZE_NONE,    14,   237,   248,    14,    65, 0x0,					STR_0190_SCROLL_BAR_SCROLLS_LIST},
{      WWT_EMPTY,   RESIZE_NONE,     0,     0,     0,     0,     0, 0x0,					STR_NULL},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   248,    66,   197, 0x0,					STR_NULL},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    63,   198,   209, STR_00E4_LOCATION,	STR_3053_CENTER_MAIN_VIEW_ON_STATION},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,    64,   128,   198,   209, STR_3033_ACCEPTS,	STR_3056_SHOW_LIST_OF_ACCEPTED_CARGO},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   129,   192,   198,   209, STR_0130_RENAME,		STR_3055_CHANGE_NAME_OF_STATION},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   193,   206,   198,   209, STR_TRAIN, STR_SCHEDULED_TRAINS_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   207,   220,   198,   209, STR_LORRY, STR_SCHEDULED_ROAD_VEHICLES_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   221,   234,   198,   209, STR_PLANE, STR_SCHEDULED_AIRCRAFT_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   235,   248,   198,   209, STR_SHIP, STR_SCHEDULED_SHIPS_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    63,   210,   221, STR_STATION_STATS, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,    64,   128,   210,   221, STR_STATION_RATING_DETAIL_KEY,						STR_NULL},
{     WWT_IMGBTN,   RESIZE_NONE,    14,   129,   192,   210,   221, 0x0,					STR_NULL},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   193,   248,   210,   221, STR_STATION_COVERAGE,						STR_NULL},
{   WIDGETS_END},
};

static const Widget _station_view_expanded_cheat_widgets[] = {
{    WWT_TEXTBTN,   RESIZE_NONE,    14,     0,    10,     0,    13, STR_00C5,		STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,   RESIZE_NONE,    14,    11,   236,     0,    13, STR_300A_0,	STR_018C_WINDOW_TITLE_DRAG_THIS},
{  WWT_STICKYBOX,   RESIZE_NONE,    14,   237,   248,     0,    13, 0x0,         STR_STICKY_BUTTON},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   236,    14,    65, 0x0,					STR_NULL},
{  WWT_SCROLLBAR,   RESIZE_NONE,    14,   237,   248,    14,    65, 0x0,					STR_0190_SCROLL_BAR_SCROLLS_LIST},
{      WWT_EMPTY,   RESIZE_NONE,     0,     0,     0,     0,     0, 0x0,					STR_NULL},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   248,    66,   197, 0x0,					STR_NULL},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    63,   198,   209, STR_00E4_LOCATION,	STR_3053_CENTER_MAIN_VIEW_ON_STATION},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,    64,   128,   198,   209, STR_3033_ACCEPTS,	STR_3056_SHOW_LIST_OF_ACCEPTED_CARGO},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   129,   192,   198,   209, STR_0130_RENAME,		STR_3055_CHANGE_NAME_OF_STATION},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   193,   206,   198,   209, STR_TRAIN, STR_SCHEDULED_TRAINS_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   207,   220,   198,   209, STR_LORRY, STR_SCHEDULED_ROAD_VEHICLES_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   221,   234,   198,   209, STR_PLANE, STR_SCHEDULED_AIRCRAFT_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   235,   248,   198,   209, STR_SHIP, STR_SCHEDULED_SHIPS_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    63,   210,   221, STR_STATION_STATS, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,    64,   128,   210,   221, STR_STATION_RATING_DETAIL_KEY,						STR_NULL},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   129,   192,   210,   221, STR_RESET_STATION, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   193,   248,   210,   221, STR_STATION_COVERAGE,						STR_NULL},
{   WIDGETS_END},
};

static const Widget _station_view_widgets[] = {
{    WWT_TEXTBTN,   RESIZE_NONE,    14,     0,    10,     0,    13, STR_00C5,		STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,   RESIZE_NONE,    14,    11,   236,     0,    13, STR_300A_0,	STR_018C_WINDOW_TITLE_DRAG_THIS},
{  WWT_STICKYBOX,   RESIZE_NONE,    14,   237,   248,     0,    13, 0x0,         STR_STICKY_BUTTON},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   236,    14,    65, 0x0,					STR_NULL},
{  WWT_SCROLLBAR,   RESIZE_NONE,    14,   237,   248,    14,    65, 0x0,					STR_0190_SCROLL_BAR_SCROLLS_LIST},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   248,    66,    97, 0x0,					STR_NULL},
{      WWT_EMPTY,   RESIZE_NONE,     0,     0,     0,     0,     0, 0x0,					STR_NULL},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    63,    98,   109, STR_00E4_LOCATION,	STR_3053_CENTER_MAIN_VIEW_ON_STATION},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,    64,   128,    98,   109, STR_3032_RATINGS,	STR_3054_SHOW_STATION_RATINGS},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   129,   192,    98,   109, STR_0130_RENAME,		STR_3055_CHANGE_NAME_OF_STATION},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   193,   206,    98,   109, STR_TRAIN, STR_SCHEDULED_TRAINS_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   207,   220,    98,   109, STR_LORRY, STR_SCHEDULED_ROAD_VEHICLES_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   221,   234,    98,   109, STR_PLANE, STR_SCHEDULED_AIRCRAFT_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   235,   248,    98,   109, STR_SHIP, STR_SCHEDULED_SHIPS_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    63,   110,   121, STR_STATION_STATS, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,    64,   128,   110,   121, STR_STATION_RATING_DETAIL_KEY,						STR_NULL},
{     WWT_IMGBTN,   RESIZE_NONE,    14,   129,   192,   110,   121, 0x0,					STR_NULL},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   193,   248,   110,   121, STR_STATION_COVERAGE,						STR_NULL},
{   WIDGETS_END},
};

static const Widget _station_view_cheat_widgets[] = {
{    WWT_TEXTBTN,   RESIZE_NONE,    14,     0,    10,     0,    13, STR_00C5,		STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,   RESIZE_NONE,    14,    11,   236,     0,    13, STR_300A_0,	STR_018C_WINDOW_TITLE_DRAG_THIS},
{  WWT_STICKYBOX,   RESIZE_NONE,    14,   237,   248,     0,    13, 0x0,         STR_STICKY_BUTTON},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   236,    14,    65, 0x0,					STR_NULL},
{  WWT_SCROLLBAR,   RESIZE_NONE,    14,   237,   248,    14,    65, 0x0,					STR_0190_SCROLL_BAR_SCROLLS_LIST},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   248,    66,    97, 0x0,					STR_NULL},
{      WWT_EMPTY,   RESIZE_NONE,     0,     0,     0,     0,     0, 0x0,					STR_NULL},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    63,    98,   109, STR_00E4_LOCATION,	STR_3053_CENTER_MAIN_VIEW_ON_STATION},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,    64,   128,    98,   109, STR_3032_RATINGS,	STR_3054_SHOW_STATION_RATINGS},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   129,   192,    98,   109, STR_0130_RENAME,		STR_3055_CHANGE_NAME_OF_STATION},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   193,   206,    98,   109, STR_TRAIN, STR_SCHEDULED_TRAINS_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   207,   220,    98,   109, STR_LORRY, STR_SCHEDULED_ROAD_VEHICLES_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   221,   234,    98,   109, STR_PLANE, STR_SCHEDULED_AIRCRAFT_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   235,   248,    98,   109, STR_SHIP, STR_SCHEDULED_SHIPS_TIP },
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    63,   110,   121, STR_STATION_STATS, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,    64,   128,   110,   121, STR_STATION_RATING_DETAIL_KEY,						STR_NULL},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   129,   192,   110,   121, STR_RESET_STATION, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   193,   248,   110,   121, STR_STATION_COVERAGE,						STR_NULL},
{   WIDGETS_END},
};

static void DrawStationViewWindow(Window *w)
{
	Station *st;
	int i;
	int num;
	int x,y;
	int pos;
	StringID str;
	uint16 station_id;

	station_id = (uint16)w->window_number;

	st = GetStation(w->window_number);

	num = 1;
	for(i=0; i!=NUM_CARGO; i++) {
		if ((st->goods[i].waiting_acceptance & 0xFFF) != 0) {
			num++;
			if (st->goods[i].enroute_from != station_id)
				num++;
		}
	}
	SetVScrollCount(w, num);

	if (_cheats.reset_station.value)
        w->disabled_state = st->owner == _local_player ? 0 : (1 << 9) | (1 << 14) | (1 << 15);
	else
		w->disabled_state = st->owner == _local_player ? 0 : (1 << 9) | (1 << 14);

	if (!(st->facilities & FACIL_TRAIN)) SETBIT(w->disabled_state,  10);
	if (!(st->facilities & FACIL_TRUCK_STOP) &&
			!(st->facilities & FACIL_BUS_STOP)) SETBIT(w->disabled_state, 11);
	if (!(st->facilities & FACIL_AIRPORT)) SETBIT(w->disabled_state, 12);
	if (!(st->facilities & FACIL_DOCK)) SETBIT(w->disabled_state, 13);

	SetDParam(0, st->index);
	SetDParam(1, st->facilities);
	DrawWindowWidgets(w);

	x = 2;
	y = 15;
	pos = w->vscroll.pos;

	if (--pos < 0) {
		str = STR_00D0_NOTHING;
		for(i=0; i!=NUM_CARGO; i++)
			if (st->goods[i].waiting_acceptance & 0xFFF)
				str = STR_EMPTY;
		SetDParam(0, str);
		DrawString(x, y, STR_0008_WAITING, 0);
		y += 10;
	}

	i = 0;
	do {
		uint waiting = (st->goods[i].waiting_acceptance & 0xFFF);
		if (waiting == 0)
			continue;

		num = (waiting + 5) / 10;
		if (num != 0) {
			int cur_x = x;
			num = min(num, 23);
			do {
				DrawSprite(_cargoc.sprites[i], cur_x, y);
				cur_x += 10;
			} while (--num);
		}

		if ( st->goods[i].enroute_from == station_id) {
			if (--pos < 0) {
				SetDParam(1, waiting);
				SetDParam(0, _cargoc.names_long_s[i] + (waiting==1 ? 0 : 32));
				DrawStringRightAligned(x + 234, y, STR_0009, 0);
				y += 10;
			}
		} else {
			/* enroute */
			if (--pos < 0) {
				SetDParam(1, waiting);
				SetDParam(0, _cargoc.names_long_s[i] + (waiting==1 ? 0 : 32));
				DrawStringRightAligned(x + 234, y, STR_000A_EN_ROUTE_FROM, 0);
				y += 10;
			}

			if (pos > -5 && --pos < 0) {
				SetDParam(0, st->goods[i].enroute_from);
				DrawStringRightAligned(x + 234, y, STR_000B, 0);
				y += 10;
			}
		}
	} while (pos > -5 && ++i != 12);

	if (IsWindowOfPrototype(w, _station_view_widgets)) {
		char *b = _userstring;

		b = InlineString(b, STR_000C_ACCEPTS);

		for (i = 0; i != NUM_CARGO; i++) {
			if (b >= endof(_userstring) - 5 - 1) break;
			if (st->goods[i].waiting_acceptance & 0x8000) {
				b = InlineString(b, _cargoc.names_s[i]);
				*b++ = ',';
				*b++ = ' ';
			}
		}

		if (b == &_userstring[3]) {
			b = InlineString(b, STR_00D0_NOTHING);
			*b++ = '\0';
		} else {
			b[-2] = '\0';
		}

		DrawStringMultiLine(2, 67, STR_SPEC_USERSTRING, 245);
	} else {

		DrawString(2, 67, STR_3034_LOCAL_RATING_OF_TRANSPORT, 0);

		y = 77;
		for(i=0; i!=NUM_CARGO; i++) {
			if (st->goods[i].enroute_from != INVALID_STATION) {
				SetDParam(0, _cargoc.names_s[i]);
				SetDParam(2, st->goods[i].rating * 101 >> 8);
				SetDParam(1, STR_3035_APPALLING + (st->goods[i].rating >> 5));
				DrawString(8, y, STR_303D, 0);
				y += 10;
			}
		}
	}
}


static void StationViewWndProc(Window *w, WindowEvent *e)
{
	switch(e->event) {
	case WE_PAINT:
		DrawStationViewWindow(w);
		break;

	case WE_CLICK:
		switch(e->click.widget) {
			case 7: {
				ScrollMainWindowToTile(GetStation(w->window_number)->xy);
				break;
			}
			case 8: {
				SetWindowDirty(w);

				/* toggle height/widget set */
				if (IsWindowOfPrototype(w, _station_view_expanded_widgets)
					|| IsWindowOfPrototype(w, _station_view_expanded_cheat_widgets)) {
					if (_cheats.reset_station.value)
						AssignWidgetToWindow(w, _station_view_cheat_widgets);
					else
						AssignWidgetToWindow(w, _station_view_widgets);
					w->height = 122;
				} else {
					if (_cheats.reset_station.value)
						AssignWidgetToWindow(w, _station_view_expanded_cheat_widgets);
					else
						AssignWidgetToWindow(w, _station_view_expanded_widgets);
					w->height = 222;
				}

				SetWindowDirty(w);
				break;
			}

			case 9: {
				Station *st = GetStation(w->window_number);
				SetDParam(0, st->town->townnametype);
				SetDParam(1, st->town->townnameparts);
				ShowQueryString(st->string_id, STR_3030_RENAME_STATION_LOADING, 31, 180, w->window_class, w->window_number);
			}	break;
			case 10: { /* Show a list of scheduled trains to this station */
				const Station *st = GetStation(w->window_number);
				ShowPlayerTrains(st->owner, w->window_number);
				break;
			}

			case 11: { /* Show a list of scheduled road-vehicles to this station */
				const Station *st = GetStation(w->window_number);
				ShowPlayerRoadVehicles(st->owner, w->window_number, CARGO_MASK_ALL);
				break;
			}
			case 12: { /* Show a list of scheduled aircraft to this station */
				const Station *st = GetStation(w->window_number);
				/* Since oilrigs have no owners, show the scheduled aircraft of current player */
				PlayerID owner = (st->owner == OWNER_NONE) ? _current_player : st->owner;
				ShowPlayerAircraft(owner, w->window_number);
				break;
			}

			case 13: { /* Show a list of scheduled ships to this station */
				const Station *st = GetStation(w->window_number);
				/* Since oilrigs/bouys have no owners, show the scheduled ships of current player */
				PlayerID owner = (st->owner == OWNER_NONE) ? _current_player : st->owner;
				ShowPlayerShips(owner, w->window_number);
				break;
			}

			case 14: { // Show Statistics for Station
				ShowStationStatsWindow(w->window_number);
				break;
			}
			case 15: { // Show Details for Station Rating
				const Station *st = GetStation(w->window_number);
				ShowStationRatingDetail(st->index);
				break;
			}
			case 16: { // Reset Station
				if (_cheats.reset_station.value) {
					Station *st = GetStation(w->window_number);
					GoodsEntry *ge;

					for(ge = st->goods; ge != endof(st->goods); ge++) {
						ge->last_speed = 0;
						ge->waiting_acceptance = 0;
						ge->days_since_pickup = 0;
						ge->enroute_from = 0xFFFF;
						ge->rating = 175;
						ge->last_speed = 0;
						ge->last_age = 0xFF;
						ge->feeder_profit = 0;
					}
					InitializeStationStats(st);
					st->had_vehicle_of_type = 0;
					st->time_since_load = 255;
					st->time_since_unload = 255;
					st->last_vehicle = INVALID_VEHICLE;

					SetWindowDirty(w);
				}
				break;
 			}
			case 17: { //ToDo: show station coverage
				break;
			}
		}
		break;
	case WE_ON_EDIT_TEXT: {
		if (e->edittext.str[0] != '\0') {
			Station* st = GetStation(w->window_number);

			_cmd_text = e->edittext.str;
			DoCommandP(st->xy, w->window_number, 0, NULL,
				CMD_RENAME_STATION | CMD_MSG(STR_3031_CAN_T_RENAME_STATION));
		}
	} break;

	case WE_DESTROY: {
		WindowNumber wno =
			(w->window_number << 16) | GetStation(w->window_number)->owner;

		DeleteWindowById(WC_TRAINS_LIST, wno);
		DeleteWindowById(WC_ROADVEH_LIST, wno);
		DeleteWindowById(WC_SHIPS_LIST, wno);
		DeleteWindowById(WC_AIRCRAFT_LIST, wno);
		DeleteWindowById(WC_STATION_STATS, w->window_number);
		break;
	}
	}
}


static const WindowDesc _station_view_desc = {
	-1, -1, 249, 122,
	WC_STATION_VIEW,0,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET | WDF_UNCLICK_BUTTONS | WDF_STICKY_BUTTON,
	_station_view_widgets,
	StationViewWndProc
};

static const WindowDesc _station_view_cheat_desc = {
	-1, -1, 249, 122,
	WC_STATION_VIEW,0,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET | WDF_UNCLICK_BUTTONS | WDF_STICKY_BUTTON,
	_station_view_cheat_widgets,
	StationViewWndProc
};

void ShowStationViewWindow(int station)
{
	Window *w;
	byte color;

	if (_cheats.reset_station.value)
		w = AllocateWindowDescFront(&_station_view_cheat_desc, station);
	else
		w = AllocateWindowDescFront(&_station_view_desc, station);
	if (w) {
		color = GetStation(w->window_number)->owner;
		if (color != 0x10)
			w->caption_color = color;
		w->vscroll.cap = 5;
	}
}

void StationStatsWndProc(Window *w, WindowEvent *e);

Widget _station_stats_widgets[] = {
{    WWT_TEXTBTN,   RESIZE_NONE,    14,     0,    10,     0,    13, STR_00C5, STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,   RESIZE_NONE,    14,    11,   398,     0,    13, STR_300A_0, STR_018C_WINDOW_TITLE_DRAG_THIS},
{  WWT_STICKYBOX,   RESIZE_NONE,    14,   399,   410,     0,    13, 0x0,         STR_STICKY_BUTTON},
{      WWT_EMPTY,   RESIZE_NONE,     0,     0,     0,     0,     0, 0x0, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    90,    14,    25, STR_RESET_STATISTICS, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,    91,   410,    14,    25, STR_TOGGLE_MINMAX, 0x0},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   410,    26,    37, 0x0, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    90,    38,    49, STR_TRAINS, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    90,    50,    61, STR_RVS, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    90,    62,    73, STR_BUSSES, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    90,    74,    85, STR_TRUCKS, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    90,    85,    97, STR_SHIPS, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    90,    98,   109, STR_AIRCRAFT, 0x0},
{     WWT_IMGBTN,   RESIZE_NONE,    14,    91,   410,    38,   109, 0x0,0x0},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   410,   110,   135, 0x0,0x0},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   398,   136,   191, 0x0,0x0},
{  WWT_SCROLLBAR,   RESIZE_NONE,    14,   399,   410,   136,   191, 0x0, STR_0190_SCROLL_BAR_SCROLLS_LIST},
{      WIDGETS_END},
};

WindowDesc _station_view_stats = {
	-1, -1, 411, 192,
	WC_STATION_STATS, 0,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET | WDF_UNCLICK_BUTTONS | WDF_STICKY_BUTTON,
	_station_stats_widgets,
	StationStatsWndProc
};

void DrawStationStatWindow(Window *w, Station *st)
{
	int i, y, numcargo=0, pos;
	GoodsEntry *ge;
	
	// count number of goods at station (months_counted will be 0 if good not delivered/picked up)
	for (ge = st->goods; ge != endof(st->goods); ge++) {
		if (ge->months_counted != 0) numcargo++;
	}
	SetVScrollCount(w, numcargo);
		
	//Get the Station name
	SetDParam(0, st->index);
	
	//and the little carrier type images
	SetDParam(1, st->facilities);
	//First draw the widgets
	DrawWindowWidgets(w);

	DrawStringCentered(45, 28, STR_VEHICLES, 0);

	//Part 1:       Find the number of carriers on the station
	y = 27;
	if (_show_average_stats) {
		DrawStringRightAligned(150, y, STR_SCHEDULED, 0);
		DrawStringRightAligned(275, y, STR_VEHICLES_MONTHS_AVERAGE, 0);
		SetDParam(0, max(st->months_counted - 1,0));
		DrawStringRightAligned(405, y, STR_MONTHS_COUNTED_NUM, 0);
	} else {
		DrawStringRightAligned(150, y, STR_SCHEDULED, 0);
		DrawStringRightAligned(295, y, STR_VEHICLES_MONTH, 0);
		DrawStringRightAligned(395, y, STR_VEHICLES_CURRENT, 0);	
	}

	y = 38;
	for (i = 0; i < STS_VEH_TYPES; i++) {
		if (w->custom[i] != 0) {
			SetDParam(0, w->custom[i]);
			DrawStringRightAligned(150, y+i*12, STR_NUMBER, 0);
			CLRBIT(w->disabled_state, i+7);
		} else {
			SETBIT(w->disabled_state, i+7);
		}
		if (_show_average_stats) {
			if (st->vehicles[i].average != 0 || st->vehicles[i].month_max !=0) {
				SetDParam(0, st->vehicles[i].average / STS_MULTIPLIER);
				DrawStringRightAligned(235, y+i*12, STR_SILVER_NUMBER, 0);
				DrawStringRightAligned(245, y+i*12, STR_BLACK_SLASH, 0);
				SetDParam(0, st->vehicles[i].month_min);
				DrawStringRightAligned(290, y+i*12, STR_ORANGE_NUMBER, 0);
				DrawStringRightAligned(300, y+i*12, STR_BLACK_SLASH, 0);
				SetDParam(0, st->vehicles[i].month_max);
				DrawStringRightAligned(350, y+i*12, STR_LTBLUE_NUMBER, 0);
			}
		} else {
			if (st->vehicles[i].last_month != 0) {
				SetDParam(0, st->vehicles[i].last_month);
				DrawStringRightAligned(295, y+i*12, STR_NUMBER, 0);
			}
			if (st->vehicles[i].this_month) {
				SetDParam(0, st->vehicles[i].this_month);
				DrawStringRightAligned(395, y+i*12, STR_WHITE_NUMBER, 0);
			}
		}
	}
	
	y = 124;
	if (_show_average_stats) {
		DrawString(5, y-10, STR_STATION_MONTHS_AVERAGE, 0);
		DrawStringRightAligned(100, y+2, STR_MONTHS_TINY, 0);
	} else {
        DrawString(5, y-10, STR_STATION_MONTHS, 0);
	}
	DrawStringRightAligned(200, y, STR_STATION_GOODS_IN, 0);
	DrawStringRightAligned(305, y, STR_STATION_GOODS_OUT, 0);
	DrawStringRightAligned(395, y, STR_STATION_GOODS_TRANSFER, 0);
	
	y += 5;
	pos = w->vscroll.pos;
	for (i = 0; i < NUM_CARGO; i++) {
		ge = &st->goods[i];
		if (ge->months_counted != 0 && ((--pos < 0) && (pos >= -4)) )
		{
			StationStats *sts = ge->cargo_amount;
			//Print the cargo name
			y += 12;
			SetDParam(0, _cargoc.names_p[i]);
			DrawString(3, y, STR_02BD, 0);
			if (_show_average_stats) {
				SetDParam(0, ge->months_counted - 1);
				DrawStringRightAligned(100, y, STR_TINY_GOLD_NUMBER, 0);
				if (sts[STS_AMOUNT_IN].average != 0 || sts[STS_AMOUNT_IN].month_max != 0) {
					SetDParam(0, sts[STS_AMOUNT_IN].average / STS_MULTIPLIER);
					SetDParam(1, sts[STS_AMOUNT_IN].month_min);
					SetDParam(2, sts[STS_AMOUNT_IN].month_max);
					DrawStringRightAligned(200, y, STR_AVERAGENUMBERS, 0);
				}
				if (sts[STS_AMOUNT_OUT].average != 0 || sts[STS_AMOUNT_OUT].month_max != 0) {
					SetDParam(0, sts[STS_AMOUNT_OUT].average / STS_MULTIPLIER);
					SetDParam(1, sts[STS_AMOUNT_OUT].month_min);
					SetDParam(2, sts[STS_AMOUNT_OUT].month_max);
					DrawStringRightAligned(305, y, STR_AVERAGENUMBERS, 0);
				}
				if (sts[STS_AMOUNT_TRANSFER].average != 0 || sts[STS_AMOUNT_TRANSFER].month_max != 0) {
					SetDParam(0, sts[STS_AMOUNT_TRANSFER].average / STS_MULTIPLIER);
					SetDParam(1, sts[STS_AMOUNT_TRANSFER].month_min);
					SetDParam(2, sts[STS_AMOUNT_TRANSFER].month_max);
					DrawStringRightAligned(395, y, STR_AVERAGENUMBERS, 0);
				}
			} else {
				if (sts[STS_AMOUNT_IN].this_month != 0 || sts[STS_AMOUNT_IN].last_month != 0) {
					SetDParam(0, sts[STS_AMOUNT_IN].this_month);
					SetDParam(1, sts[STS_AMOUNT_IN].last_month);
					DrawStringRightAligned(200, y, STR_CNUMBERS, 0);
				}
				if (sts[STS_AMOUNT_OUT].this_month != 0 || sts[STS_AMOUNT_OUT].last_month != 0) {
					SetDParam(0, sts[STS_AMOUNT_OUT].this_month);
					SetDParam(1, sts[STS_AMOUNT_OUT].last_month);
					DrawStringRightAligned(305, y, STR_CNUMBERS, 0);
				}
				if (sts[STS_AMOUNT_TRANSFER].this_month != 0 || sts[STS_AMOUNT_TRANSFER].last_month != 0) {
					SetDParam(0, sts[STS_AMOUNT_TRANSFER].this_month);
					SetDParam(1, sts[STS_AMOUNT_TRANSFER].last_month);
					DrawStringRightAligned(395, y, STR_CNUMBERS, 0);
				}
			}
		}
	}
}

void StationStatsWndProc(Window *w, WindowEvent *e)
{
	Station *st = GetStation(w->window_number);
	switch(e->event)
	{
	case WE_TICK: {
//		static int counter = 0;
//		if (++counter % 40) return;
		int i;
		
		for (i = 0; i < STS_VEH_TYPES; i++)
			w->custom[i] = st->veh_scheduled[i];
		InvalidateWindow(WC_STATION_STATS, w->window_number);
		break;
	}
	case WE_PAINT: {
		DrawStationStatWindow(w, st);
		break;
		}
	case WE_CLICK: {
		switch (e->click.widget)
		{
			case 4:			// Reset Statistics
				if (st->owner == _current_player) {
					InitializeStationStats(st);
					InvalidateWindow(WC_STATION_STATS, w->window_number);
				}
				break;
			case 5:			// Toggle Average and This Month
				_show_average_stats ^= 1;
				InvalidateWindow(WC_STATION_STATS, w->window_number);
				break;
			case 7:			//Trains
				ShowPlayerTrains(st->owner, st->index);
				break;
			case 8: 		//Road Vehicles
				ShowPlayerRoadVehicles(st->owner, st->index, CARGO_MASK_ALL);
				break;
			case 9:			// Buses
				ShowPlayerRoadVehicles(st->owner, st->index, 1 << GC_PASSENGERS);
				break;
			case 10:		// Trucks
				ShowPlayerRoadVehicles(st->owner, st->index, CARGO_MASK_ALL &~(1 << GC_PASSENGERS));
				break;
			case 11: 		//Ships
				ShowPlayerShips(st->owner, st->index);
				break;
			case 12: 		//Aircraft
				ShowPlayerAircraft(st->owner, st->index);
				break;
		}
	} break;
	case WE_DESTROY: {
//		DeleteWindowById(WC_TRAINS_LIST, st->owner + ( (st->index + 1) << 8));
//		DeleteWindowById(WC_ROADVEH_LIST, st->owner + ( (st->index + 1) << 8) + 64 + 128);
//		DeleteWindowById(WC_SHIPS_LIST, st->owner + ( (st->index + 1) << 8));
//		DeleteWindowById(WC_AIRCRAFT_LIST, st->owner + ( (st->index + 1) << 8));
	} break;
	}
}

void ShowStationStatsWindow(int station)
{
	Window *w;
	byte color;

	Station *st = GetStation(station);
		
	w = AllocateWindowDescFront(&_station_view_stats, st->index);
	if (w) {
		int i;
		color = st->owner;
		if (color != 0x10)
			w->caption_color = color;
		w->vscroll.cap = 4;
		for (i = 0; i < STS_VEH_TYPES; i++)
			w->custom[i] = st->veh_scheduled[i];
	}
}

static void StationRatingDetailWndProc(Window *w, WindowEvent *e)
{
	// String IDs to display
	static const uint16 rating_str[] = {
		STR_STATION_RATING_DETAIL_WAITING,
		STR_STATION_RATING_DETAIL_SPEED,
		STR_STATION_RATING_DETAIL_AGE,
		STR_STATION_RATING_DETAIL_PICKUP,
		STR_STATION_RATING_DETAIL_OTHER,
		STR_STATION_RATING_DETAIL_TOTAL
	};
	// More string IDs to display
	static const uint16 rating_str_ex[] = {
		STR_STATION_RATING_DETAIL_WAITING_EX,
		STR_STATION_RATING_DETAIL_SPEED_EX,
		STR_STATION_RATING_DETAIL_AGE_EX,
		STR_STATION_RATING_DETAIL_PICKUP_EX,
		STR_NULL,
		STR_NULL
	};
	static const int rating_max_score[] = {40, 42, 33, 130, 26, 255};

	switch(e->event) {
	case WE_PAINT: {
		int i, x, cur_x, val;
		byte cargo;
		uint16 y = 16;
		int color_done, color_notdone;
		int max_score;
		Station *st = GetStation(w->window_number);
		RatingStats *rs;

		// Disable cargo types that don't have ratings and click first cargo
		w->disabled_state = 0;
		for (i=0; i!=NUM_CARGO; i++) {
			if(st->goods[i].enroute_from == INVALID_STATION) {
				SETBIT(w->disabled_state, i + 9);
				CLRBIT(w->click_state, i + 9);
			} else if(w->click_state == 0) {
				SETBIT(w->click_state, i + 9);
			}
		}
		if(w->click_state == 0)
			SETBIT(w->disabled_state, 21);

		// Draw standard stuff
		SetDParam(0, st->index);
		SetDParam(1, st->facilities);
		DrawWindowWidgets(w);

		// Paint the cargo icons
		cur_x = 9;
		for (i = 0; i != NUM_CARGO; i++) {
			if(st->goods[i].enroute_from != INVALID_STATION)
				DrawSprite(_cargoc.sprites[i], cur_x, y);
			cur_x += 28;
		}

		// The colors used to show how the progress is going
		color_done = _color_list[6].window_color_1b;
		color_notdone = _color_list[4].window_color_1b;

		// No cargo type selected
		if(w->click_state == 0)
			break;

		// The type of cargo of which we check the detail service rating
		cargo = FindFirstBit(w->click_state) - 9;
		
		if (cargo < NUM_CARGO) 
			w->listopt.cargo_mask = 1 << GetGlobalCargoID(_opt_ptr->landscape, cargo);
		else
			w->listopt.cargo_mask = CARGO_MASK_ALL;
		w->listopt.veh_type_mask = VEHICLE_MASK_ALL;
		w->listopt.xy = st->xy;

		rs = &st->rating_stats[cargo];

		y += 18;

		for(i=0; i<NUM_RATINGS; i++) {
			DrawString(7, y, rating_str[i], 0);

			// Draw the raw values
			switch(i) {
				case RATING_WAITING:
					SetDParam(1, rs->waiting);
					SetDParam(0, _cargoc.names_long_s[cargo] + (rs->waiting == 1 ? 0 : 32));
					break;
				case RATING_SPEED:
					SetDParam(0, rs->last_speed);
					break;
				case RATING_AGE:
					SetDParam(0, rs->last_age);
					break;
				case RATING_PICKUP:
					SetDParam(0, rs->days_since_pickup);
					break;
				default:
					break;
			}

			if(rating_str_ex[i] != STR_NULL)
				DrawStringRightAligned(205, y, rating_str_ex[i], 0);

			// Calculate the %-bar
			val = rs->ratings[i];

			max_score = rating_max_score[i];
			if (val > max_score) x = 50;
			else if (val <= 0) x = 0;
			else x = ((val * 50) / max_score);

			// Draw the bar
			if (x != 0)
				GfxFillRect(210, y-2, x + 210, y+10, color_done);
			if (x != 50)
				GfxFillRect(x + 210, y-2, 50 + 210, y+10, color_notdone);

			// Calculate the %
			if (val > max_score) x = 100;
			else x = ((val * 100) / max_score);

			// Draw it
			SetDParam(0, x);
			DrawStringCentered(235, y, STR_STATION_RATING_DETAIL_PERCENT, 0);

			// Draw the score and max_score
			SetDParam(0, val);
			SetDParam(1, max_score);
			DrawString(265, y, STR_STATION_RATING_DETAIL_SCORES, 0);

			y += 20;
		}

		break;
	}

	case WE_CLICK:
		// Check which button is clicked
		if (IS_INT_INSIDE(e->click.widget, 9, 21)) {
			// Is it no on disable?
			if ((w->disabled_state & (1 << e->click.widget)) == 0) {
				Station *st = GetStation(w->window_number);
				Window *w2 = FindWindowById(WC_VEHICLES_LIST, (st->index << 16) | st->owner);
				vehiclelist_d *vl;
				w->click_state = 1 << e->click.widget;
				w->listopt.cargo_mask = 1 << GetGlobalCargoID(_opt_ptr->landscape, FindFirstBit(w->click_state) - 9);
				SetWindowDirty(w);
				if (w2) {
					vl = &WP(w2, vehiclelist_d);
					vl->flags |= VL_REBUILD;
					w2->listopt.cargo_mask = w->listopt.cargo_mask;
					SetWindowDirty(w2);
				}
			}
		} else if (e->click.widget == 21) {
			Station *st = GetStation(w->window_number);
			ShowPlayerVehicles(st->owner, st->index, w->listopt.veh_type_mask, w->listopt.cargo_mask, w->listopt.xy);
			w->click_state = 1 << (GetLocalCargoID(FindFirstBit(w->listopt.cargo_mask)) + 9);
			SetWindowDirty(w);
		}
		break;

	case WE_CREATE:
		{
			w->hidden_state = 0;
			w->disabled_state = 0;
			w->click_state = 0;

			SetWindowDirty(w);
		}
		break;
	}
}

static const Widget _station_rating_detail_widgets[] = {
{    WWT_TEXTBTN,   RESIZE_NONE,    14,     0,    10,     0,    13, STR_00C5,STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,   RESIZE_NONE,    14,    11,   325,     0,    13, STR_STATION_RATING_DETAIL,	STR_018C_WINDOW_TITLE_DRAG_THIS},
{  WWT_STICKYBOX,   RESIZE_NONE,    14,   326,   337,     0,    13, 0x0, STR_STICKY_BUTTON},

{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   337,    28,    47, 0x0,STR_STATION_RATING_DETAIL_WAITING_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   337,    48,    67, 0x0,STR_STATION_RATING_DETAIL_SPEED_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   337,    68,    87, 0x0,STR_STATION_RATING_DETAIL_AGE_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   337,    88,   107, 0x0,STR_STATION_RATING_DETAIL_PICKUP_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   337,   108,   127, 0x0,STR_STATION_RATING_DETAIL_OTHER_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   337,   128,   147, 0x0,STR_STATION_RATING_DETAIL_TOTAL_TIP},

{     WWT_IMGBTN,   RESIZE_NONE,    14,     1,    28,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,    29,    56,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,    57,    84,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,    85,   112,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,   113,   140,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,   141,   168,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,   169,   196,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,   197,   224,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,   225,   252,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,   253,   280,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,   281,   308,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{     WWT_IMGBTN,   RESIZE_NONE,    14,   309,   336,    14,    27, 0x0,STR_STATION_RATING_DETAIL_TOGGLE_CARGO_TIP},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,   336,    148,   160, STR_VEHICLE_CARGO_LIST, 0x0},
{   WIDGETS_END},
};

static const WindowDesc _station_rating_detail_desc = {
	-1, -1, 338, 161,
	WC_STATION_RATING_DETAIL,0,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET | WDF_UNCLICK_BUTTONS | WDF_STICKY_BUTTON,
	_station_rating_detail_widgets,
	StationRatingDetailWndProc
};

void ShowStationRatingDetail(int station)
{
	Window *w;
	byte color;

	w = AllocateWindowDescFront(&_station_rating_detail_desc, station);
	if (w) {
		color = GetStation(w->window_number)->owner;
		if (color != 0x10)
			w->caption_color = color;
		w->vscroll.cap = 5;
	}
}

static void PlayerVehiclesWndProc(Window *w, WindowEvent *e)
{
	int station = (int)w->window_number >> 16;
	int owner = w->window_number & 0xff;
	vehiclelist_d *vl = &WP(w, vehiclelist_d);

	switch(e->event) {
	case WE_PAINT: {
		int x = 2;
		int y = PLY_WND_PRC__OFFSET_TOP_WIDGET;
		int max;
		int i;

		BuildVehicleListMasked(vl, &w->listopt, owner);
		SortVehicleList(vl);

		SetVScrollCount(w, vl->list_length);
		// disable 'Sort By' tooltip on Unsorted sorting criteria
		if (vl->sort_type == SORT_BY_UNSORTED)
			w->disabled_state |= (1 << 3);

		/* draw the widgets */
		{
			const Player *p = GetPlayer(owner);
			if (station == -1) {
				/* Company Name -- (###) Trains */
				SetDParam(0, p->name_1);
				SetDParam(1, p->name_2);
				SetDParam(2, w->vscroll.count);
				w->widget[1].unkA = STR_VEHICLES;
			} else {
				/* Station Name -- (###) Trains */
				SetDParam(0, station);
				SetDParam(1, w->vscroll.count);
				w->widget[1].unkA = STR_SCHEDULED;
			}
			DrawWindowWidgets(w);
		}
		/* draw sorting criteria string */
		DrawString(85, 15, _vehicle_sort_listing[vl->sort_type], 0x10);
		/* draw arrow pointing up/down for ascending/descending sorting */
		DoDrawString(vl->flags & VL_DESC ? "\xAA" : "\xA0", 69, 15, 0x10);

		max = min(w->vscroll.pos + w->vscroll.cap, vl->list_length);
		for (i = w->vscroll.pos; i < max; ++i) {
			Vehicle *v = GetVehicle(vl->sort_list[i].index);
			StringID str;

			assert(v->owner == owner);

			switch (v->type) {
				case VEH_Train:
				{
					DrawTrainImage(
						v, x + 21, y + 6 + _traininfo_vehicle_pitch, w->hscroll.cap, 0, INVALID_VEHICLE);

					SetDParam(0, v->unitnumber);
					if (IsTileDepotType(v->tile, TRANSPORT_RAIL) && (v->vehstatus & VS_HIDDEN))
						str = STR_021F;
					else
						str = v->age > v->max_age - 366 ? STR_00E3 : STR_00E2;
					DrawString(x, y + 2, str, 0);
					if (v->string_id != STR_SV_TRAIN_NAME) {
						SetDParam(0, v->string_id);
						DrawString(x + 21, y, STR_01AB, 0);
					}
					break;
				}
				case VEH_Road:
				{
					DrawRoadVehImage(v, x + 22, y + 6, INVALID_VEHICLE);
					SetDParam(0, v->unitnumber);
					if (IsTileDepotType(v->tile, TRANSPORT_ROAD) && (v->vehstatus & VS_HIDDEN))
						str = STR_021F;
					else
						str = v->age > v->max_age - 366 ? STR_00E3 : STR_00E2;
					DrawString(x, y + 2, str, 0);
					if (v->string_id != STR_SV_ROADVEH_NAME) {
						SetDParam(0, v->string_id);
						DrawString(x + 24, y, STR_01AB, 0);
					}
					break;
				}
				case VEH_Aircraft:
				{
					DrawAircraftImage(v, x + 19, y + 6, INVALID_VEHICLE);
					SetDParam(0, v->unitnumber);
					if (IsAircraftHangarTile(v->tile) && (v->vehstatus & VS_HIDDEN))
						str = STR_021F;
					else
						str = v->age > v->max_age - 366 ? STR_00E3 : STR_00E2;
					DrawString(x, y + 2, str, 0);
					if (v->string_id != STR_SV_AIRCRAFT_NAME) {
						SetDParam(0, v->string_id);
						DrawString(x + 19, y, STR_01AB, 0);
					}
					break;
				}
				case VEH_Ship:
				{
					DrawShipImage(v, x + 19, y + 6, INVALID_VEHICLE);
					SetDParam(0, v->unitnumber);
					if (IsTileDepotType(v->tile, TRANSPORT_WATER) && (v->vehstatus & VS_HIDDEN))
						str = STR_021F;
					else
						str = v->age > v->max_age - 366 ? STR_00E3 : STR_00E2;
					DrawString(x, y + 2, str, 0);
					if (v->string_id != STR_SV_SHIP_NAME) {
						SetDParam(0, v->string_id);
						DrawString(x + 12, y, STR_01AB, 0);
					}
					break;
				}
				default:
					continue;
			}
			DrawVehicleProfitButton(v, x, y + 13);
			SetDParam(0, v->profit_this_year);
			SetDParam(1, v->profit_last_year);
			DrawString(x + 21, y + 18, STR_0198_PROFIT_THIS_YEAR_LAST_YEAR, 0);

			y += PLY_WND_PRC__SIZE_OF_ROW_BIG;
		}
		break;
	}

	case WE_CLICK: {
		switch(e->click.widget) {
		case 3: /* Flip sorting method ascending/descending */
			vl->flags ^= VL_DESC;
			vl->flags |= VL_RESORT;
			_sorting.masked.order = !!(vl->flags & VL_DESC);
			SetWindowDirty(w);
			break;

		case 4: case 5:/* Select sorting criteria dropdown menu */
			ShowDropDownMenu(w, _vehicle_sort_listing, vl->sort_type, 5, 0, 0);
			return;

		case 7: { /* Matrix to show vehicles */
			uint32 id_v = (e->click.pt.y - PLY_WND_PRC__OFFSET_TOP_WIDGET) / PLY_WND_PRC__SIZE_OF_ROW_BIG;

			if (id_v >= w->vscroll.cap) { return;} // click out of bounds

			id_v += w->vscroll.pos;
			{
				Vehicle *v;
				if (id_v >= vl->list_length) return; // click out of list bound
				v = GetVehicle(vl->sort_list[id_v].index);

				assert(v->owner == owner);
				
				switch (v->type)
				{
				case VEH_Train:
                    ShowTrainViewWindow(v);
					break;
				case VEH_Ship:
					ShowShipViewWindow(v);
					break;
				case VEH_Road:
					ShowRoadVehViewWindow(v);
					break;
				case VEH_Aircraft:
					ShowAircraftViewWindow(v);
					break;
				}
			}
		} break;
		}
	}	break;

	case WE_DROPDOWN_SELECT: /* we have selected a dropdown item in the list */
		if (vl->sort_type != e->dropdown.index) {
			// value has changed -> resort
			vl->flags |= VL_RESORT;
			vl->sort_type = e->dropdown.index;
			_sorting.masked.criteria = vl->sort_type;

			// enable 'Sort By' if a sorter criteria is chosen
			if (vl->sort_type != SORT_BY_UNSORTED)
				CLRBIT(w->disabled_state, 3);
		}
		SetWindowDirty(w);
		break;

	case WE_CREATE: /* set up resort timer */
		vl->sort_list = NULL;
		vl->flags = VL_REBUILD | (_sorting.train.order << (VL_DESC - 1));
		vl->sort_type = _sorting.masked.criteria;
		vl->resort_timer = DAY_TICKS * PERIODIC_RESORT_DAYS;
		break;

	case WE_DESTROY:
		free(vl->sort_list);
		break;

	case WE_TICK: /* resort the list every 20 seconds orso (10 days) */
		if (--vl->resort_timer == 0) {
			DEBUG(misc, 1) ("Periodic resort trains list player %d station %d",
				owner, station);
			vl->resort_timer = DAY_TICKS * PERIODIC_RESORT_DAYS;
			vl->flags |= VL_RESORT;
			SetWindowDirty(w);
		}
		break;

	case WE_RESIZE:
		/* Update the scroll + matrix */
		w->hscroll.cap += e->sizing.diff.x / 29;
		w->vscroll.cap += e->sizing.diff.y / PLY_WND_PRC__SIZE_OF_ROW_BIG;
		w->widget[7].unkA = (w->vscroll.cap << 8) + 1;
		break;
	}
}

static const Widget _player_vehicles_widgets[] = {
{   WWT_CLOSEBOX,   RESIZE_NONE,    14,     0,    10,     0,    13, STR_00C5,							STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,  RESIZE_RIGHT,    14,    11,   312,     0,    13, STR_VEHICLES,				STR_018C_WINDOW_TITLE_DRAG_THIS},
{  WWT_STICKYBOX,     RESIZE_LR,    14,   313,   324,     0,    13, 0x0,										STR_STICKY_BUTTON},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,    80,    14,    25, SRT_SORT_BY,						STR_SORT_ORDER_TIP},
{      WWT_PANEL,   RESIZE_NONE,    14,    81,   232,    14,    25, 0x0,										STR_SORT_CRITERIA_TIP},
{   WWT_CLOSEBOX,   RESIZE_NONE,    14,   233,   243,    14,    25, STR_0225,							STR_SORT_CRITERIA_TIP},
{      WWT_PANEL,  RESIZE_RIGHT,    14,   244,   324,    14,    25, 0x0,										STR_NULL},
{     WWT_MATRIX,     RESIZE_RB,    14,     0,   312,    26,   207, 0x701,									STR_883D_TRAINS_CLICK_ON_TRAIN_FOR},
{  WWT_SCROLLBAR,    RESIZE_LRB,    14,   313,   324,    26,   207, 0x0,										STR_0190_SCROLL_BAR_SCROLLS_LIST},
{      WWT_PANEL,    RESIZE_RTB,    14,     0,   312,   208,   219, 0x0,										STR_NULL},
{  WWT_RESIZEBOX,   RESIZE_LRTB,    14,   313,   324,   208,   219, 0x0,										STR_RESIZE_BUTTON},
{   WIDGETS_END},
};

static const WindowDesc _player_vehicles_desc = {
	-1, -1, 325, 220,
	WC_VEHICLES_LIST,0,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET | WDF_UNCLICK_BUTTONS | WDF_STICKY_BUTTON | WDF_RESIZABLE,
	_player_vehicles_widgets,
	PlayerVehiclesWndProc
};

void ShowPlayerVehicles(int player, int station, uint32 type_mask, uint32 cargo_mask, TileIndex xy)
{
	Window *w;

	w = AllocateWindowDescFront(&_player_vehicles_desc, (station << 16) | player);
	if (w) {
		w->listopt.cargo_mask = cargo_mask;
		w->listopt.veh_type_mask = type_mask;
		w->listopt.xy = xy;

		w->caption_color = player;
		w->hscroll.cap = 10;
		w->vscroll.cap = 5; // maximum number of vehicles shown
		w->widget[7].unkA = (w->vscroll.cap << 8) + 1;
		w->resize.step_height = PLY_WND_PRC__SIZE_OF_ROW_BIG;
		w->resize.step_width = 29;
		w->resize.height = 220 - (PLY_WND_PRC__SIZE_OF_ROW_BIG * 3); /* Minimum of 4 vehicles */
	} else {
		w = FindWindowById(WC_VEHICLES_LIST, (station << 16) | player);
		if (w) {
			vehiclelist_d *vl;
			vl = &WP(w, vehiclelist_d);
			vl->flags |= VL_REBUILD;
			w->listopt.cargo_mask = cargo_mask;
			w->listopt.veh_type_mask = type_mask;
			w->listopt.xy = xy;
			SetWindowDirty(w);
		}
	}
}
