#include "stdafx.h"
#include "openttd.h"

#include "command.h"
#include "gfx.h"
#include "map.h"
#include "order.h"
#include "saveload.h"
#include "station.h"
#include "tile.h"
#include "town.h"
#include "waypoint.h"
#include "table/sprites.h"
#include "table/strings.h"
#include "table/track_land.h"
#include "gui.h"

enum {
	/* Max waypoints: 64000 (8 * 8000) */
	WAYPOINT_POOL_BLOCK_SIZE_BITS = 3,       /* In bits, so (1 << 3) == 8 */
	WAYPOINT_POOL_MAX_BLOCKS      = 8000,

	MAX_WAYPOINTS_PER_TOWN        = 64,
};

/**
 * Called if a new block is added to the waypoint-pool
 */
static void WaypointPoolNewBlock(uint start_item)
{
	Waypoint *wp;

	FOR_ALL_WAYPOINTS_FROM(wp, start_item)
		wp->index = start_item++;
}

/* Initialize the town-pool */
MemoryPool _waypoint_pool = { "Waypoints", WAYPOINT_POOL_MAX_BLOCKS, WAYPOINT_POOL_BLOCK_SIZE_BITS, sizeof(Waypoint), &WaypointPoolNewBlock, 0, 0, NULL };

/* Create a new waypoint */
Waypoint *AllocateWaypoint(void)
{
	Waypoint *wp;

	FOR_ALL_WAYPOINTS(wp) {
		if (wp->xy == 0) {
			uint index = wp->index;

			memset(wp, 0, sizeof(Waypoint));
			wp->index = index;

			return wp;
		}
	}

	/* Check if we can add a block to the pool */
	if (AddBlockToPool(&_waypoint_pool))
		return AllocateWaypoint();

	return NULL;
}

/* Fetch a waypoint by tile */
Waypoint *GetWaypointByTile(TileIndex tile)
{
	Waypoint *wp;

	FOR_ALL_WAYPOINTS(wp) {
		if (wp->xy == tile)
			return wp;
	}

	return NULL;
}

/* Update the sign for the waypoint */
void UpdateWaypointSign(Waypoint *wp)
{
	Point pt = RemapCoords2(TileX(wp->xy) * 16, TileY(wp->xy) * 16);
	SetDParam(0, wp->index);
	UpdateViewportSignPos(&wp->sign, pt.x, pt.y - 0x20, STR_WAYPOINT_VIEWPORT);
}

/* Redraw the sign of a waypoint */
void RedrawWaypointSign(Waypoint *wp)
{
	MarkAllViewportsDirty(
		wp->sign.left - 6,
		wp->sign.top,
		wp->sign.left + (wp->sign.width_1 << 2) + 12,
		wp->sign.top + 48);
}

/* Update all signs */
void UpdateAllWaypointSigns(void)
{
	Waypoint *wp;

	FOR_ALL_WAYPOINTS(wp) {
		if (wp->xy)
			UpdateWaypointSign(wp);
	}
}

/* Set the default name for a waypoint */
void MakeDefaultWaypointName(Waypoint *wp)
{
	Waypoint *local_wp;
	bool used_waypoint[MAX_WAYPOINTS_PER_TOWN];
	int i;

	wp->town_index = ClosestTownFromTile(wp->xy, (uint)-1)->index;

	memset(used_waypoint, 0, sizeof(used_waypoint));

	/* Find an unused waypoint number belonging to this town */
	FOR_ALL_WAYPOINTS(local_wp) {
		if (wp == local_wp)
			continue;

		if (local_wp->xy && local_wp->string == STR_NULL && local_wp->town_index == wp->town_index)
			used_waypoint[local_wp->town_cn] = true;
	}

	/* Find an empty spot */
	for (i = 0; used_waypoint[i] && i < MAX_WAYPOINTS_PER_TOWN; i++) {}

	wp->string = STR_NULL;
	wp->town_cn = i;
}

/* Find a deleted waypoint close to a tile. */
static Waypoint *FindDeletedWaypointCloseTo(TileIndex tile)
{
	Waypoint *wp, *best = NULL;
	uint thres = 8, cur_dist;

	FOR_ALL_WAYPOINTS(wp) {
		if (wp->deleted && wp->xy) {
			cur_dist = DistanceManhattan(tile, wp->xy);
			if (cur_dist < thres) {
				thres = cur_dist;
				best = wp;
			}
		}
	}

	return best;
}

/** Convert existing rail to waypoint. Eg build a waypoint station over
 * piece of rail
 * @param x,y coordinates where waypoint will be built
 * @param p1 graphics for waypoint type, bit 8 signifies custom waypoint gfx (& 0x100)
 * @param p2 unused
 */
int32 CmdBuildTrainWaypoint(int x, int y, uint32 flags, uint32 p1, uint32 p2)
{
	TileIndex tile = TileVirtXY(x, y);
	Waypoint *wp;
	uint tileh;
	uint dir;

	SET_EXPENSES_TYPE(EXPENSES_CONSTRUCTION);

	/* if custom gfx are used, make sure it is within bounds */
	if (p1 > 0x100 + (uint)GetCustomStationsCount(STAT_CLASS_WAYP)) return CMD_ERROR;

	if (!IsTileType(tile, MP_RAILWAY) || ((dir = 0, _map5[tile] != 1) && (dir = 1, _map5[tile] != 2)))
		return_cmd_error(STR_1005_NO_SUITABLE_RAILROAD_TRACK);

	if (!CheckTileOwnership(tile))
		return CMD_ERROR;

	if (!EnsureNoVehicle(tile)) return CMD_ERROR;

	tileh = GetTileSlope(tile, NULL);
	if (tileh != 0) {
		if (!_patches.build_on_slopes || tileh & 0x10 || !(tileh & (0x3 << dir)) || !(tileh & ~(0x3 << dir)))
			return_cmd_error(STR_0007_FLAT_LAND_REQUIRED);
	}

	/* Check if there is an already existing, deleted, waypoint close to us that we can reuse. */
	wp = FindDeletedWaypointCloseTo(tile);
	if (wp == NULL) {
		wp = AllocateWaypoint();
		if (wp == NULL) return CMD_ERROR;

		wp->town_index = 0;
		wp->string = STR_NULL;
		wp->town_cn = 0;
	}

	if (flags & DC_EXEC) {
		ModifyTile(tile, MP_MAP5, RAIL_TYPE_WAYPOINT | dir);
		if (--p1 & 0x100) { // waypoint type 0 uses default graphics
			// custom graphics
			_map3_lo[tile] |= 16;
			_map3_hi[tile] = p1 & 0xff;
		}

		wp->deleted = 0;
		wp->xy = tile;
		wp->build_date = _date;
		InitializeWaypointStats(wp);

		if (wp->town_index == STR_NULL)
			MakeDefaultWaypointName(wp);

		UpdateWaypointSign(wp);
		RedrawWaypointSign(wp);
	}

	return _price.build_train_depot;
}

/* Internal handler to delete a waypoint */
static void DoDeleteWaypoint(Waypoint *wp)
{
	Order order;

	wp->xy = 0;

	order.type = OT_GOTO_WAYPOINT;
	order.station = wp->index;
	DeleteDestinationFromVehicleOrder(order);

	if (wp->string != STR_NULL)
		DeleteName(wp->string);

	RedrawWaypointSign(wp);
}

/* Daily loop for waypoints */
void WaypointsDailyLoop(void)
{
	Waypoint *wp;

	/* Check if we need to delete a waypoint */
	FOR_ALL_WAYPOINTS(wp) {
		if (wp->deleted && !--wp->deleted) {
			DoDeleteWaypoint(wp);
		}
	}
}

/* Remove a waypoint */
int32 RemoveTrainWaypoint(TileIndex tile, uint32 flags, bool justremove)
{
	Waypoint *wp;

	/* Make sure it's a waypoint */
	if (!IsTileType(tile, MP_RAILWAY) || !IsRailWaypoint(_map5[tile]))
		return CMD_ERROR;

	if (!CheckTileOwnership(tile) && !(_current_player == OWNER_WATER))
		return CMD_ERROR;

	if (!EnsureNoVehicle(tile))
		return CMD_ERROR;

	if (flags & DC_EXEC) {
		int direction = _map5[tile] & RAIL_WAYPOINT_TRACK_MASK;

		wp = GetWaypointByTile(tile);

		wp->deleted = 30; // let it live for this many days before we do the actual deletion.
		RedrawWaypointSign(wp);

		if (justremove) {
			ModifyTile(tile, MP_MAP5, 1<<direction);
			_map3_lo[tile] &= ~16;
			_map3_hi[tile] = 0;
		} else {
			DoClearSquare(tile);
			SetSignalsOnBothDir(tile, direction);
		}
	}

	return _price.remove_train_depot;
}

/** Delete a waypoint
 * @param x,y coordinates where waypoint is to be deleted
 * @param p1 unused
 * @param p2 unused
 */
int32 CmdRemoveTrainWaypoint(int x, int y, uint32 flags, uint32 p1, uint32 p2)
{
	TileIndex tile = TileVirtXY(x, y);
	SET_EXPENSES_TYPE(EXPENSES_CONSTRUCTION);
	return RemoveTrainWaypoint(tile, flags, true);
}

/** Rename a waypoint.
 * @param x,y unused
 * @param p1 id of waypoint
 * @param p2 unused
 */
int32 CmdRenameWaypoint(int x, int y, uint32 flags, uint32 p1, uint32 p2)
{
	Waypoint *wp;
	StringID str;

	if (!IsWaypointIndex(p1)) return CMD_ERROR;

	if (_cmd_text[0] != '\0') {
		str = AllocateNameUnique(_cmd_text, 0);
		if (str == 0)
			return CMD_ERROR;

		if (flags & DC_EXEC) {
			wp = GetWaypoint(p1);
			if (wp->string != STR_NULL)
				DeleteName(wp->string);

			wp->string = str;
			wp->town_cn = 0;

			UpdateWaypointSign(wp);
			MarkWholeScreenDirty();
		} else {
			DeleteName(str);
		}
	}	else {
		if (flags & DC_EXEC) {
			wp = GetWaypoint(p1);
			if (wp->string != STR_NULL)
				DeleteName(wp->string);

			MakeDefaultWaypointName(wp);
			UpdateWaypointSign(wp);
			MarkWholeScreenDirty();
		}
	}
	return 0;
}

/* This hacks together some dummy one-shot Station structure for a waypoint. */
Station *ComposeWaypointStation(TileIndex tile)
{
	Waypoint *wp = GetWaypointByTile(tile);
	static Station stat;

	stat.train_tile = stat.xy = wp->xy;
	stat.town = GetTown(wp->town_index);
	stat.string_id = wp->string == STR_NULL ? /* FIXME? */ 0 : wp->string;
	stat.build_date = wp->build_date;
	stat.class_id = 6;
	stat.stat_id = wp->stat_id;

	return &stat;
}

extern uint16 _custom_sprites_base;

/* Draw a waypoint */
void DrawWaypointSprite(int x, int y, int stat_id, int railtype)
{
	StationSpec *stat;
	uint32 relocation;
	DrawTileSprites *cust;
	DrawTileSeqStruct const *seq;
	uint32 ormod, img;

	ormod = SPRITE_PALETTE(PLAYER_SPRITE_COLOR(_local_player));

	x += 33;
	y += 17;

	/* draw default waypoint graphics of ID 0 */
	if (stat_id == 0) {
		const DrawTrackSeqStruct *dtss = _track_depot_layout_table[4];

		img = dtss++->image;
		if (img & 0x8000) img = (img & 0x7FFF) + railtype*TRACKTYPE_SPRITE_PITCH;
		DrawSprite(img, x, y);

		for (; dtss->image != 0; dtss++) {
			Point pt = RemapCoords(dtss->subcoord_x, dtss->subcoord_y, 0);
			img = dtss->image;
			if (img & 0x8000) img |= ormod;
			DrawSprite(img, x + pt.x, y + pt.y);
		}
		return;
	}

	stat = GetCustomStation(STAT_CLASS_WAYP, stat_id - 1);
	assert(stat);
	relocation = GetCustomStationRelocation(stat, NULL, 1);
	// emulate station tile - open with building
	// add 1 to get the other direction
	cust = &stat->renderdata[2];

	img = cust->ground_sprite;
	img += railtype * ((img < _custom_sprites_base) ? TRACKTYPE_SPRITE_PITCH : 1);

	if (img & 0x8000) img = (img & 0x7FFF);
	DrawSprite(img, x, y);

	foreach_draw_tile_seq(seq, cust->seq) {
		Point pt = RemapCoords(seq->delta_x, seq->delta_y, seq->delta_z);
		uint32 image = seq->image + relocation;

		DrawSprite((image&0x3FFF) | ormod, x + pt.x, y + pt.y);
	}
}

/* Fix savegames which stored waypoints in their old format */
void FixOldWaypoints(void)
{
	Waypoint *wp;

	/* Convert the old 'town_or_string', to 'string' / 'town' / 'town_cn' */
	FOR_ALL_WAYPOINTS(wp) {
		if (wp->xy == 0)
			continue;

		wp->town_index = ClosestTownFromTile(wp->xy, (uint)-1)->index;
		wp->town_cn = 0;
		if (wp->string & 0xC000) {
			wp->town_cn = wp->string & 0x3F;
			wp->string = STR_NULL;
		}
	}
}

void InitializeWaypoints(void)
{
	CleanPool(&_waypoint_pool);
	AddBlockToPool(&_waypoint_pool);
}

static const SaveLoad _waypoint_desc[] = {
	SLE_CONDVAR(Waypoint, xy, SLE_FILE_U16 | SLE_VAR_U32, 0, 5),
	SLE_CONDVAR(Waypoint, xy, SLE_UINT32, 6, 255),
	SLE_CONDVAR(Waypoint, town_index, SLE_UINT16, 12, 255),
	SLE_CONDVAR(Waypoint, town_cn, SLE_UINT8, 12, 255),
	SLE_VAR(Waypoint,string,		SLE_UINT16),
	SLE_VAR(Waypoint,deleted,						SLE_UINT8),

	SLE_CONDVAR(Waypoint, build_date, SLE_UINT16, 3, 255),
	SLE_CONDVAR(Waypoint, stat_id, SLE_UINT8, 3, 255),

//	SLE_CONDVAR(Waypoint,months_counted,	SLE_UINT16, 15, 255),

	SLE_END()
};

static const SaveLoad _stats_desc[] = {
	SLE_CONDVAR(StationStats,this_month,	SLE_UINT16, 15, 255),
	SLE_CONDVAR(StationStats,last_month,	SLE_UINT16, 15, 255),
	SLE_CONDVAR(StationStats,month_min,	SLE_UINT16, 15, 255),
	SLE_CONDVAR(StationStats,month_max,	SLE_UINT16, 15, 255),
	SLE_CONDVAR(StationStats,average,	SLE_UINT32, 15, 255),
	SLE_END()
};

static void Save_WAYP(void)
{
	Waypoint *wp;
//	int i;

	FOR_ALL_WAYPOINTS(wp) {
		if (wp->xy != 0) {
			SlSetArrayIndex(wp->index);
			SlObject(wp, _waypoint_desc);
		}
//		for (i = 0; i < STS_VEH_TYPES; i++) 
//			SlObject(&wp->vehicles[i], _stats_desc);

	}
}

static void Load_WAYP(void)
{
	int index;

	while ((index = SlIterateArray()) != -1) {
		Waypoint *wp;

		if (!AddBlockIfNeeded(&_waypoint_pool, index))
			error("Waypoints: failed loading savegame: too many waypoints");

		wp = GetWaypoint(index);
		SlObject(wp, _waypoint_desc);
		InitializeWaypointStats(wp);
	}
}

const ChunkHandler _waypoint_chunk_handlers[] = {
	{ 'CHKP', Save_WAYP, Load_WAYP, CH_ARRAY | CH_LAST},
};

void WaypointMonthlyLoop(void)
{
	Waypoint *wp;
	StationStats *sts;
	int i;

 	FOR_ALL_WAYPOINTS(wp) {
		// if we have stats for this station the number of counted months is >0
		if (wp->months_counted > 0)
		{
			// update vehicle-counts and min/max
			for (i = 0; i < STS_WP_TYPES / 2; i++) {
				sts = &wp->vehicles[i];
				if (sts->month_max == 0) sts->month_min = sts->this_month;
				sts->month_min = min(sts->month_min, sts->this_month);
				sts->month_max = max(sts->month_max, sts->this_month);
				sts->average = CalcNewAverage(sts->average, sts->this_month * STS_MULTIPLIER, wp->months_counted - 1);
				sts->last_month = sts->this_month;
				sts->this_month = 0;
			}
			if (wp->months_counted == (wp->months_counted /12) * 12)
			{ // do this once a year to have yearly stats
				for (i = STS_WP_TYPES / 2; i < STS_WP_TYPES; i++) {
					sts = &wp->vehicles[i];
					if (sts->month_max == 0) sts->month_min = sts->this_month;
					sts->month_min = min(sts->month_min, sts->this_month);
					sts->month_max = max(sts->month_max, sts->this_month);
					sts->average = CalcNewAverage(sts->average, sts->this_month * STS_MULTIPLIER, (wp->months_counted / 12) - 1);
					sts->last_month = sts->this_month;
					sts->this_month = 0;
				}
			}
			wp->months_counted++; // one more month counted
		}
		InvalidateWindow(WC_STATION_STATS, wp->index);
	}
}

void SearchVehiclesForWaypoint(Waypoint *wp)
{
	Vehicle *v;
	Order* ord;
	
	ord = NULL;
	if (wp->xy == 0) return;
	wp->veh_scheduled = 0;
	
	FOR_ALL_VEHICLES(v) {
	
		//Now run this stuff for sane vehicles only
		if ( (v->num_orders != 0) && (v->owner == GetTileOwner(wp->xy)) &&
			( (v->type == VEH_Train) && (v->subtype == TS_Front_Engine) ) )       //Trains (first engine, that contains the orders)
		{
			ord = v->orders;
			while(ord != NULL) {
				if (ord->station == wp->index && ord->type == OT_GOTO_WAYPOINT) {
					wp->veh_scheduled++;
					break;
				}
				ord = ord->next;
			}
		}
	} 
}

void InitializeWaypointStats(Waypoint *wp)
{
	StationStats *sts;

	wp->months_counted = 0;
	for (sts = wp->vehicles; sts != endof(wp->vehicles); sts++) {
		sts->last_month = 0;
		sts->this_month = 0;
		sts->month_min = 65535;
		sts->month_max = 0;
		sts->average = 0;
	}
	SearchVehiclesForWaypoint(wp);
}

void WaypointStatsWndProc(Window *w, WindowEvent *e);

Widget _waypoint_stats_widgets[] = {
{    WWT_TEXTBTN,   RESIZE_NONE,    14,     0,    10,     0,    13, STR_00C5, STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,   RESIZE_NONE,    14,    11,   288,     0,    13, STR_WAYPOINT_VIEWPORT, STR_018C_WINDOW_TITLE_DRAG_THIS},
{  WWT_STICKYBOX,   RESIZE_NONE,    14,   289,   300,     0,    13, 0x0,         STR_STICKY_BUTTON},
{      WWT_EMPTY,   RESIZE_NONE,     0,     0,     0,     0,     0, 0x0, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,   150,    14,    25, STR_RESET_STATISTICS, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,   151,   300,    14,    25, STR_TRAINS, 0x0},
{ WWT_PUSHTXTBTN,   RESIZE_NONE,    14,     0,   300,    26,    37, STR_STS_TOGGLE_MONTH_YEAR, 0x0},
{     WWT_IMGBTN,   RESIZE_NONE,    14,     0,   300,    38,   140, 0x0, 0x0},
{      WIDGETS_END},
};

WindowDesc _waypoint_view_stats = {
	-1, -1, 301, 141,
	WC_WAYPOINT_STATS, 0,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET | WDF_UNCLICK_BUTTONS | WDF_STICKY_BUTTON,
	_waypoint_stats_widgets,
	WaypointStatsWndProc
};

void DrawWaypointStatWindow(Window *w, Waypoint *wp)
{
	int i, j, x, y;
	
	//Get the Station name
	SetDParam(0, wp->index);
	//First draw the widgets
	DrawWindowWidgets(w);

	x = 10;
	y = 40;
	DrawString(x, y,	STR_SCHEDULED, 0);
	if (_show_yearly_stats) {
		DrawString(x, y + 20,	STR_STS_VEHICLES_LAST_YEAR, 0);
		DrawString(x, y + 32,	STR_STS_VEHICLES_THIS_YEAR, 0);
		SetDParam(0, max((wp->months_counted - 1) / 12 ,0));
		DrawString(x, y + 90,	STR_STS_YEARS_COUNTED_NUM, 0);
	} else {
		DrawString(x, y + 20,	STR_VEHICLES_MONTH, 0);
		DrawString(x, y + 32,	STR_VEHICLES_CURRENT, 0);
		SetDParam(0, max(wp->months_counted - 1,0));
		DrawString(x, y + 90,	STR_MONTHS_COUNTED_NUM, 0);
	}
	DrawString(x, y + 50,	STR_AVERAGE, 0);
	DrawString(x, y + 62,	STR_MINIMUM, 0);
	DrawString(x, y + 74,	STR_MAXIMUM, 0);

	x = 170;
	SetDParam(0, wp->veh_scheduled);
	DrawStringRightAligned(x, y, STR_NUMBER, 0);
	DrawStringRightAligned(290, y, STR_STS_NOT_SCHEDULED, 0);
	if (wp->veh_scheduled != 0) {
		CLRBIT(w->disabled_state, 5);
	} else {
		SETBIT(w->disabled_state, 5);
	}

	j = _show_yearly_stats ? 2 : 0;
	for (i = 0; i < STS_WP_TYPES / 2; i++)
	{
		SetDParam(0, wp->vehicles[i+j].last_month);
		DrawStringRightAligned(x + i * 120, y + 20, STR_NUMBER, 0);
		SetDParam(0, wp->vehicles[i+j].this_month);
		DrawStringRightAligned(x + i * 120, y + 32, STR_WHITE_NUMBER, 0);

		if (wp->months_counted > (_show_yearly_stats ? 12 : 1)) {
			SetDParam(0, wp->vehicles[i+j].average / STS_MULTIPLIER);
			DrawStringRightAligned(x + i * 120, y + 50, STR_SILVER_NUMBER, 0);
			SetDParam(0, wp->vehicles[i+j].month_min);
			DrawStringRightAligned(x + i * 120, y + 62, STR_ORANGE_NUMBER, 0);
			SetDParam(0, wp->vehicles[i+j].month_max);
			DrawStringRightAligned(x + i * 120, y + 74, STR_LTBLUE_NUMBER, 0);
		}
	}
}

void WaypointStatsWndProc(Window *w, WindowEvent *e)
{
	Waypoint *wp = GetWaypoint(w->window_number);
	switch(e->event)
	{
		case WE_TICK: {
//			static int counter = 0;
//			if (++counter % 40) return;
//			SearchVehiclesForWaypoint(wp, &w->custom[0]);
			w->custom[0] = wp->veh_scheduled;
			InvalidateWindow(WC_WAYPOINT_STATS, w->window_number);
			break;
		}
		case WE_PAINT: {
			DrawWaypointStatWindow(w, wp);
			break;
			}
		case WE_CLICK: {
			switch (e->click.widget)
			{
				case 4:			// Reset Statistics
					if (GetTileOwner(wp->xy) == _current_player) {
						InitializeWaypointStats(wp);
						InvalidateWindow(WC_WAYPOINT_STATS, w->window_number);
					}
					break;
				case 5:			//Trains
					ShowWaypointTrains(_current_player, wp->index, CARGO_MASK_ALL);
					break;
				case 6:
				_show_yearly_stats ^= 1;
				InvalidateWindow(WC_WAYPOINT_STATS, w->window_number);
				break;
			}
		} break;
		case WE_DESTROY: {
			DeleteWindowById(WC_TRAINS_LIST, GetTileOwner(wp->xy) + ( (wp->index + 1) << 8));
		} break;
	}
}

void ShowWaypointStatsWindow(Waypoint *wp)
{
	Window *w;
	byte color;

	w = AllocateWindowDescFront(&_waypoint_view_stats, wp->index);
	if (w) {
		color = GetTileOwner(wp->xy);
		if (color != 0x10)
			w->caption_color = color;
	}
}
