#ifndef COMMAND_HH
#define COMMAND_HH

#include <context.hh>

class command
{
	std::string name_;
	context params_;

public:
	command(const std::string& name = ""): name_(name) {}

	command(const std::string& name, const context& ctx)
		: name_(name), params_(ctx)
	{}

	command(const std::string& name, const command& cmd)
		: name_(name), params_(cmd.params_)
	{}

	inline command&
	operator()(const std::string&, const std::string&);

	inline const std::string&
	operator()(const std::string&) const;

	inline command&
	add(const std::string&, const std::string&);

	inline command&
	add(const std::string&, const std::string&, context::policy_t);

	inline command&
	add(const context&);

	inline command&
	add(const context&, context::policy_t);

	inline command&
	add(const command&);

	inline command&
	add(const command&, context::policy_t);

	inline const std::string&
	get(const std::string&) const;

	inline bool
	exists(const std::string&) const;

	inline command&
	remove(const std::string&);

	inline command&
	remove(const context&);

	inline command&
	remove(const command&);

	inline const std::string&
	name(void) const;

	inline command&
	name(const std::string& n);

	template <typename A> inline A for_each(A a);
	template <typename A> inline A for_each_pair(A a);
	template <typename A> inline A for_each_key(A a);
	template <typename A> inline A for_each_value(A a);
	template <typename A> inline A for_each(A a) const;
	template <typename A> inline A for_each_pair(A a) const;
	template <typename A> inline A for_each_key(A a) const;
	template <typename A> inline A for_each_value(A a) const;
};

command&
command::operator()(const std::string& k, const std::string& v)
{
	return add(k, v);
}

const std::string&
command::operator()(const std::string& k) const
{
	return get(k);
}

inline command&
operator+=(command& cmd, const context& newctx)
{
	return cmd.add(newctx);
}

inline command&
operator+=(command& cmd, const command& newcmd)
{
	return cmd.add(newcmd);
}

inline command&
operator-=(command& cmd, const std::string& key)
{
	return cmd.remove(key);
}

inline command&
operator-=(command& cmd, const context& otherctx)
{
	return cmd.remove(otherctx);
}

inline command&
operator-=(command& cmd, const command& othercmd)
{
	return cmd.remove(othercmd);
}

command&
command::add(const std::string& k, const std::string& v)
{
	return params_.add(k, v), *this;
}

command&
command::add(const std::string& k, const std::string& v, context::policy_t p)
{
	return params_.add(k, v, p), *this;
}

command&
command::add(const context& ctx)
{
	return params_.add(ctx), *this;
}

command&
command::add(const context& ctx, context::policy_t p)
{
	return params_.add(ctx, p), *this;
}

command&
command::add(const command& cmd)
{
	return params_.add(cmd.params_), *this;
}

command&
command::add(const command& cmd, context::policy_t p)
{
	return params_.add(cmd.params_, p), *this;
}

const std::string&
command::get(const std::string& k) const
{
	return params_.get(k);
}

bool
command::exists(const std::string& k) const
{
	return params_.exists(k);
}

command&
command::remove(const std::string& k)
{
	return params_.remove(k), *this;
}

command&
command::remove(const context& ctx)
{
	return params_.remove(ctx), *this;
}

command&
command::remove(const command& cmd)
{
	return remove(cmd.params_);
}

const std::string&
command::name(void) const
{
	return name_;
}

command&
command::name(const std::string& n)
{
	return (name_ = n), *this;
}

template <typename A> A command::for_each_pair(A a)
{
	return params_.for_each_pair(a);
}

template <typename A> A command::for_each(A a)
{
	return params_.for_each(a);
}

template <typename A> A command::for_each_key(A a)
{
	return params_.for_each_key(a);
}

template <typename A> A command::for_each_value(A a)
{
	return params_.for_each_value(a);
}

template <typename A> A command::for_each_pair(A a) const
{
	return params_.for_each_pair(a);
}

template <typename A> A command::for_each(A a) const
{
	return params_.for_each(a);
}

template <typename A> A command::for_each_key(A a) const
{
	return params_.for_each_key(a);
}

template <typename A> A command::for_each_value(A a) const
{
	return params_.for_each_value(a);
}

#endif // COMMAND_HH
