#ifndef EVAL_HH
#define EVAL_HH

#include <string>
#include <list>
#include <map>
#include <stdexcept>
#include <iosfwd>
#include <context.hh>

struct result
{
	enum {eof, string, list} t_;
	std::string str_;
	std::list<result> lst_;

	result(void): t_(eof) {}
	result(const std::string& str): t_(string), str_(str) {}
	result(const std::list<result>& lst): t_(list), lst_(lst) {}

	bool is_eof(void) const { return t_ == eof; }
	bool is_string(void) const { return t_ == string; }
	bool is_list(void) const { return t_ == list; }
};

struct parse_error: public std::runtime_error
{
	parse_error(const std::string& str): std::runtime_error(str) {}
};

struct eval_error: public std::runtime_error
{
	eval_error(const std::string& str): std::runtime_error(str) {}
};

struct evaluator;

typedef result (*func)(const std::list<result>&, const evaluator&);

struct evaluator
{
	std::map<std::string, func> macrotab_;
	std::map<std::string, func> functab_;
	std::map<std::string, context> ctxttab_;

	result
	operator()(const result&) const;

	bool is_special(const std::string& name) const
	{
		return macrotab_.find(name) != macrotab_.end();
	}

	bool is_function(const std::string& name) const
	{
		return macrotab_.find(name) != macrotab_.end();
	}

	func special(const std::string&) const;
	func function(const std::string&) const;
};


result
func_lookup(const std::list<result>&, const evaluator&);

void
print(std::ostream& os, const result& r);

result
read(std::istream& is);

#endif // EVAL_HH
