/*
 * xmlscope - minimalistic library for handling XML
 * Copyright (C) 2003 Alexey Voinov <voins@voins.program.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * xmlscope.hh: minimalistic XML declarations
 */
#ifndef _XMLSCOPE_HH
#define _XMLSCOPE_HH
#include <list>
#include <string>
#include <map>
#include <iosfwd>
#include <stdexcept>
#include <vector>
#include <boost/shared_ptr.hpp>

namespace xml
{
	struct node;
	typedef std::list<node> nodes_t;

	struct node
	{
		typedef std::vector<std::string> namespaces_t;
		typedef std::map<std::string, std::string> attributes_t;

		enum {normal, text} type;
		std::string name;
		int ns;

		attributes_t attrs;
		nodes_t nodes;
		boost::shared_ptr<namespaces_t> nses;

		node(boost::shared_ptr<namespaces_t>& nses_)
			: ns(0), nses(nses_)
		{
		}

		bool prop_exists(const std::string& name) const
		{
			return attrs.find(name) != attrs.end();
		}

		const std::string& prop_get(const std::string& name) const
		{
			attributes_t::const_iterator i = attrs.find(name);
			if(i == attrs.end())
				throw std::runtime_error("no such attribute");
			return i->second;
		}

		std::string value_get(void) const
		{
			std::string value;
			for(nodes_t::const_iterator i = nodes.begin();
					i != nodes.end(); ++i)
			{
				if(i->type == text)
					value += i->name;
				else if(i->type == normal)
					value += i->value_get();
			}
			return value;
		}

		int ns_exists(const std::string& name) const
		{
			namespaces_t::const_iterator ns =
				std::find(nses->begin(), nses->end(), name);
			return ns != nses->end();
		}

		int get_ns_index(const std::string& name) const
		{
			namespaces_t::const_iterator ns =
				std::find(nses->begin(), nses->end(), name);
			if(ns == nses->end())
				throw std::runtime_error("no such namespace");
			return ns - nses->begin();
		}

		const std::string& get_ns_name(int ndx) const
		{
			return nses->at(ndx);
		}

		int define_ns(const std::string& name)
		{
			namespaces_t::iterator ns =
				std::find(nses->begin(), nses->end(), name);
			if(ns == nses->end())
				ns = nses->insert(ns, name);
			return ns - nses->begin();
		}
	};

	void load(std::istream& is, nodes_t& result);
	void load(std::istream& is, nodes_t& result,
			boost::shared_ptr<node::namespaces_t>& nses);
	void process_xi(nodes_t& tree, const std::string& filename,
			const std::string& vroot);
	void process_ns(nodes_t& tree);
};

#endif // _XMLSCOPE_HH

