#include <bus/console.hh>
#include <bus/ctrlc.hh>
#include <utils/str.hh>

#include <map>
#include <istream>
#include <ostream>

namespace
{
	struct exit_bus {};

	struct exit_console {};

	typedef void (*cons_cmd)(std::istream& is, std::ostream& os,
							const std::string&);

	struct command_desc
	{
		cons_cmd code;
		std::string short_help;
		std::string full_help;

		command_desc(void) {}

		command_desc(cons_cmd code_, const std::string& short_help_,
				const std::string& full_help_)
			: code(code_),
			  short_help(short_help_),
			  full_help(full_help_)
		{}
	};

	typedef std::map<std::string, command_desc> commands_t;

	static commands_t commands;

	static void dispatch_int(std::istream& is, std::ostream& os,
			const std::string& name, const std::string& args)
	{
		if(name.empty()) return;
		commands_t::const_iterator i = commands.find(name);
		if(i == commands.end()) throw console_error("unknown command");
		i->second.code(is, os, args);
	}

	static void dispatch(std::istream& is, std::ostream& os,
						const std::string& cmd)
	{
		std::string::size_type pos = cmd.find_first_of(" \t");
		if(pos == std::string::npos)
			dispatch_int(is, os, cmd, "");
		else
			dispatch_int(is, os,
				cmd.substr(0, pos), cmd.substr(pos + 1));
	}

	static void full_help(std::ostream& os, const std::string& command)
	{
		commands_t::const_iterator i = commands.find(command);
		if(i == commands.end()) throw console_error("unknown command");
		os	<< "name   : " << command << "\n"
			<< "summary: " << i->second.short_help << "\n\n"
			<< i->second.full_help << std::endl;
	}

	struct short_help
	{
		short_help(std::ostream& os): os_(os) {}
		template <typename P> void operator()(const P& p)
		{
			os_ << p.second.short_help << std::endl;
		}
	private:
		std::ostream& os_;
	};

	static void help_cmd(std::istream&, std::ostream& os,
						const std::string& args)
	{
		if(args.empty())
			std::for_each(commands.begin(), commands.end(),
								short_help(os));
		else
			full_help(os, args);
	}

	void exit_cmd(std::istream&, std::ostream&, const std::string&)
	{
		throw exit_bus();
	}

	void run_cmd(std::istream&, std::ostream&, const std::string&)
	{
		throw exit_console();
	}
}

void execute_command(std::istream& is, std::ostream& os, const std::string& cmd)
{
	try
	{
		dispatch(is, os, alt::trim(cmd));
	}
	catch(const console_error& x)
	{
		os << "error: " << x.why() << std::endl;
	}
}

bool console(std::istream& is, std::ostream& os)
{
	try
	{
		while(is)
		{
			os << "\n] ";
			std::string buffer;
			getline(is, buffer);
			if(ctrlc_pressed())
			{
				os << std::endl;
				break;
			}
			else if(is.eof())
			{
				os << std::endl;
				is.clear();
				return true;
			}
			else
			{
				try
				{
					dispatch(is, os, alt::trim(buffer));
				}
				catch(const console_error& x)
				{
					os << "error: " << x.why() << std::endl;
				}
			}
		}
	}
	catch(const exit_bus&)
	{
		return false;
	}
	catch(const exit_console&)
	{
		return true;
	}
	return false;
}

void console_init(std::istream& is, std::ostream& os, int argc, char *argv[])
{
	commands["help"] = command_desc(help_cmd, "help [<command>]",
			"shows summary of all available commands if argument "
			"is ommited or show description of specified command");

	commands["exit"] = command_desc(exit_cmd, "exit",
			"stop all processing and shutdown WooBus");

	commands["run"] = command_desc(run_cmd, "run",
			"exit console and continue processing");

	commands["insert"] = command_desc(insert_cmd,
			"insert <point> <command>",
			"insert new module before <point>.\n"
			"<point> can be:\n"
			"  \"<begin>\"       - first module at bus\n"
			"  \"<end>\"         - one past last module at bus\n"
			"  module name     - name of module on bus\n"
			"  pid             - pid of running module\n"
			"  position in bus - position of module as shown by "
			"list\n"
			"<command> is used to start new process it can "
			"include list of space delimited arguments. Spaces "
			"can be protected with \\ character.");

	commands["append"] = command_desc(append_cmd,
			"append <point> <command>",
			"insert new module after <point>. See \"help insert\" "
			"for description of what <point> is.");

	commands["remove"] = command_desc(remove_cmd,
			"remove <point>",
			"remove module at <point>. See \"help insert\" "
			"for description of what <point> is.");

	commands["list"] = command_desc(list_cmd,
			"list",
			"list all current modules with short summaries.");

	commands["start"] = command_desc(start_cmd,
			"start <point>",
			"start module process. See \"help insert\" "
			"for description of what <point> is.");

	commands["stop"] = command_desc(stop_cmd,
			"stop <point>",
			"stop running module process. See \"help insert\" "
			"for description of what <point> is.");

	commands["status"] = command_desc(status_cmd,
			"status",
			"display current status of protocol");

	commands["setstate"] = command_desc(setstate_cmd,
			"setstate <state>",
			"enforce protocol state. possible values for <state> "
			"are: wait_msg, offer_msg, wait_accept, send_accept, "
			"read_msg, send_msg, send_eom");

	commands["setcurrent"] = command_desc(setcurrent_cmd,
			"setcurrent <point>",
			"set current module for protocol. See \"help insert\" "
			"for description of what <point> is. Special point "
			"value <none> can be used also.");

	commands["setneighbour"] = command_desc(setneighbour_cmd,
			"setneighbour <point>",
			"set current module for protocol. See \"help insert\" "
			"for description of what <point> is. Special point "
			"value <none> can be used also.");

	commands["load"] = command_desc(load_cmd,
			"load <filename>",
			"Load and execute console commands from file.");

	while(--argc)
		execute_command(is, os, *(++argv));
}
