#include <bus/bus.hh>
#include <bus/console.hh>
#include <utils/str.hh>

namespace
{
	struct show_mod_summary
	{
		show_mod_summary(std::ostream& os): os_(os), num_(1) {}
		void operator ()(const woobus::module& m)
		{
			os_ << num_++ << ": " << m.name();
			if(m.is_running())
				os_ << " [running " << m.pid() << "] ";
			else
				os_ << " [stopped] ";
			os_ << "\"" << m.command() << "\"" << std::endl;
		}
	private:
		std::ostream& os_;
		int num_;
	};

	struct match_mod_name_p
	{
		match_mod_name_p(const std::string& name): name_(name) {}
		bool operator ()(const woobus::module& m) const
		{
			return name_ == m.name();
		}
	private:
		std::string name_;
	};

	struct match_mod_num_p
	{
		match_mod_num_p(int pos): pos_(pos), num_(1) {}
		bool operator ()(const woobus::module&)
		{
			return num_++ == pos_;
		}
	private:
		int pos_, num_;
	};

	struct match_mod_pid_p
	{
		match_mod_pid_p(int pid): pid_(pid) {}
		bool operator ()(const woobus::module& m)
		{
			return m.is_running() && pid_ == m.pid();
		}
	private:
		int pid_;
	};
}

bus_iterator find_point(const std::string& point)
{
	if(point == "<end>") return modules_end();
	if(point == "<begin>") return modules_begin();

	bus_iterator i = modules_find(match_mod_name_p(point));
	if(i != modules_end()) return i;

	if (point.find_first_not_of("012345689") == std::string::npos)
	{
		int num_pt = ::atoi(point.c_str());
		if((i = modules_find(match_mod_pid_p(num_pt))) != modules_end())
			return i;
		if((i = modules_find(match_mod_num_p(num_pt))) != modules_end())
			return i;
	}

	throw console_error("cannot find point specified");
}

bus_iterator not_modules_end(const bus_iterator& i)
{
	if(i == modules_end()) throw console_error("invalid point");
	return i;
}

void insert_cmd(std::istream&, std::ostream&, const std::string& args_)
{
	std::string args(args_);
	std::string point = alt::split_on_unescaped_space(args);

	if(point.empty()) throw console_error("<point> is missing");
	if(args.empty()) throw console_error("<command> is missing");

	modules_insert(find_point(point), woobus::module(args));
}

void append_cmd(std::istream&, std::ostream&, const std::string& args_)
{
	std::string args(args_);
	std::string point = alt::split_on_unescaped_space(args);

	if(point.empty()) throw console_error("<point> is missing");
	if(args.empty()) throw console_error("<command> is missing");

	modules_append(find_point(point), woobus::module(args));
}

void remove_cmd(std::istream&, std::ostream&, const std::string& point)
{
	modules_remove(not_modules_end(find_point(point)));
}

void list_cmd(std::istream&, std::ostream& os, const std::string&)
{
	std::for_each(modules_begin(), modules_end(), show_mod_summary(os));
}
