#include <iostream>
#include <fstream>
#include <sstream>
#include <command_io.hh>
#include <matcher.hh>
#include <eval.hh>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

typedef std::pair<matcher, std::string> script_block_t;
typedef std::vector<script_block_t> script_t;

void
load_script(std::istream& is, script_t& script);

int
usage(void)
{
	std::cerr
		<< "usage: commander <interpreter> <script> init|convert"
		<< std::endl;
	return 1;
}

struct print_matcher
{
	void operator()(const matcher& label) const
	{
		std::cout << label.as_command();
	}
};

void
init(const script_t& script)
{
	std::for_each(script.begin(), script.end(), first(print_matcher()));
}


struct matching
{
	matching(const command& cmd, context& match)
		: cmd_(cmd), match_(match) {}

	bool operator()(const script_block_t& block)
	{
		match_ = context();
		return block.first.matches(cmd_, match_);
	}

private:
	const command& cmd_;
	context& match_;
};

void
run_shell(const std::string& shell, const std::string& input)
{
	pid_t child;
	int p[2];
	pipe(p);
	if(!(child = fork()))
	{
		close(p[1]);
		close(0); dup2(p[0], 0);
		execl(shell.c_str(), shell.c_str(), NULL);
		exit(1);
	}
	else
	{
		close(p[0]);
		size_t pos = 0;
		while(pos < input.size())
		{
			ssize_t res = write(p[1],
					input.data() + pos, input.size() - pos);
			if(res < 0) throw std::runtime_error(
						"cannot send script to shell");
			pos += res;
		}
		close(p[1]);
		int status;
		waitpid(child, &status, 0);
		if(WEXITSTATUS(status) != 0)
			throw std::runtime_error("shell returned error");
	}

}

void
convert(const std::string& shell, const script_t& script)
{
	context match;
	command cmd;
	std::cin >> cmd;
	script_t::const_iterator i = std::find_if(script.begin(), script.end(),
							matching(cmd, match));
	if(i != script.end())
	{
		evaluator eval;
		eval.functab_["lookup"] = func_lookup;
		eval.ctxttab_["match"] = match;
		std::istringstream is(i->second);
		std::ostringstream os;
		while(true)
		{
			result r = read(is);
			if(r.is_eof()) break;
			print(os, eval(r));
		}
		run_shell(shell, os.str());
	}
}

int
main(int argc, char *argv[])
{
	if(argc < 4) return usage();
	std::string shell(argv[1]);
	std::string action(argv[3]);
	script_t script;

	try
	{
		{
			std::ifstream is(argv[2]);
			load_script(is, script);
		}
		if(action == "init") init(script);
		else if(action == "convert") convert(shell, script);
		else throw std::runtime_error("invalid action");
	}
	catch(const std::runtime_error& x)
	{
		// FIXME: print in woo format
		std::cerr << "shit happened: " << x.what() << std::endl;
		return 2;
	}
	return 0;
}
