#include <iostream>
#include <string>
#include <vector>
#include <sys/types.h>
#include <regex.h>
#include <stdexcept>

#include <command.hh>
#include <matcher.hh>


inline bool
starts_with_and_space(const std::string& str, const std::string& start,
			std::string::size_type pos = 0)
{
	return str.size() >= start.size()
		&& str.compare(pos, start.size(), start) == 0
		&& (str.size() == start.size()
			|| (str.size() > start.size()
				&& (str[start.size()] == ' '
					|| str[start.size()] == '\t')));
}

typedef std::pair<matcher, std::string> script_block_t;
typedef std::vector<script_block_t> script_t;

bool
is_label(const std::string& line)
{
	return starts_with_and_space(line, "%.")
		|| starts_with_and_space(line, "%label");
}

bool
is_comment(const std::string& line)
{
	return starts_with_and_space(line, "%;")
		|| starts_with_and_space(line, "%comment");
}


char
char_before_is_esc(const std::string& line, std::string::size_type pos)
{
	if(line.empty() || pos == 0 || pos > line.size()) return false;
	if(pos != std::string::npos) return line[pos - 1] == '\\';
	return *line.rbegin() == '\\';
}


std::string
split_on_unescaped_space(std::istream& is, std::string& line)
{
	if(line.empty()) return "";
	std::string::size_type pos = 0;
	while(true)
	{
		pos = line.find_first_of(" \t", pos);
		if(!char_before_is_esc(line, pos)) break;
		if(pos == std::string::npos)
		{
			std::string buffer;
			getline(is, buffer);
			if(!is && buffer.empty())
				throw std::runtime_error("unexpected eof");
			pos = line.size();
			line += ' ' + buffer;
		}
		else pos++;
	}
	std::string::size_type pos2 = line.find_first_not_of(" \t", pos);
	std::string ans = line.substr(0, pos);
	line.erase(0, pos2);
	return ans;
}


std::string
split_on_eq_sign(std::string& line)
{
	if(line.empty()) return "";
	std::string::size_type pos = 0;
	while(true)
	{
		pos = line.find('=', pos);
		if(pos == std::string::npos)
			throw std::runtime_error("= is missing");
		if(!char_before_is_esc(line, pos)) break;
		pos++;
	}
	std::string ans = line.substr(0, pos);
	line.erase(0, pos + 1);
	return ans;
}


void
parse_label(std::istream& is, std::string line, matcher& label)
{
	label.clear();

	split_on_unescaped_space(is, line);
	std::string namere = split_on_unescaped_space(is, line);
	if(namere.empty()) throw std::runtime_error("empty label");

	label(namere);

	while(!line.empty())
	{
		std::string avalre = split_on_unescaped_space(is, line);
		std::string anamere = split_on_eq_sign(avalre);

		label(anamere, avalre);
	}
}


std::string
load_block(std::istream& is, std::string& block)
{
	block.clear();
	while(is)
	{
		std::string line;
		getline(is, line);
		if(!is) return "";
		if(is_label(line)) return line;
		if(is_comment(line)) continue;
		block += line + '\n';
	}
	return "";
}

void
load_script(std::istream& is, script_t& script)
{
	std::string block;
	std::string line;

	line = load_block(is, block);
	while(is && !line.empty())
	{
		matcher label;
		parse_label(is, line, label);
		line = load_block(is, block);
		script.push_back(make_pair(label, block));
	}
}
