#include <matcher.hh>
#include <command.hh>

#include <stdexcept>
#include <sstream>
#include <sys/types.h>
#include <regex.h>

matcher&
matcher::add(const std::string& re)
{
	if(!name_re.empty()) throw std::runtime_error("double name re");
	name_re = re;
	return *this;
}


matcher&
matcher::add(const std::string& n_re, const std::string& v_re)
{
	attr_re.push_back(make_pair(n_re, v_re));
	return *this;
}


void
matcher::clear(void)
{
	name_re.clear();
	attr_re.clear();
}

std::string
regex_error2string(int rc, const regex_t& re)
{
	std::vector<char> buffer(regerror(rc, &re, NULL, 0));
	regerror(rc, &re, &buffer[0], buffer.size());
	return std::string(&buffer[0], buffer.size());
}

regex_error::regex_error(int rc, const regex_t& re)
	: std::runtime_error(regex_error2string(rc, re))
{
}

std::string
num2string(int n)
{
	std::ostringstream os;
	os << n;
	return os.str();
}

struct fill_match
{
	fill_match(const std::string& data, context& match)
		: data_(data), match_(match) {}

	void operator()(const regmatch_t& rm)
	{
		if(rm.rm_so != -1)
			match_(num2string(match_.size()),
					std::string(data_, rm.rm_so, rm.rm_eo));
	}

private:
	const std::string& data_;
	context& match_;
};

bool
re_match(const std::string& retext, const std::string& data, context& match)
{
	int rc;
	regex_t re;
	if((rc = regcomp(&re, retext.c_str(), REG_EXTENDED)) != 0)
		throw regex_error(rc, re);
	std::vector<regmatch_t> raw_match(
			std::count(retext.begin(), retext.end(), '(') + 1);

	rc = regexec(&re, data.c_str(), raw_match.size(), &raw_match[0], 0);

	for_each(raw_match.begin(), raw_match.end(), fill_match(data, match));

	regfree(&re);
	return rc != REG_NOMATCH;
}

struct attr_match
{
	attr_match(const std::string& name_re, const std::string& val_re,
			context& match)
		: name_re_(name_re), val_re_(val_re),
			match_(match), answer_(false)
	{}

	void operator()(const std::string& name, const std::string& val)
	{
		answer_ = answer_
			|| (re_match(name_re_, name, match_)
					&& re_match(val_re_, val, match_));
	}

	operator bool() const { return answer_; }

private:
	const std::string& name_re_;
	const std::string& val_re_;
	context& match_;
	bool answer_;
};

struct not_re_match
{
	not_re_match(const command& cmd, context& match)
		: cmd_(cmd), match_(match) {}

	bool operator()(const std::pair<std::string, std::string>& p)
	{
		return !cmd_.for_each(
				attr_match(p.first, p.second, match_));
	}

private:
	const command& cmd_;
	context& match_;
};

bool
matcher::matches(const command& cmd, context& match) const
{
	return re_match(name_re, cmd.name(), match)
			&& (find_if(attr_re.begin(), attr_re.end(),
					not_re_match(cmd, match))
						== attr_re.end());
}


struct add_to_command
{
	add_to_command(command& cmd): cmd_(cmd) {}

	void operator()(const std::string& name_re, const std::string& val_re)
	{
		cmd_(name_re, val_re);
	}

private:
	command& cmd_;
};

command
matcher::as_command(void) const
{
	command cmd(name_re);
	std::for_each(attr_re.begin(), attr_re.end(),
		unpair(add_to_command(cmd)));
	return cmd;
}
