#include <module.hh>
#include <utils/os.hh>
#include <utils/fdstream.hh>

#include <iostream>

namespace
{
	static std::string name_from_command(const std::string& command)
	{

		std::string::size_type pos = command.find_first_of(" \t");
		std::string name;

		if(pos == std::string::npos) name = command;
		else name = command.substr(0, pos);

		pos = name.rfind("/");
		if(pos == std::string::npos) return name;
		else return name.substr(pos + 1);
	}
}

struct channel
{
	channel(int in, int out): is(out), os(in) {}

	alt::ifdstream is;
	alt::ofdstream os;
};

module::module(const std::string& command)
	: name_(name_from_command(command)), command_(command),
	  pid_(0), in_(-1), out_(-1), io_(0)
{
}

module::module(const std::string& name, const std::string& command)
	: name_(name), command_(command),
	  pid_(0), in_(-1), out_(-1), io_(0)
{
}

module::~module(void)
{
	shutdown_io();
}

void module::startup_io(void)
{
	if(!io_)
	{
		if(!is_running())
			throw err::not_running("cannot start up io subsys");
		if(in_ < 0 || out_ < 0) throw err::io("no active fd");
		io_ = new channel(in_, out_);
	}
}

void module::shutdown_io(void)
{
	if(io_) { delete io_; io_ = 0; }
	if(in_ >= 0) { ::close(in_); in_ = -1; }
	if(out_ >= 0) { ::close(out_); out_ = -1; }
}

bool module::is_running(void) const
{
	if(pid_)
		try
		{
			return alt::wait(pid_, alt::nohang).is_running();
		} catch(...) {}
	return false;
}

void module::start(const std::string& args)
{
	if(is_running()) return;
	shutdown_io();

	int stdin_in = -1, stdout_out = -1;
	try
	{
		alt::pipe(stdin_in, in_);
		alt::pipe(out_, stdout_out);
		if(!(pid_ = alt::fork()))
		{
			try
			{
				alt::setpgid();
				::close(in_); ::close(out_);
				::close(0); alt::dup2(stdin_in, 0);
				::close(1); alt::dup2(stdout_out, 1);
				if(args.empty()) alt::exec(command_);
				else alt::exec(command_ + " " + args);
			}
			catch(...)
			{
			}
			exit(1);
		}
		::close(stdin_in); ::close(stdout_out);
		startup_io();
	}
	catch(...)
	{
		if(stdin_in >= 0) close(stdin_in);
		if(stdout_out >= 0) close(stdout_out);
		if(in_ >= 0) close(in_);
		if(out_ >= 0) close(out_);
		throw;
	}
}

void module::stop(void)
{
	if(!is_running()) return;
	alt::kill(pid_, SIGINT);
	alt::wait(pid_);
	shutdown_io();
	pid_ = 0;
}

std::istream& module::is(void)
{
	if(io_) return io_->is;
	throw err::io("io subsys is not active");
}

std::ostream& module::os(void)
{
	if(io_) return io_->os;
	throw err::io("io subsys is not active");
}
