#include <lowlevelio.hh>
#include <sys/select.h>
#include <errno.h>
#include <unistd.h>

fd_set *descriptor::fdset(fd_set *fds) const
{
	FD_SET(fd_, fds);
	return fds;
}

bool descriptor::has_data(void) const
{
	struct timeval tv = {0, 0};

	fd_set fds;
	FD_ZERO(&fds);

	int count = select(fd_ + 1, fdset(&fds), 0, 0, &tv);
	if(count < 0) throw err::sys(errno);

	return count != 0;
}

bool descriptor::can_write(void) const
{
	struct timeval tv = {0, 0};

	fd_set fds;
	FD_ZERO(&fds);

	int count = select(fd_ + 1, 0, fdset(&fds), 0, &tv);
	if(count < 0) throw err::sys(errno);

	return count != 0;
}

ssize_t descriptor::read(char *buf, size_t len) const
{
	int count = ::read(fd_, buf, len);
	if(count < 0) throw err::sys(errno);
	if(count == 0) throw err::eof();
	return count;
}

ssize_t descriptor::write(const char *buf, size_t len) const
{
	int count = ::write(fd_, buf, len);
	if(count < 0) throw err::sys(errno);
	return count;
}

struct setup_fds
{
	setup_fds(fd_set *fds, int& maxfd): fds_(fds), maxfd_(maxfd) {}
	void operator ()(const descriptor& fd)
	{
		fd.fdset(fds_);
		maxfd_ = fd.maxfd(maxfd_);
	}
	fd_set *fds_;
	int& maxfd_;
};

bool select_read(const std::list<descriptor>& lst, struct timeval& tv)
{
	fd_set fds;
	int maxfd = -1;
	FD_ZERO(&fds);

	std::for_each(lst.begin(), lst.end(), setup_fds(&fds, maxfd));
	int count = select(maxfd + 1, &fds, 0, 0, &tv);

	if(count < 0) throw err::sys(errno);
	return count != 0;
}
