#include <dialog.hh>
#include <command_set.hh>
#include <xmlscope.hh>

#include <iostream>
#include <fstream>

struct parse_attr
{
	parse_attr(command& cmd): cmd_(cmd) {}

	void operator()(const std::string& name, const std::string& value)
	{
		if(name == "name") return;
		cmd_(name, value);
	}

private:
	command& cmd_;
};

struct parse_param
{
	parse_param(command& cmd): cmd_(cmd) {}

	void operator()(xml::node& n) const
	{
		if(n.type != xml::node::normal || n.name != "param") return;
		if(!n.prop_exists("name")) return;
		std::string name = n.prop_get("name");

		std::string value;
		if(n.prop_exists("value"))
			value = n.prop_get("value");
		else
			value = n.value_get();

		cmd_(name, value);
	}

private:
	command& cmd_;
};

struct parse_command
{
	parse_command(command_set& c): c_(c), ndx_(-1) {}

	void operator()(xml::node& n)
	{
		if(n.type != xml::node::normal || n.name != "command") return;
		if(!n.prop_exists("name")) return;

		command cmd(n.prop_get("name"));

		std::for_each(n.attrs.begin(), n.attrs.end(),
					unpair<parse_attr>(parse_attr(cmd)));

		std::for_each(n.nodes.begin(), n.nodes.end(),
					parse_param(cmd));

		if(ndx_ < 0) ndx_ = c_.add(cmd);
		else c_.add(ndx_, cmd);
	}

	operator long() const { return ndx_; }

private:
	command_set& c_;
	long ndx_;
};

struct generate_id
{
	generate_id(void): id_('a') {}
	std::string operator()(const std::string& prefix)
	{
		return prefix + id_++;
	}

private:
	char id_;
};

struct parse_dialog
{
	parse_dialog(dialog& d, command_set& c): d_(d), c_(c){}

	void operator()(xml::node& n)
	{
		if(n.type == xml::node::normal && n.name == "label")
		{
			std::string value;
			if(n.prop_exists("value"))
				value = n.prop_get("value");
			else
				value = n.value_get();
			d_.add_label(value);
		}
		else if(n.type == xml::node::normal && n.name == "edit")
		{
			std::string value, id;
			if(n.prop_exists("value"))
				value = n.prop_get("value");
			else
				value = n.value_get();
			if(n.prop_exists("id"))
				id = n.prop_get("id");
			else
				id = ids_("edit/");

			d_.add_edit(id, value);
		}
		else if(n.type == xml::node::normal && n.name == "button")
		{
			std::string id, label;
			if(n.prop_exists("label"))
				label = n.prop_get("label");
			else
				label = n.value_get();
			if(n.prop_exists("id"))
				id = n.prop_get("id");
			else
				id = ids_("button/");

			long cmdidx = std::for_each(
					n.nodes.begin(), n.nodes.end(),
					parse_command(c_));

			d_.add_button(id, label, cmdidx);
		}
	}

private:
	dialog& d_;
	command_set& c_;
	generate_id ids_;
};

bool
load_dialog(const std::string& filename, dialog& d, command_set& c)
{
	std::ifstream is(filename.c_str());
	if(!is) return false;

	xml::nodes_t data;
	xml::load(is, data);
	xml::process_ns(data);
	xml::process_xi(data, "./", "/");

	std::for_each(data.begin(), data.end(), parse_dialog(d, c));
	return true;
}
