/*
 * xmlscope - minimalistic library for handling XML
 * Copyright (C) 2003 Alexey Voinov <voins@voins.program.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * DESCRIPTION: partial xincludes support
 */
#include <xmlscope.hh>

#include <fstream>
#include <iterator>

namespace
{
	const char *xi_namespace = "http://www.w3.org/2001/XInclude";

	bool inet_uri(const std::string& uri)
	{
		return uri.compare(0, 7, "http://") == 0 ||
			uri.compare(0, 6, "ftp://") == 0;
	}

	class includer
	{
		std::list<xml::node>& tree_;
		std::string base_;
		std::string vroot_;
		int xi_ns_;
	public:
		includer(
			std::list<xml::node>& tree,
			const std::string& filename,
			const std::string& vroot,
			int xi_ns)
				: tree_(tree), vroot_(vroot), xi_ns_(xi_ns)
		{
			size_t pos = filename.rfind('/');
			if(pos != std::string::npos)
				base_ = filename.substr(0, pos + 1);

			if(vroot[vroot.size() - 1] == '/')
				vroot_ = vroot.substr(0, vroot.size() - 1);
		}

		std::string get_filename(const std::string& uri);
		std::list<xml::node>::iterator fallback(
				std::list<xml::node>::iterator i);
		std::list<xml::node>::iterator replace_node(
				std::list<xml::node>::iterator i);
	};

	std::string includer::get_filename(const std::string& uri)
	{
		if(uri.compare(0, 7, "file://") == 0)
			return get_filename(uri.substr(7));

		if(uri[0] == '/') return vroot_ + uri;
		else if(!inet_uri(uri)) return base_ + uri;
		return "";
	}

	struct fallback_node_p: std::unary_function<xml::node, bool>
	{
		fallback_node_p(int xi_ns) : xi_ns_(xi_ns) {}

		bool operator () (const xml::node& n)
		{
			return n.type == xml::node::normal
				&& n.ns == xi_ns_
				&& n.name == "fallback";
		}

		int xi_ns_;
	};

	std::list<xml::node>::iterator includer::fallback(
			std::list<xml::node>::iterator i)
	{
		std::list<xml::node>::iterator j =
			std::find_if(i->nodes.begin(), i->nodes.end(),
					fallback_node_p(xi_ns_));
		if(j != i->nodes.end())
		{
			xml::process_xi(j->nodes, base_, vroot_);
			tree_.insert(i, j->nodes.begin(), j->nodes.end());
		}
		return tree_.erase(i);
	}

	std::list<xml::node>::iterator includer::replace_node(
			std::list<xml::node>::iterator i)
	{
		if(i->attrs["href"].empty()) return fallback(i);

		std::string newfile = get_filename(i->attrs["href"]);
		if(inet_uri(newfile)) return fallback(i);

		std::ifstream is(newfile.c_str());
		if(!is)	return fallback(i);

		if(i->attrs["mode"].empty() || i->attrs["mode"] == "xml")
		{
			std::list<xml::node> newtree;
			xml::load(is, newtree, i->nses);
			xml::process_ns(newtree);
			xml::process_xi(newtree, base_, vroot_);

			tree_.insert(i, newtree.begin(), newtree.end());
		}
		else if(i->attrs["mode"] == "text")
		{
			xml::node n(i->nses);
			n.type = xml::node::text;
			n.name = std::string(
					std::istream_iterator<char>(is),
					std::istream_iterator<char>());

			tree_.insert(i, n);
		}
		else return fallback(i);
		return tree_.erase(i);
	}
};

namespace xml
{
	void process_xi(
			xml::nodes_t& tree,
			const std::string& filename,
			const std::string& vroot)
	{
		if(tree.empty()) return;
		if(!tree.begin()->ns_exists(xi_namespace)) return;

		int xi_ns = tree.begin()->get_ns_index(xi_namespace);
		includer obj(tree, filename, vroot, xi_ns);
		for(xml::nodes_t::iterator i = tree.begin();
				i != tree.end(); ++i)
		{
			if(i->type == xml::node::normal
				&& i->ns == xi_ns
				&& i->name == "include")
					i = obj.replace_node(i);
			else if(!i->nodes.empty())
				process_xi(i->nodes, filename, vroot);
		}
	}
};

