/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2005 ALT Linux Ltd.
 *  Copyright (c) 2005 Alexey Voinov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <rpm/rpmlib.h>
#include <rpm/rpmcli.h>
#include <fcntl.h>
#include <libguile.h>

typedef Header rpmHeader;

static scm_t_bits rpmdb_tag;
static scm_t_bits rpmts_tag;


SCM scm_rpm_open_db(void)
{
	rpmdb db;

	rpmReadConfigFiles(0, 0);
	if(rpmdbOpen("", &db, O_RDWR, 0644) != 0)
		scm_error(scm_misc_error_key, "rpm-open-db", "could not open RPM database", SCM_BOOL_F, SCM_BOOL_F);
	SCM_RETURN_NEWSMOB(rpmdb_tag, db);
}

SCM scm_rpm_create_transaction(SCM db)
{
	SCM_ASSERT(SCM_SMOB_PREDICATE(rpmdb_tag, db), db, SCM_ARG1, "rpm-create-transaction");

	SCM_RETURN_NEWSMOB(rpmts_tag, rpmtransCreateSet((rpmdb)SCM_SMOB_DATA(db), ""));
}

SCM scm_rpm_dep_order(SCM ts)
{
	SCM_ASSERT(SCM_SMOB_PREDICATE(rpmts_tag, ts), ts, SCM_ARG1, "rpm-dep-order");

	if(rpmdepOrder((rpmTransactionSet)SCM_SMOB_DATA(ts)) > 0)
		scm_error(scm_misc_error_key, "rpm-dep-order", "ordering failed", SCM_BOOL_F, SCM_BOOL_F);

	return SCM_UNSPECIFIED;
}

static void *scm_rpm_show_progress(const void *arg, const rpmCallbackType what,
		const unsigned long amount, const unsigned long total,
		const void *pkgKey, void *data)
{
	SCM func = data;
	Header h = (Header)arg;
	const char *filename = pkgKey;
	static FD_t fd = NULL;
	char *s;

	switch(what)
	{
		case RPMCALLBACK_INST_OPEN_FILE:
			if(!filename || !*filename) return NULL;
			fd = Fopen(filename, "r.ufdio");
			if(fd) fd = fdLink(fd, "persist (showProgress)");
			return fd;

		case RPMCALLBACK_INST_CLOSE_FILE:
			fd = fdFree(fd, "persist (showProgress)");
			if(fd)
			{
				FClose(fd);
				fd = NULL;
			}
			return NULL;

		case RPMCALLBACK_INST_START:
			s = headerSprintf(h, "%{NAME}", rpmTagTable, rpmHeaderFormats, NULL);
			scm_call_4(func, SCM_MAKINUM(what), scm_makfrom0str(s), SCM_MAKINUM(0), SCM_MAKINUM(0));
			s = _free(s);
			return NULL;

		case RPMCALLBACK_TRANS_PROGRESS:
		case RPMCALLBACK_INST_PROGRESS:
			scm_call_4(func, SCM_MAKINUM(what), SCM_BOOL_F, SCM_MAKINUM(amount), SCM_MAKINUM(total));
			return NULL;

		case RPMCALLBACK_TRANS_START:
		case RPMCALLBACK_TRANS_STOP:
			scm_call_4(func, SCM_MAKINUM(what), SCM_BOOL_F, SCM_MAKINUM(0), SCM_MAKINUM(0));
			return NULL;

		case RPMCALLBACK_UNINST_PROGRESS:
		case RPMCALLBACK_UNINST_START:
		case RPMCALLBACK_UNINST_STOP:
		case RPMCALLBACK_UNPACK_ERROR:
		case RPMCALLBACK_CPIO_ERROR:
		default:
			break;
	}
	return NULL;
}

SCM scm_rpm_run_transaction(SCM ts, SCM progress)
{
	rpmProblemSet probs;
	SCM_ASSERT(SCM_SMOB_PREDICATE(rpmts_tag, ts), ts, SCM_ARG1, "rpm-dep-order");

	if(rpmRunTransactions((rpmTransactionSet)SCM_SMOB_DATA(ts),
			      rpmShowProgress, (void*)progress, NULL,
			      &probs, 0, 0) > 0)
		scm_error(scm_misc_error_key, "rpm-run-transaction", "error while running transaction", SCM_BOOL_F, SCM_BOOL_F);

}

SCM scm_rpm_close_db(SCM db)
{
	SCM_ASSERT(SCM_SMOB_PREDICATE(rpmdb_tag, db), db, SCM_ARG1, "rpm-create-transaction");

	rpmdbClose((rpmdb)SCM_SMOB_DATA(db));
}

static void scm_rpm_install_internal(const char *package, rpmTransactionSet ts, int upgrade)
{
	FD_t fd;
	rpmHeader hdr;

	fd = Fopen(package, "r.ufdio");
	if (!fd)
		scm_error(scm_misc_error_key, "rpm-close-db", "could not open file", SCM_BOOL_F, SCM_BOOL_F);
	if(rpmReadPackageHeader(fd, &hdr, 0, NULL, NULL))
	{
		Fclose(fd);
		scm_error(scm_misc_error_key, "rpm-close-db", "could not read package header", SCM_BOOL_F, SCM_BOOL_F);
	}
	if(rpmtransAddPackage(ts, hdr, NULL, package, upgrade, 0))
	{
		headerFree(hdr);
		Fclose(fd);
		scm_error(scm_misc_error_key, "rpm-close-db", "could not install package", SCM_BOOL_F, SCM_BOOL_F);
	}
	headerFree(hdr);
	Fclose(fd);
}

SCM scm_rpm_trans_install(SCM ts, SCM package)
{
	SCM_ASSERT(SCM_SMOB_PREDICATE(rpmts_tag, ts), ts, SCM_ARG1, "rpm-trans-install");
	SCM_ASSERT(SCM_STRINGP(package), package, SCM_ARG2, "rpm-trans-install");

	scm_rpm_install_internal(SCM_STRING_CHARS(package),(rpmTransactionSet)SCM_SMOB_DATA(ts), 0);

	return SCM_UNSPECIFIED;
}

SCM scm_rpm_trans_upgrade(SCM ts, SCM package)
{
	SCM_ASSERT(SCM_SMOB_PREDICATE(rpmts_tag, ts), ts, SCM_ARG1, "rpm-trans-upgrade");
	SCM_ASSERT(SCM_STRINGP(package), package, SCM_ARG2, "rpm-trans-upgrade");

	scm_rpm_install_internal(SCM_STRING_CHARS(package),(rpmTransactionSet)SCM_SMOB_DATA(ts), 1);

	return SCM_UNSPECIFIED;
}

/*SCM scm_rpm_trans_erase(SCM ts, SCM package)
{
	rpmdbMatchIterator mi;
	mi = rpmdbInitIterator(db, RPMDBI_LABEL, package, 0);
	while((hdr = rpmdbNextIterator(mi)) != NULL)
	{
		unsigned int recOffset = rpmdbGetIteratorOffset(mi);
		if(recOffset)
		{
			if(rpmtransRemovePackage(ts, recOffset))
			{
				scm_error(...);
			}
		}
	}
	rpmdbFreeIterator(mi);

	return SCM_UNSPECIFIED;
}*/

void scm_init_rpm(void)
{
	rpmdb_tag = scm_make_smob_type("rpmdb", 0);
	rpmts_tag = scm_make_smob_type("rpmts", 0);

	scm_c_define_gsubr("rpm-trans-install", 2, 0, 0,
			   (SCM (*)())scm_rpm_trans_install);
	scm_c_define_gsubr("rpm-trans-upgrade", 2, 0, 0,
			   (SCM (*)())scm_rpm_trans_upgrade);
	scm_c_define_gsubr("rpm-open-db", 0, 0, 0,
			   (SCM (*)())scm_rpm_open_db);
	scm_c_define_gsubr("rpm-create-transaction", 1, 0, 0,
			   (SCM (*)())scm_rpm_create_transaction);
	scm_c_define_gsubr("rpm-dep-order", 1, 0, 0,
			   (SCM (*)())scm_rpm_dep_order);
	scm_c_define_gsubr("rpm-close-db", 1, 0, 0,
			   (SCM (*)())scm_rpm_close_db);
	scm_c_define_gsubr("rpm-run-transaction", 2, 0, 0,
			   (SCM (*)())scm_rpm_run_transaction);

}
