/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <iostream>
#include <string>
#include <vector>
#include <sys/types.h>
#include <regex.h>
#include <stdexcept>

#include <utils/str.hh>
#include <command.hh>
#include <matcher.hh>


typedef std::pair<matcher, std::string> script_block_t;
typedef std::vector<script_block_t> script_t;

bool
is_label(const std::string& line)
{
	return alt::starts_with_and_space(line, "%.")
		|| alt::starts_with_and_space(line, "%label");
}

bool
is_comment(const std::string& line)
{
	return alt::starts_with_and_space(line, "%;")
		|| alt::starts_with_and_space(line, "%comment");
}


char
char_before_is_esc(const std::string& line, std::string::size_type pos)
{
	if(line.empty() || pos == 0 || pos > line.size()) return false;
	if(pos != std::string::npos) return line[pos - 1] == '\\';
	return *line.rbegin() == '\\';
}


std::string
split_on_unescaped_space(std::istream& is, std::string& line)
{
	if(line.empty()) return "";
	std::string::size_type pos = 0;
	while(true)
	{
		pos = line.find_first_of(" \t", pos);
		if(!char_before_is_esc(line, pos)) break;
		if(pos == std::string::npos)
		{
			std::string buffer;
			getline(is, buffer);
			if(!is && buffer.empty())
				throw std::runtime_error("unexpected eof");
			pos = line.size();
			line += ' ' + buffer;
		}
		else pos++;
	}
	std::string::size_type pos2 = line.find_first_not_of(" \t", pos);
	std::string ans = line.substr(0, pos);
	line.erase(0, pos2);
	return ans;
}


std::string
split_on_eq_sign(std::string& line)
{
	if(line.empty()) return "";
	std::string::size_type pos = 0;
	while(true)
	{
		pos = line.find('=', pos);
		if(pos == std::string::npos)
			throw std::runtime_error("= is missing");
		if(!char_before_is_esc(line, pos)) break;
		pos++;
	}
	std::string ans = line.substr(0, pos);
	line.erase(0, pos + 1);
	return ans;
}


void
parse_label(std::istream& is, std::string line, matcher& label)
{
	label.clear();

	split_on_unescaped_space(is, line);
	std::string namere = split_on_unescaped_space(is, line);
	if(namere.empty()) throw std::runtime_error("empty label");

	label(namere);

	while(!line.empty())
	{
		std::string avalre = split_on_unescaped_space(is, line);
		std::string anamere = split_on_eq_sign(avalre);

		label(anamere, avalre);
	}
}


std::string
load_block(std::istream& is, std::string& block)
{
	block.clear();
	while(is)
	{
		std::string line;
		getline(is, line);
		if(!is) return "";
		if(is_label(line)) return line;
		if(is_comment(line)) continue;
		block += line + '\n';
	}
	return "";
}

void
load_script(std::istream& is, script_t& script)
{
	std::string block;
	std::string line;

	line = load_block(is, block);
	while(is && !line.empty())
	{
		matcher label;
		parse_label(is, line, label);
		line = load_block(is, block);
		script.push_back(make_pair(label, block));
	}
}
