/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Alexey Voinov
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <iostream>
#include <fstream>

#include <utils/str.hh>
#include <utils/os.hh>

#include <eval.hh>
#include <matcher.hh>
#include <command_io.hh>

typedef std::pair<matcher, std::string> int_script_block_t;
typedef std::vector<int_script_block_t> script_blocks_t;
typedef std::pair<matcher,script_blocks_t> ext_script_block_t;
typedef std::vector<ext_script_block_t> script_t;

int
usage(void)
{
	std::cerr
		<< "usage: commander <interpreter> <script> init|unconvert"
		<< std::endl;
	return 1;
}

std::string::size_type slash_tail(const std::string& str)
{
	if (str.empty()) return str.size()-1;
	std::string::size_type pos;
	for(pos = str.size()-1;pos;--pos)
		if (str[pos] != '\\') break;
	return pos;
}

bool join_lines(std::istream& is,std::string& line)
{
	line.clear();
	std::string str;
	//join strings with "\" at the end
	while(std::getline(is,str))
	{
		std::string::size_type slash_pos = slash_tail(str);
		std::string::size_type sl_num = (str.size()-slash_pos-1);

		line += str.substr(0,slash_pos+1).append(sl_num/2,'\\');
		if (0 == (sl_num % 2)) break;
	}
	return is;
}


bool is_ext_label(const std::string& line)
{
	return alt::starts_with_and_space(line, "%.")
		|| alt::starts_with_and_space(line, "%label");
}

bool is_int_label(const std::string& line)
{
	return alt::starts_with_and_space(line, "%..")
		|| alt::starts_with_and_space(line, "%sublabel");
}

bool is_comment(const std::string& line)
{
	return alt::starts_with_and_space(line, "%;")
		|| alt::starts_with_and_space(line, "%comment");
}

char
char_before_is_esc(const std::string& line, std::string::size_type pos)
{
	if(line.empty() || pos == 0 || pos > line.size()) return false;
	if(pos != std::string::npos) return line[pos - 1] == '\\';
	return *line.rbegin() == '\\';
}


std::string split_on_unescaped_space(std::string& line)
{
	if(line.empty()) return "";
	std::string::size_type pos = 0;
	while(true)
	{
		pos = line.find_first_of(" \t", pos);
		if(!char_before_is_esc(line, pos)) break;
		pos++;
	}
	std::string::size_type pos2 = line.find_first_not_of(" \t", pos);
	std::string ans = line.substr(0, pos);
	line.erase(0, pos2);
	return ans;
}

std::string
split_on_eq_sign(std::string& line)
{
	if(line.empty()) return "";
	std::string::size_type pos = 0;
	while(true)
	{
		pos = line.find('=', pos);
		if(pos == std::string::npos)
			throw std::runtime_error("= is missing");
		if(!char_before_is_esc(line, pos)) break;
		pos++;
	}
	std::string ans = line.substr(0, pos);
	line.erase(0, pos + 1);
	return ans;
}

void parse_label(matcher& m,std::string& line)
{
	m.clear();

	split_on_unescaped_space(line);//cut first field
	std::string namere = split_on_unescaped_space(line);//regexp name
	m(namere);
	while(!line.empty())
	{
		std::string avalre = split_on_unescaped_space(line);
		std::string anamere = split_on_eq_sign(avalre);
		m(anamere, avalre);//pair attr=value
	}
}

std::string load_block(std::istream& is,std::string& int_label,std::string& block)
{
	block.clear();
	std::string line;
	while(join_lines(is,line))
	{
		if (is_ext_label(line))
			return line;
		else if (is_int_label(line))
		{
			int_label = line;
			return "";
		}
		else if (is_comment(line))
			;
		else
			block += line + '\n';
	}
	return "";
}


std::string load_ext(std::istream& is)
{
	std::string line;
	while(join_lines(is,line))
		if (is_ext_label(line)) return line;
	return "";
}

void load_script(std::istream& is,script_t& script)
{
	std::string ext_label,int_label,block;

	matcher ext_matcher,int_matcher;
	
	ext_label = load_block(is,int_label,block);//load first ext label
	parse_label(ext_matcher,ext_label);//fill appropriate matcher
	ext_label = load_block(is,int_label,block);//load first int label
	bool dead_area = false;
	
	script_blocks_t int_scripts;
	while(is)
	{
		if (!ext_label.empty()) //need to update ext_label
		{
			//push back current internal matchings
			script.push_back(make_pair(ext_matcher,int_scripts));
			int_scripts.clear();

			parse_label(ext_matcher,ext_label);
			dead_area = true;
		}
		else//need to update int_label
		{
			parse_label(int_matcher,int_label);
			dead_area = false;
		}
		ext_label = load_block(is,int_label,block);
		if (!dead_area)
			int_scripts.push_back(make_pair(int_matcher,block));
	}
	
	script.push_back(make_pair(ext_matcher,int_scripts));//push rest of the data
}

struct matching
{
	matching(const command& cmd,context& match):
		cmd_(cmd),
		match_(match)
	{}

	bool operator()(const ext_script_block_t& block)
	{
		match_ = context();
		return block.first.matches(cmd_, match_);
	}
private:
	const command& cmd_;
	context& match_;
};


struct append_attr
{
	append_attr(std::string& str): str_(str) {}
	void operator()(const std::string& name,const std::string& value)
	{
		str_.append(" "+name+"=\""+value+"\"");
	}
	std::string& str_;
};

context get_attr_list(const command& cmd)
{
	context ctx;
	std::string list;
	cmd.for_each(append_attr(list));
	ctx("all",list);
	return ctx;
}

struct try_unconvert
{
	try_unconvert(const std::string& shell,const command& cmd,const context& match1):
		shell_(shell),
		cmd_(cmd),
		match1_(match1),
		attr_list_(get_attr_list(cmd_))
	{}
	void operator()(const int_script_block_t& item)
	{
		context match2;
		if (item.first.matches(cmd_,match2))
		{
			evaluator eval;
			eval.functab_["lookup"] = func_lookup;
			eval.ctxttab_["match"] = match1_;
			eval.ctxttab_["match2"] = match2;
			eval.ctxttab_["attrs"] = attr_list_;
			alt::run_shell(shell_, eval(item.second));
		}
	}
	
	const std::string& shell_;
	const command& cmd_;
	const context& match1_;
	const context  attr_list_;
};



void
unconvert(std::istream& is,const std::string& shell, const script_t& script)
{
	context match1;
	command first_cmd;
	is >> first_cmd;//read first command
	script_t::const_iterator i = std::find_if(script.begin(), script.end(),
							matching(first_cmd,match1));
	if(i != script.end())
	{
		command cmd;
		while(is>>cmd)
		{
			const script_blocks_t& subscripts=i->second;
			std::for_each(subscripts.begin(),
			      subscripts.end(),
			      try_unconvert(shell,cmd,match1));
			std::cout<<command("/rcommander/stop");
		}
	}
}

struct print_matcher
{
	void operator()(const matcher& label) const
	{
		std::cout << label.as_command();
	}
};

void
init(const script_t& script)
{
	std::for_each(script.begin(), script.end(), first(print_matcher()));
}


int
main(int argc, char *argv[])
{
	int res = EXIT_SUCCESS;
	if(argc < 4) return usage();
	std::string shell(argv[1]);
	std::string action(argv[3]);

	std::ifstream	is(argv[2]);
	script_t	script;

	load_script(is,script);
	
	if ("init" == action)
		init(script);
	else if ("unconvert" == action)
		unconvert(std::cin,shell,script);
	else
	{
		std::cerr<<"unknown action";
		res = EXIT_FAILURE;
	}
	
	return res;
}

