//run as: admfsd /tmp/fuse -d -s -o direct_io

#include <iostream>
#include <stdexcept>

#include <fcntl.h>
#include <fuse.h>


#include <admfs/conductor.hh>
#include <admfs/support.hh>



using namespace admfs;

std::string new_node;

/**
 * we assume that first level files are directories, and
 * all directories contains only files
 */
static
int admfs_getattr(const char *path, struct stat *stbuf)
{
    //knavery for fuse internal mknod algorithm: first mknod then getattr on success
    //there are no races, 'cause fuse always made sequence mknod -> lookup without
    //any other calls between it
    if (new_node == path)
    {
	new_node.clear();
        stbuf->st_mode = S_IFREG | 0644;
        stbuf->st_nlink = 1;
        stbuf->st_size = 0;
	return 0;
    }

    memset(stbuf, 0, sizeof(struct stat));

    std::string t = conductor::instance().type(path);
    if (t == "d")
    {
        stbuf->st_mode = S_IFDIR | 0755;
        stbuf->st_nlink = 2;
    }
    else if (t == "r")
    {
        stbuf->st_mode = S_IFREG | 0644;
        stbuf->st_nlink = 1;
        stbuf->st_size = 0;
    }
    else
	return -ENOENT;

    return 0;
}


static
int admfs_getdir(const char *path, fuse_dirh_t h, fuse_dirfil_t filler)
{
    dirent_vector items;
    conductor::instance().list(path,items);

    filler(h, ".", 0);
    filler(h, "..", 0);
    for (size_t i=0;i<items.size();++i)
    {
	filler(h,items[i].c_str(),0);
    }

    return 0;
}


static
int admfs_mkdir(const char *path, mode_t)
{
    //allow to create directories only on first level
    if (dir_level(path) != 2)
	return -EPERM;
    
    return (conductor::instance().add_plugin(path))?0:-EPERM;
}

static
int admfs_rmdir(const char *path)
{
    //allow to create directories only on first level
    if (dir_level(path) != 2)
	return -EPERM;

    conductor::instance().del_plugin(path);
    return 0;
}


static
int admfs_unlink(const char *path)
{
    if (dir_level(path) != 3)
	return -EPERM;
    
    conductor::instance().destroy(path);

    return 0;
}

static
int admfs_open(const char *path, int mode)
{
    if (((mode&O_CREAT) != O_CREAT) && 
        (conductor::instance().type(path,false)!="r"))
    return -ENOENT;

    return (conductor::instance().open(path,mode))?0:-EPERM;
}

static
int admfs_read(const char *path, char *buffer, size_t buflen, off_t off)
{
    return conductor::instance().read(path,buffer,buflen,off);
}

static
int admfs_write(const char *path, const char *buffer, size_t buflen, off_t off)
{
    return conductor::instance().write(path,buffer,buflen,off);
}

/** 
 * I cannot made really fake mknod, 'cause fuse always internally calls getattr on success.
 * But I don't want to made mknod, 'cause it's forse us always has support for
 * empty constructors in backends
 */ 
int admfs_mknod(const char *path, mode_t, dev_t)
{
    new_node = path;//save path for getattr
    return 0;
}

static
int admfs_truncate(const char *path, off_t len)
{
    conductor::instance().truncate(path,len);
    return 0;
}


static
int admfs_release(const char *path, int)
{
    conductor::instance().close(path);
    return 0;
}

int main(int argc, char *argv[])
{
    try
    {
	fuse_operations admfs_operations;
	memset(&admfs_operations,0,sizeof(fuse_operations));
    
	admfs_operations.getattr = admfs_getattr;
	admfs_operations.getdir = admfs_getdir;
	admfs_operations.mkdir = admfs_mkdir;
	admfs_operations.rmdir = admfs_rmdir;
	admfs_operations.unlink = admfs_unlink;
	admfs_operations.open = admfs_open;
	admfs_operations.truncate = admfs_truncate;
	admfs_operations.read = admfs_read;
	admfs_operations.write = admfs_write;
	admfs_operations.mknod = admfs_mknod;
	admfs_operations.release = admfs_release;
    
	fuse_main(argc, argv, &admfs_operations);
    }
    catch(std::runtime_error& e)
    {
	std::cerr<<"runtime error:"<<e.what()<<std::endl;
    }
    catch(std::exception& e)
    {
	std::cerr<<"exception:"<<e.what()<<std::endl;
    }
    catch(...)
    {
	std::cerr<<"unknown exception"<<std::endl;
    }

    return 0;
}

