//for DEBUG
#include <iostream>

#include <fcntl.h>

#include <utils/pstream.hh>
#include <utils/line_iterator.hh>
#include <utils/str.hh>
#include <utils/os.hh>

#include <admfs/conductor.hh>


const std::string plugin_dir="samples/";
const std::string stderr_name = ".stderr";

using namespace admfs;

namespace
{
    struct root_inserter
    {
	root_inserter(dirent_vector& vec): vec_(vec) {}
	void operator()(const supporter_map::value_type& item)
	{
	    vec_.push_back(item.first);
	}
	dirent_vector& vec_;
    };
    
    struct plugin_inserter
    {
	plugin_inserter(const std::string& object,dirent_vector& vec,cache_type& ca):
	    obj_(object),
	    vec_(vec),
	    ca_(ca)
	{}
	void operator()(const std::string& item)
	{
	    std::string::size_type pos = item.find(" ");
	    vec_.push_back(item.substr(0,pos));
	    ca_[obj_+item.substr(0,pos)]=item.substr(pos+1,std::string::npos);
	}
	std::string obj_;
	dirent_vector& vec_;
	cache_type& ca_;
    };
    
    pid_t subprocess(const std::string& cmd,int &in,int& out,int& err)
    {
	in = out = err = -1;
	int stdin_fd = -1,stdout_fd = -1, stderr_fd = -1;
	pid_t pid;
	try
	{
	    alt::pipe(stdin_fd,in);
	    alt::pipe(out, stdout_fd);
	    alt::pipe(err, stderr_fd);
	    if(!(pid = alt::fork()))
	    {
		try
		{
		    ::close(in);
		    ::close(out);
		    ::close(err);
		    
		    ::close(STDIN_FILENO); alt::dup2(stdin_fd, STDIN_FILENO);
	    	    ::close(STDOUT_FILENO); alt::dup2(stdout_fd, STDOUT_FILENO);
		    ::close(STDERR_FILENO); alt::dup2(stderr_fd, STDERR_FILENO);
		    alt::exec(cmd);
	    	}
		catch(...)
		{
		}
		::exit(1);
	    }
	    ::close(stdin_fd);
	    ::close(stdout_fd);
	    ::close(stderr_fd);
	}
	catch(...)
	{
	    if (stdin_fd >= 0) ::close(stdin_fd);
	    if (stdout_fd >= 0) ::close(stdout_fd);
	    if (stderr_fd >= 0) ::close(stderr_fd);

	    if (in >= 0) ::close(in);
	    if (out >= 0) ::close(out);
	    if (err >= 0) ::close(err);
	}
	return pid;
    }
    
    /** split path to module and object names */
    void split(const std::string& path,std::string& module,std::string& object)
    {
	std::string::size_type pos1 = path.find_first_not_of("/");
	std::string::size_type pos2 = path.find("/",pos1);

	std::string::size_type pos3 = path.find_first_not_of("/",pos2);

	if (pos1 != std::string::npos)
	    module = path.substr(pos1,(pos2 == std::string::npos)?pos2:pos2-pos1);
	if (pos3 != std::string::npos)
	    object = path.substr(pos3,std::string::npos);
    }
}

/** @todo: add support for .error file */
std::string conductor_impl::type(const std::string& path,bool use_cache)
{

    if (path == "/") return "d";

    std::string module,object;
    split(path,module,object);

    std::string res;
    
    if (!module.empty() && supporters_.find(module) != supporters_.end())
	res = "d";

    if (!object.empty())
    {

	if (object == stderr_name)
	{//special processing for .stderr file
	    res = supporters_[module].err_buffer_.empty()?"":"r";
	    return res;
	}
	
	bool cache_search = supporters_[module].cache_.find(object) != supporters_[module].cache_.end();

	if (use_cache && cache_search)
	    res = supporters_[module].cache_[object];
	else
	{//
	    alt::ipstream is(plugin_dir + module + " -f "+object);
	    std::string s;
	    std::getline(is,s);
	    if (!s.empty())
	    {//add to cache
		std::string::size_type pos = s.find(" ");
		supporters_[module].cache_[object] = s.substr(pos+1,std::string::npos);
		res = supporters_[module].cache_[object];
	    }
	    else
	    {
		res = "";//unknown object
		if(cache_search) //invalidate cache if we need it
		    supporters_[module].cache_.erase(object);
	    }
	}
    }

    return res;
}


bool conductor_impl::add_plugin(const std::string& path)
{//simple check for existance of the plugin file in plugin dir
    if (!access((plugin_dir+alt::trim_right(path," \t/")).c_str(),X_OK))
    {
	supporters_[alt::trim(path," \t/")]=runtime_info();
	return true;
    }
    return false;
}

void conductor_impl::del_plugin(const std::string& path)
{
    supporters_.erase(alt::trim(path," \t/"));
}

/** free data buffer for the object */
/** @todo: add data dumping if was write mode */
void conductor_impl::close(const std::string& path)
{
    std::string module,object;
    split(path,module,object);
    
    if (module.empty() || object.empty()) return;
    
    //unknown module or object not opened
    if (supporters_.find(module) == supporters_.end() ||
	!supporters_[module].opened_ )
	return;
    
    supporters_[module].opened_ = false;

    //dump buffer contents
    int mode = supporters_[module].open_mode_;

    //FIXME: what about append mode?
    if (((mode&O_WRONLY) == O_WRONLY) ||
        ((mode&O_RDWR) == O_RDWR))
    {
	supporters_[module].buffer_.push_back(0);

	int in,out,err;
	pid_t pid = subprocess(plugin_dir+module+" -w "+object,in,out,err);

	std::vector<char> &buff= supporters_[module].buffer_;
	off_t off = 0;
	while(off<buff.size())
	{
	    int len = ::write(in,&buff[0]+off,buff.size()-off);
	    off += len;
	}
	::close(in); //send EOF

	int len;
	std::vector<char> tmp(BUFSIZ);
	while ((len=::read(err,&tmp[0],tmp.size())) > 0)
	    std::copy(tmp.begin(),
	              tmp.begin()+len,
		      std::back_inserter(supporters_[module].err_buffer_));

	::close(out);
	::close(err);
	alt::wait(pid);
    }

    supporters_[module].buffer_.clear();
}

/** fill internal buffer with appropriate data */
/** @todo add support for .error file */
size_t conductor_impl::write(const std::string& path, const char *buffer, size_t buflen, off_t off)
{
    std::string module,object;
    split(path,module,object);
    
    if (module.empty() || object.empty()) return 0;
    
    size_t old_size = supporters_[module].buffer_.size();
    
    //expand buffer if we need it
    if (off+buflen > supporters_[module].buffer_.size())
    {
	supporters_[module].buffer_.resize(off+buflen);
	std::fill(supporters_[module].buffer_.begin()+old_size,
		  supporters_[module].buffer_.end(),
		  0);
    }
    
    //copy data 
    std::copy(buffer,
              buffer+buflen,
	      supporters_[module].buffer_.begin()+off);

    return buflen;
}

/** fill external buffer with appropriate data */
/** @todo add support for .error file */
size_t conductor_impl::read(const std::string& path, char *buffer, size_t buflen, off_t off)
{
    std::string module,object;
    split(path,module,object);
    
    if (module.empty() || object.empty()) return 0;
    
    std::vector<char> &buff = (object == stderr_name)?
			    supporters_[module].err_buffer_:
			    supporters_[module].buffer_;

    if (off >= buff.size()) return 0;

    int len = std::min(static_cast<off_t>(buflen),
                       buff.size()-off);

    if (!len) return 0;

    std::uninitialized_copy(buff.begin()+off,
			    buff.begin()+off+len,
			    buffer);
    return len;
}

void conductor_impl::truncate(const std::string& path,off_t len)
{
    std::string module,object;
    split(path,module,object);

    if (module.empty() || object.empty()) return;
    if (supporters_.find(module) == supporters_.end()) return;

    size_t old_size = supporters_[module].buffer_.size();
    supporters_[module].buffer_.resize(len);
    
    if (len > old_size)
	std::fill(supporters_[module].buffer_.begin()+old_size,
		  supporters_[module].buffer_.end(),
		  0);
}

/** retrieve object information (if not write only), fill buffers with data */
/** @todo add support for .error file */
bool conductor_impl::open(const std::string& path,int mode)
{

    std::string module,object;
    split(path,module,object);

    if (module.empty() || object.empty()) return false;

    //unknown module
    if (supporters_.find(module) == supporters_.end())
	return false;

    //file opened and still not closed
    if (supporters_[module].opened_)
	return false;

    if (object == stderr_name)
	return !supporters_[module].err_buffer_.empty();

    supporters_[module].opened_ = true;
    supporters_[module].open_mode_ = mode;
    if ((mode&O_WRONLY) != O_WRONLY)
    {//if mode != write only we fill buffer with object's data
	int in,out,err;
	pid_t pid = subprocess(plugin_dir+module+" -r "+object,in,out,err);

	int len;
    	std::vector<char> buff(BUFSIZ);
	
	while ((len=::read(out,&buff[0],buff.size())) > 0)
	    std::copy(buff.begin(),
	              buff.begin()+len,
		      std::back_inserter(supporters_[module].buffer_));

	while ((len=::read(err,&buff[0],buff.size())) > 0)
	    std::copy(buff.begin(),
	              buff.begin()+len,
		      std::back_inserter(supporters_[module].err_buffer_));
	
	::close(in);
	::close(out);
	::close(err);
	alt::wait(pid);

    }

    return true;
}

/**
 * 1. reparce path to determine supporter and it's object (READY)
 * 2. run subprocess with "-d name key" and read it's stderr (TODO)
 * 3. if stderr read result is not empty then create .stderr file (TODO)
 */
void conductor_impl::destroy(const std::string& path)
{

    std::string module,object;
    split(path,module,object);

    if (module.empty() || object.empty()) return;

    if (supporters_.find(module) == supporters_.end())
	return;

    if (object == stderr_name)
    {//special processing for .stderr file
	supporters_[module].err_buffer_.clear();
	return;
    }

    int in,out,err;
    pid_t pid = subprocess(plugin_dir+module+" -d "+object,in,out,err);

    std::vector<char> buff(BUFSIZ);
    int len;
    
    while ((len=::read(err,&buff[0],buff.size())) > 0)
	std::copy(buff.begin(),
	          buff.begin()+len,
		  std::back_inserter(supporters_[module].err_buffer_));

    ::close(in);
    ::close(out);
    ::close(err);
    alt::wait(pid);

    supporters_[module].cache_.erase(object);//invalidate cache entry
}

/** @todo add support for .error file */
void conductor_impl::list(const std::string& path,dirent_vector& dirlist)
{
    if ("/" == path)
    {
	std::for_each(supporters_.begin(),
	              supporters_.end(),
		      root_inserter(dirlist));
    }
    else
    {//get info from appropriate plugin
	std::string module,object;
	split(path,module,object);

	if (module.empty()) return;
	if (supporters_.find(module) == supporters_.end()) return;

	cache_type &cache = supporters_[module].cache_;

	std::string object_dir=object.empty()?"":(object+"/");
	if (cache.size())
	{
	    cache_type::iterator start = cache.lower_bound(object_dir);
	    cache_type::iterator stop = start;
	    while (stop != cache.end() && alt::starts_with(stop->first, object_dir)) ++stop;

	    cache.erase(start,stop);
	}
	
	alt::ipstream is(plugin_dir+module+" -l "+(object.empty()?"/":object));
	std::string str;
	std::for_each(alt::istream_line_iterator(is),
		      alt::istream_line_iterator(),
		      plugin_inserter(object_dir,dirlist,cache));

	if (!supporters_[module].err_buffer_.empty())
	    dirlist.push_back(stderr_name);
    }
}
