#include <iostream>
#include <fstream>
#include <string>
#include <sstream>

#include <getopt.h>

#include <command_io.hh>
#include <bus_io.hh>
#include <utils/str.hh>

std::string from_path="/tmp/input";
std::string to_path="/tmp/output";


template <typename S, typename K>
void split_discard_char(const std::string& str, S s, K k)
{
	std::string::size_type pos = str.find(s);
	if(pos != std::string::npos) k(str.substr(0, pos), str.substr(pos + 1));
	else k(str, "");
}

struct add_attr
{
	add_attr(command& cmd): cmd_(cmd) {}
	void operator ()(const std::string& first, const std::string& second)
	{
		cmd_(first, second);
	}
private:
	command& cmd_;
};


struct dump_options
{
    dump_options(std::ostream& os): os_(os) {}
    void operator()(const std::string& key,const std::string& value)
    {
	os_<<" "<<key<<"="<<value;
    }
    std::ostream &os_;
};

/** print program usage information */
void usage(int retcode)
{
	extern const char *__progname;
	std::cout << "Usage: " << __progname << " [-hvra] [--help] [--version] [--admfs-path <path>]" << std::endl;
	std::cout << "testing module for woobus" << std::endl;
	std::cout << "Dumps HOO commands from file to bus, and reads answers" << std::endl;
	std::cout << "\t-h, --help              display help screen" << std::endl;
	std::cout << "\t-v, --version           display version information" << std::endl;
	std::cout << "\t-f, --from		file with HOO commands" << std::endl;
	std::cout << "\t-t, --to		file with answers" << std::endl;
	std::cout << std::endl;
	std::cout << "Report bugs to <inger@altlinux.org>" << std::endl;
	exit(retcode);
}

void bus_main(std::istream& is,std::ostream& os,std::ostream& log)
{
    std::ifstream questions(from_path.c_str());
    std::ofstream answers(to_path.c_str());

    if (!questions || !answers)
    {
	log<<"unable to open input or output files"<<std::endl;
	return;
    }

    std::string buf;
    while (std::getline(questions,buf) && is && os)
    {

	alt::args_iterator args(buf);
	command cmd(*args);
	while(++args != alt::args_iterator())
		split_discard_char(*args, '=', add_attr(cmd));
	std::ostringstream ost;
	ost<<cmd;

	send(">>",ost.str(),is,os,log);//sent question
	
	log<<"waiting for answer"<<std::endl;

	while(is)
	{
	    std::string direction;
	    std::getline(is,direction);
	    if ("<<" == direction)
	    {
		std::istringstream ist(receive(is,os,log));
		command cmd;
		if (ist.str().empty())
		    answers<<"** empty answer **"<<std::endl;
		else
		    while(ist>>cmd)
		    {
			answers<<cmd.name();
			cmd.for_each(dump_options(answers));
			answers<<std::endl;
		    }
		break;
	    }
	    else
		os<<"<>"<<std::endl;
	}
    }
    log<<"cat done\n";
    for(;;);
}


int main(int argc,char *argv[])
{
    for(;;)
    {
    	static struct option long_options[] =
	{
	    {"help", no_argument, 0, 'h'},
	    {"version", no_argument, 0, 'v'},
	    {"admfs-path", required_argument, 0, 'p'},
	    {0, 0, 0, 0}
	};
	int c = getopt_long(argc, argv, "hvf:t:", long_options, NULL);
	if ( -1 == c) break;
	switch (c)
	{
	    case 'h':
		usage(EXIT_SUCCESS);
	    case 'v':
		std::cout << "executor version " << VERSION << std::endl;
		std::cout << "Written by Stanislav Ievlev" << std::endl << std::endl;
		std::cout << "Copyright (C) 2004 ALT Linux Team" << std::endl;
		return (EXIT_SUCCESS);
	    case 'f':
		from_path = optarg;
		break;
	    case 't':
		to_path = optarg;
		break;
	    default:
		usage(EXIT_FAILURE);
	}
    }
    
    bus_main(std::cin,std::cout,std::cerr);
}
