/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <iostream>
#include <stdexcept>

#include <getopt.h>
#include <fcntl.h>
#include <fuse.h>

#include <conductor.hh>

std::string new_node;
std::string plugin_dir="/usr/lib/alterator/backends/";

void usage(int retcode)
{
	extern const char *__progname;
	std::cout << "Usage: " << __progname << " [-hvmdp] [--help] [--version] [--plugin-path <path>] [--mount-point <mnt>]" << std::endl;
	std::cout << "Alterator module between brook and amdfs" << std::endl;
	std::cout << "Convert HOO commands to operations on admfs filesystems and inversely" << std::endl;
	std::cout << "\t-h, --help             display help screen" << std::endl;
	std::cout << "\t-v, --version          display version information" << std::endl;
	std::cout << "\t-p, --plugin-path     path to directory with backends" << std::endl;
	std::cout << "\t-m, --mount-point     path to directory with admfs mounted in it" << std::endl;
	std::cout << "\t-d, --debug           run in debug mode" << std::endl;
	std::cout << std::endl;
	std::cout << "Report bugs to <inger@altlinux.org>" << std::endl;
	exit(retcode);
}



/**
 * we assume that first level files are directories, and
 * all directories contains only files
 */
static
int admfs_getattr(const char *path, struct stat *stbuf)
{
    //knavery for fuse internal mknod algorithm: first mknod then getattr on success
    //there are no races, 'cause fuse always made sequence mknod -> lookup without
    //any other calls between it
    if (new_node == path)
    {
	new_node.clear();
        stbuf->st_mode = S_IFREG | 0644;
        stbuf->st_nlink = 1;
        stbuf->st_size = 0;
	return 0;
    }

    memset(stbuf, 0, sizeof(struct stat));

    std::string t = conductor::instance().type(path);
    if (t == "d")
    {
        stbuf->st_mode = S_IFDIR | 0755;
        stbuf->st_nlink = 2;
    }
    else if (t == "r")
    {
        stbuf->st_mode = S_IFREG | 0644;
        stbuf->st_nlink = 1;
        stbuf->st_size = 0;
    }
    else
	return -ENOENT;

    return 0;
}


static
int admfs_getdir(const char *path, fuse_dirh_t h, fuse_dirfil_t filler)
{
    dirent_vector items;
    conductor::instance().list(path,items);

    filler(h, ".", 0);
    filler(h, "..", 0);
    for (size_t i=0;i<items.size();++i)
    {
	filler(h,items[i].c_str(),0);
    }

    return 0;
}


static
int admfs_mkdir(const char *path, mode_t)
{
    //allow to create directories only on first level
    if (dir_level(path) != 2)
	return -EPERM;
    
    return (conductor::instance().add_plugin(path,plugin_dir))?0:-EPERM;
}

static
int admfs_rmdir(const char *path)
{
    if (dir_level(path) == 2)
    	conductor::instance().del_plugin(path);
    else
    	conductor::instance().destroy(path);
    return 0;
}


static
int admfs_unlink(const char *path)
{
    if (dir_level(path) != 3)
	return -EPERM;
    
    conductor::instance().destroy(path);

    return 0;
}

static
int admfs_open(const char *path, int mode)
{
    if (((mode&O_CREAT) != O_CREAT) && 
        (conductor::instance().type(path,false)!="r"))
    return -ENOENT;

    return (conductor::instance().open(path,mode))?0:-EPERM;
}

static
int admfs_read(const char *path, char *buffer, size_t buflen, off_t off)
{
    return conductor::instance().read(path,buffer,buflen,off);
}

static
int admfs_write(const char *path, const char *buffer, size_t buflen, off_t off)
{
    return conductor::instance().write(path,buffer,buflen,off);
}

/** 
 * I cannot made really fake mknod, 'cause fuse always internally calls getattr on success.
 * But I don't want to made mknod, 'cause it's forse us always has support for
 * empty constructors in backends
 */ 
int admfs_mknod(const char *path, mode_t, dev_t)
{
    new_node = path;//save path for getattr
    return 0;
}

static
int admfs_truncate(const char *path, off_t len)
{
    conductor::instance().truncate(path,len);
    return 0;
}


static
int admfs_release(const char *path, int)
{
    conductor::instance().close(path);
    return 0;
}

int main(int argc, char *argv[])
{
	bool do_debug=false;
	std::string mount_point="/amdfs";
	int res = 0;

    for(;;)
    {
    	static struct option long_options[] =
	{
	    {"help", no_argument, 0, 'h'},
	    {"version", no_argument, 0, 'v'},
	    {"plugin-dir", required_argument, 0, 'p'},
	    {"debug", required_argument, 0, 'd'},
	    {"mount-point", required_argument, 0, 'm'},
	    {0, 0, 0, 0}
	};
	int c = getopt_long(argc, argv, "hvp:dm:", long_options, NULL);
	if ( -1 == c) break;
	switch (c)
	{
	    case 'h':
	    	usage(EXIT_SUCCESS);
		break;
	    case 'v':
		std::cout << "admfsd version 1.0" << std::endl;
		std::cout << "Written by Stanislav Ievlev" << std::endl << std::endl;
		std::cout << "Copyright (C) 2004 ALT Linux Team" << std::endl;
		return (EXIT_SUCCESS);
	    case 'p':
		plugin_dir = optarg;
		break;
	    case 'm':
		mount_point = optarg;
		break;
	    case 'd':
	    	do_debug=true;
		break;
	    default://ignore other options - pass them to fuse_main
	    	usage(EXIT_FAILURE);
	    	break;
	}
    }

    try
    {
	fuse_operations admfs_operations;
	memset(&admfs_operations,0,sizeof(fuse_operations));
    
	admfs_operations.getattr = admfs_getattr;
	admfs_operations.getdir = admfs_getdir;
	admfs_operations.mkdir = admfs_mkdir;
	admfs_operations.rmdir = admfs_rmdir;
	admfs_operations.unlink = admfs_unlink;
	admfs_operations.open = admfs_open;
	admfs_operations.truncate = admfs_truncate;
	admfs_operations.read = admfs_read;
	admfs_operations.write = admfs_write;
	admfs_operations.mknod = admfs_mknod;
	admfs_operations.release = admfs_release;

	std::vector<const char*> fuse_args;
	fuse_args.push_back(argv[0]);
	fuse_args.push_back(strdup("-s"));
	if (do_debug) fuse_args.push_back(strdup("-d"));
	fuse_args.push_back(strdup("-o"));
	fuse_args.push_back(strdup("direct_io"));
	fuse_args.push_back(strdup(mount_point.c_str()));
	fuse_args.push_back(0);
  
	res = fuse_main(fuse_args.size()-1,(char **)&fuse_args[0],&admfs_operations);
    }
    catch(std::runtime_error& e)
    {
	std::cerr<<"runtime error:"<<e.what()<<std::endl;
    }
    catch(std::exception& e)
    {
	std::cerr<<"exception:"<<e.what()<<std::endl;
    }
    catch(...)
    {
	std::cerr<<"unknown exception"<<std::endl;
    }

    return res;
}

