/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#include <iostream>
#include <fstream>
#include <string>
#include <sstream>
#include <cerrno>

#include <getopt.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <command_io.hh>
#include <conductor.hh>
#include <bus_io.hh>
#include <utils/line_iterator.hh>
#include <utils/str.hh>
#include <utils/dirstream.hh>

std::string admfs_path="/admin";

struct append_options
{
    append_options(command& cmd): cmd_(cmd) {}
    void operator()(const std::string& line)
    {
	std::string::size_type pos = line.find(":");
	cmd_(line.substr(0,pos),line.substr(pos+1,std::string::npos));
    }
    command &cmd_;
};

/**
 * @todo it's possible we need some encoding
 */
struct dump_options
{
    dump_options(std::ostream& os): os_(os) {}
    void operator()(const std::string& key,const std::string& value)
    {
	if ("action" != key) os_<<key<<":"<<value<<std::endl;
    }
    std::ostream &os_;
};

/** print program usage information */
void usage(int retcode)
{
	extern const char *__progname;
	std::cout << "Usage: " << __progname << " [-hvra] [--help] [--version] [--admfs-path <path>]" << std::endl;
	std::cout << "Alterator module between brook and amdfs" << std::endl;
	std::cout << "Convert HOO commands to operations on admfs filesystems and inversely" << std::endl;
	std::cout << "\t-h, --help             display help screen" << std::endl;
	std::cout << "\t-v, --version          display version information" << std::endl;
	std::cout << "\t-a, --admfs-path       path to directory with admfs mounted in it" << std::endl;
	std::cout << std::endl;
	std::cout << "Report bugs to <inger@altlinux.org>" << std::endl;
	exit(retcode);
}

/* write options to the file */
bool wr(const std::string& path,command& cmd)
{
    std::ofstream os(path.c_str());
    if (!os)
	return false;
    cmd.for_each(dump_options(os));
    return true;
}

void sure_plugin(const std::string& path)
{
	if (::access(path.c_str(),R_OK)) ::mkdir(path.c_str(),0755);
}

void forward(std::istream& in,std::ostream& out)
{
    command cmd;
    while(in>>cmd)
    {
	//ignore control commands
	if (cmd.name().empty() || alt::starts_with(cmd.name(),"/ctrl/"))
	{
	    out<<command("");
	    continue;
	}
	
	command error("/error"+cmd.name());

	if (!cmd.exists("action"))
	{
		out<<error("reason","action field not found");
		continue;
	}

	const std::string object=cmd.name();
	const std::string path=admfs_path+"/"+cmd.name();
	const std::string action = cmd("action");

	error("action",action);

	//don't allow relative directories in result path
	if (!valid_path(path))
	{
		out<<error("reason","invalid path detected");
		continue;
	}
	
	const std::string directory=
		object.substr(0,object.find("/",
			(object[0] == '/')?object.find_first_not_of("/"):0));

	sure_plugin(admfs_path+"/"+directory);

	struct stat st;
	if (stat(path.c_str(),&st))
	{
	    if ("new" == action)
	    {
		if ('/' == *path.rbegin()) //if path ends with '/' - it's a directory
		{
		    if ((mkdir(path.c_str(),0755) && errno != EEXIST) ||
		        access(path.c_str(),R_OK))
			    out<<error("reason","failed");
		}
		else
		{
		    if (!wr(path,cmd) ||
			access(path.c_str(),R_OK))
			    out<<error("reason","failed");
		}
	    }
	    else
		out<<error("reason","absent");

	    continue;
	}

	if (S_ISDIR(st.st_mode))
	{
	    if ("new" == action)
	    {//ignore new for existent items
	    }
	    else if ("delete" == action)
	    {
	        if (rmdir(path.c_str()) ||
		    (!access(path.c_str(),R_OK)))
			out<<error("reason","failed");
	    }
	    else if ("read" == action)
	    {
		alt::dirstream is(path.c_str());
		if (!is)
		    out<<error("reason","failed");
		else
		{
		    std::string line;
		    while(std::getline(is,line))
		    {
			if (("." != line) && (".." != line))
			out<<command(cmd.name()+"/"+line);
		    }
		}
	    }
	    else
		out<<error("reason","undefined");
	}
	else if (S_ISREG(st.st_mode))
	{
	    if ("new" == action)
	    {//ignore new for existent items
	    }
	    else if ("delete" == action)
	    {
		if (unlink(path.c_str()) ||
		   (!access(path.c_str(),R_OK)))
		        out<<error("reason","failed");
	    }
	    else if ("read" == action)
	    {
	        std::ifstream is(path.c_str());
	        if (is)
		{
		    command answer(cmd.name());
		    std::for_each(alt::istream_line_iterator(is),
		    	          alt::istream_line_iterator(),
				  append_options(answer));
		    out<<answer;
		}
		else
		    out<<error("reason","failed");
		}
	    else if ("write" == action)
	    {
		if (!wr(path,cmd))
		    out<<error("reason","failed");
	    }
	    else
		out<<error("reason","undefined");
	}
	else
	    out<<error("reason","unsupported");
    }
}


void bus_main(std::istream& is,std::ostream& os,std::ostream& log)
{
    while (is)
    {
	std::string direction;
	std::getline(is,direction);
	if (">>" == direction)
	{
	    std::istringstream question(receive(is,os,log));
	    std::ostringstream answer;
	    forward(question,answer);
	    send("<<",answer.str(),is,os,log);
	}
	else if ("<<" == direction)
	{
	    os<<"<>"<<std::endl;
	}
	else
	    log << "--invalid address line: \""
	    <<direction << "\"" << std::endl;
    }
}


int main(int argc,char *argv[])
{
    for(;;)
    {
    	static struct option long_options[] =
	{
	    {"help", no_argument, 0, 'h'},
	    {"version", no_argument, 0, 'v'},
	    {"admfs-path", required_argument, 0, 'a'},
	    {0, 0, 0, 0}
	};
	int c = getopt_long(argc, argv, "hva:", long_options, NULL);
	if ( -1 == c) break;
	switch (c)
	{
	    case 'h':
		usage(EXIT_SUCCESS);
	    case 'v':
		std::cout << "executor version " << VERSION << std::endl;
		std::cout << "Written by Stanislav Ievlev" << std::endl << std::endl;
		std::cout << "Copyright (C) 2004 ALT Linux Team" << std::endl;
		return (EXIT_SUCCESS);
	    case 'a':
		admfs_path = optarg;
		break;
	    default:
		usage(EXIT_FAILURE);
	}
    }
    
    bus_main(std::cin,std::cout,std::cerr);
}
