/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
//for DEBUG
#include <iostream>

#include <sstream>

#include <fcntl.h>

#include <utils/line_iterator.hh>
#include <utils/str.hh>
#include <utils/os.hh>

#include <conductor.hh>

//const std::string stderr_name = ".stderr";


bool valid_path(const std::string& path)
{
	return
	(path.find_first_of("\"'\\*?|;[]{}`<>()$&") == std::string::npos) &&
	(path.find("/../") == std::string::npos);
}

int dir_level(const std::string& path)
{
	int level = 1;
	std::string::size_type pos = 0;
	while (true)
	{
		pos = path.find("/",pos);//first find begin slash
		if (pos == std::string::npos)
	    		break;

		pos = path.find_first_not_of("/",pos);//second 
		if (pos == std::string::npos)
	    		break;
		++level;
    	}
    	return level;
}

namespace
{



    std::string quote(const std::string& str)
    {
    	std::ostringstream ost;
	std::string::size_type len = str.length();
	for(std::string::size_type pos = 0;pos<len;++pos)
	{
		if (str[pos] == ' ') ost<<"\\";
		ost<<str[pos];
	}
	return ost.str();
    }

    struct root_inserter
    {
	root_inserter(dirent_vector& vec): vec_(vec) {}
	void operator()(const supporter_map::value_type& item)
	{
	    vec_.push_back(item.first);
	}
	dirent_vector& vec_;
    };
    
    struct plugin_inserter
    {
	plugin_inserter(const std::string& object,dirent_vector& vec,cache_type& ca):
	    obj_(object),
	    vec_(vec),
	    ca_(ca)
	{}
	void operator()(std::string item)
	{
	    std::string field1=alt::split_on_unescaped_space(item);
	    ca_[obj_+field1]=item;
	    vec_.push_back(field1);
	}
	std::string obj_;
	dirent_vector& vec_;
	cache_type& ca_;
    };
    
    /** split path to backend and object names */
    void split(const std::string& path,std::string& backend,std::string& object)
    {
	std::string::size_type pos1 = path.find_first_not_of("/");
	std::string::size_type pos2 = path.find("/",pos1);

	std::string::size_type pos3 = path.find_first_not_of("/",pos2);

	if (pos1 != std::string::npos)
	    backend = path.substr(pos1,(pos2 == std::string::npos)?pos2:pos2-pos1);
	if (pos3 != std::string::npos)
	    object = path.substr(pos3,std::string::npos);
    }
}

/** @todo: add support for .error file */
std::string conductor_impl::type(const std::string& path,bool use_cache)
{
    if (path == "/") return "d";

    std::string backend,object;
    split(path,backend,object); 

    supporter_map::iterator it = supporters_.find(backend);
    if (backend.empty() || it == supporters_.end())
	    return "";

    std::string res = "d";
    plugin_info& info=it->second;
    
    if (!object.empty())
    {
/*
	if (object == stderr_name)
	{//special processing for .stderr file
	    res = info.err_buffer_.empty()?"":"r";
	    return res;
	}
*/	
	bool cache_search = info.cache_.find(object) != info.cache_.end();

	if (use_cache && cache_search)
	    res = info.cache_[object];
	else
	{//
	    module &m = info.plugin_;
	    m.start(" -f "+quote(object));
	    std::string s;
	    std::getline(m.is(),s);
	    m.stop();

	    if (!s.empty())
	    {//add to cache
	    	alt::split_on_unescaped_space(s);
		info.cache_[object] = s;
		res = info.cache_[object];
	    }
	    else
	    {
		res = "";//unknown object
		if(cache_search) //invalidate cache if we need it
		    info.cache_.erase(object);
	    }
	}
    }

    return res;
}


bool conductor_impl::add_plugin(const std::string& path,const std::string& plugin_dir)
{//simple check for existance of the plugin file in plugin dir
    const std::string plugin = alt::trim(path," \t/");
    const std::string command = plugin_dir+"/"+plugin;

    if (!access(command.c_str(),X_OK))
    {
	supporters_[plugin]=plugin_info(command);
	return true;
    }
    return false;
}

void conductor_impl::del_plugin(const std::string& path)
{
    supporters_.erase(alt::trim(path," \t/"));
}

/** free data buffer for the object */
/** @todo: add data dumping if was write mode */
void conductor_impl::close(const std::string& path)
{
    std::string backend,object;
    split(path,backend,object);
    
    if (backend.empty() || object.empty()) return;
    
    supporter_map::iterator it=supporters_.find(backend);
    //unknown backend or object not opened
    if (it == supporters_.end() ||
	!it->second.opened_)
	return;
    
    plugin_info& info=it->second;
    
    info.opened_ = false;

    //dump buffer contents
    const int mode = info.open_mode_;
    //FIXME: what about append mode?
    if (((mode&O_WRONLY) == O_WRONLY) ||
        ((mode&O_RDWR) == O_RDWR))
    {
	module &m = info.plugin_;

	m.start(" -w "+quote(object));
	
	m.os()<<std::string(info.buffer_.begin(),
			    info.buffer_.end());
	m.close_os();

	//NOTE: wait here or read stderr
	std::copy(std::istreambuf_iterator<char>(m.es().rdbuf()),
		  std::istreambuf_iterator<char>(),
		  std::ostreambuf_iterator<char>(std::cerr.rdbuf()));

	m.stop();
    }

    info.buffer_.clear();
}

/** fill internal buffer with appropriate data */
/** @todo add support for .error file */
size_t conductor_impl::write(const std::string& path, const char *buffer, size_t buflen, off_t off)
{
    std::string backend,object;
    split(path,backend,object);
    
    if (backend.empty() || object.empty()) return 0;
    
    std::vector<char> &int_buf=supporters_[backend].buffer_;
     
    if (off+buflen > int_buf.size())
    {//expand buffer if we need it
	const size_t old_size = int_buf.size();
	int_buf.resize(off+buflen);
	std::fill(int_buf.begin()+old_size,
		  int_buf.end(),
		  0);
    }
    
    //copy data 
    std::copy(buffer,
              buffer+buflen,
	      int_buf.begin()+off);

    return buflen;
}

/** fill external buffer with appropriate data */
/** @todo add support for .error file */
size_t conductor_impl::read(const std::string& path, char *buffer, size_t buflen, off_t off)
{
    std::string backend,object;
    split(path,backend,object);
    
    if (backend.empty() || object.empty()) return 0;
    
/*    std::vector<char> &buff = (object == stderr_name)?
			    supporters_[backend].err_buffer_:
			    supporters_[backend].buffer_;
*/
    std::vector<char> &buff = supporters_[backend].buffer_;

    if (off >= static_cast<off_t>(buff.size())) return 0;

    int len = std::min(buflen,
                       static_cast<size_t>(buff.size()-off));

    if (!len) return 0;

    std::uninitialized_copy(buff.begin()+off,
			    buff.begin()+off+len,
			    buffer);
    return len;
}

void conductor_impl::truncate(const std::string& path,off_t len)
{
    std::string backend,object;
    split(path,backend,object);

    if (backend.empty() || object.empty()) return;
    if (supporters_.find(backend) == supporters_.end()) return;

    const size_t old_size = supporters_[backend].buffer_.size();
    supporters_[backend].buffer_.resize(len);
    
    if (len > static_cast<off_t>(old_size))
	std::fill(supporters_[backend].buffer_.begin()+old_size,
		  supporters_[backend].buffer_.end(),
		  0);
}

/** retrieve object information (if not write only), fill buffers with data */
/** @todo add support for .error file */
bool conductor_impl::open(const std::string& path,int mode)
{

    std::string backend,object;
    split(path,backend,object);

    if (backend.empty() || object.empty()) return false;


    supporter_map::iterator it = supporters_.find(backend);
    
    //unknown backend
    if (it == supporters_.end())
	return false;

    plugin_info &info=it->second;
    
    if (info.opened_) return false;//file still opened

/*    if (object == stderr_name) return !info.err_buffer_.empty();*/

    info.opened_ = true;
    info.open_mode_ = mode;
    if ((mode&O_WRONLY) != O_WRONLY)
    {//if mode != write only we fill buffer with object's data
	module &m=info.plugin_;
	m.start(" -r "+quote(object));

	std::copy(std::istreambuf_iterator<char>(m.is().rdbuf()),
			  std::istreambuf_iterator<char>(),
		  std::back_inserter(info.buffer_));

	std::copy(std::istreambuf_iterator<char>(m.es().rdbuf()),
		  std::istreambuf_iterator<char>(),
		  std::ostreambuf_iterator<char>(std::cerr.rdbuf()));
	m.stop();
    }

    return true;
}

/**
 * 1. reparce path to determine supporter and it's object
 * 2. run subprocess with "-d name key" and read it's stderr
 * 3. if stderr read result is not empty then create .stderr file
 */
void conductor_impl::destroy(const std::string& path)
{
    std::string backend,object;
    split(path,backend,object);

    if (backend.empty() || object.empty()) return;

    supporter_map::iterator it = supporters_.find(backend);
    if (it == supporters_.end()) return;

    plugin_info &info = it->second;
    
/*
    if (object == stderr_name)
    {//special processing for .stderr file
	info.err_buffer_.clear();
	return;
    }
*/
    module &m = info.plugin_;
    m.start(" -d "+quote(object));

    //NOTE: wait here or read stderr
    std::copy(std::istreambuf_iterator<char>(m.es().rdbuf()),
	      std::istreambuf_iterator<char>(),
	      std::ostreambuf_iterator<char>(std::cerr.rdbuf()));

    m.stop();

    info.cache_.erase(object);//invalidate cache entry
}

/** @todo add support for .error file */
void conductor_impl::list(const std::string& path,dirent_vector& dirlist)
{
    if ("/" == path)
    {
	std::for_each(supporters_.begin(),
	              supporters_.end(),
		      root_inserter(dirlist));
    }
    else
    {//get info from appropriate plugin
	std::string backend,object;
	split(path,backend,object);

	supporter_map::iterator it = supporters_.find(backend);
	if (backend.empty() || it == supporters_.end()) return;

	plugin_info &info = it->second;
	cache_type &cache = info.cache_;

	std::string object_dir=object.empty()?"":(object+"/");
	if (cache.size())
	{
	    cache_type::iterator start = cache.lower_bound(object_dir);
	    cache_type::iterator stop = start;
	    while (stop != cache.end() && alt::starts_with(stop->first, object_dir)) ++stop;

	    cache.erase(start,stop);
	}
	
	module &m = info.plugin_;
	m.start(" -l "+quote(object.empty()?"/":object));
	std::string str;
	std::for_each(alt::istream_line_iterator(m.is()),
		      alt::istream_line_iterator(),
		      plugin_inserter(object_dir,dirlist,cache));
	m.stop();

/*	if (!info.err_buffer_.empty()) dirlist.push_back(stderr_name);*/
    }
}


