/*
 *  ALTerator - ALT Linux configuration project
 *
 *  Copyright (c) 2004,2005 ALT Linux Ltd.
 *  Copyright (c) 2004,2005 Stanislav Ievlev
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */
#ifndef ALTUTILS_DIRBUF_HH__
#define ALTUTILS_DIFBUF_HH__

#include <streambuf>
#include <stdexcept>
#include <vector>

#include <dirent.h>
#include <sys/types.h>

namespace alt
{
    template <typename charT, typename Traits = std::char_traits<charT> >
    class dir_buf;

    /**
     * buffer over readdir calls
     */
    template <typename charT, typename Traits>
    class dir_buf: public std::basic_streambuf<charT,Traits>
    {
	typedef Traits				traits_type;
	typedef charT		   		char_type;
	typedef typename traits_type::int_type	int_type;
	typedef std::vector<char_type>		vector_type;
	using std::basic_streambuf<charT,Traits>::gptr;
	using std::basic_streambuf<charT,Traits>::egptr;
	public:
	    dir_buf(const char* path, std::ios_base::openmode mode, size_t size):
		    dir_handle_(0),
		    mode_(mode),
		    buffer_(size)
	    {
		char_type *ptr = &buffer_[0];
		setg(ptr,
		     ptr+size,
		     ptr+size);
		
		if ((dir_handle_ = opendir(path)) == 0)
		    throw std::runtime_error("unable to open directory for reading");
	    }
	    virtual ~dir_buf()
	    {
		if (dir_handle_) closedir(dir_handle_);
	    }
	protected:
	    virtual int_type underflow()
	    {
		char_type *ptr = &buffer_[0];
		if (gptr() < egptr())
        	return traits_type::to_int_type(*gptr());

		const int size = buffer_.size();
	
		struct dirent *dent = readdir(dir_handle_);
		
		
		if (!dent)
		{
		    setg(ptr,
			 ptr+size,
			 ptr+size);
		    return traits_type::eof();
    		}

		const int len = strlen(dent->d_name);
		const int writelen = std::min(size,len);

		std::copy(dent->d_name,
		          dent->d_name+writelen,
			  buffer_.begin());
		buffer_[writelen]='\n';//append string separator
		
		setg(ptr,
	             ptr,
		     ptr+writelen+1);

	        return traits_type::to_int_type(*gptr());
	    }
	    
	    virtual int_type overflow (int_type c = traits_type::eof())
	    {
		return c;
	    }

	    virtual
	    std::streamsize xsputn (const char_type* p,std::streamsize n)
	    {
    		return n;
	    }

	private:
		DIR			*dir_handle_;
		std::ios_base::openmode	mode_;
		bool			del_;
		vector_type		buffer_;
    };
}

#endif
